/*
Program that saves compliant and some non compliant histories in two files
(useful for learning protocols).

Usage:

 ?- compile(sciff).
 ?- compile(save_histories).
 ?- project(PROJECTNAME).
 ?- save_histories_timeout('positive.txt','negative.txt',10000,run(_)).

saves the positive examples in positive.txt and the negative in negative.txt
and generates positive histories up to a timeout of 10000ms.
In this case, the project PROJECTNAME should have the option 'fulfiller' set to 'on'.


Another possibility (more time expensive, but with which you could generate more histories)
is

 ?- save_histories_timeout('positive.txt','negative.txt',10000,iter_gsciff(MIN,MAX)).
 
 where MIN and MAX should be the minimal and maximal length of a (positive) history.
 Untested for negative.

*/

:-use_module(library(timeout)).
:- use_module(library(random)).

:- dynamic failure_history/2.

% Gen Pred = iter_gsciff(3,5)

save_histories_timeout(FileName,FailureFileName,TimeOut,GenPred):-
    open(FileName,write,Stream),
    (time_out(save_histories(Stream,_,GenPred),TimeOut,_) ; true),!,
    close(Stream),
    open(FailureFileName,write,Stream),
    (save_failure(Stream) ; true),
    close(Stream).

save_histories(Stream,Patterns,GenPred):-
    call(GenPred),
    get_history(Hist),
    nl,
    write('*** Generated non-ground history ***'), nl,
    write(Hist),
    (failure_history(N,_) -> N1 is N+1 ; N1 is 0),
    assert(failure_history(N1,Hist)),
    ground_domain_vars(Hist),
    retractall(failure_history(N1,_)),
    foreach(Hist,%unify_with_pattern(_,Patterns)
        ground_history(_)
    ),
    write(Stream,Hist),
    nl(Stream), %write(Stream,'============='), nl(Stream),
    %write(saved), nl,
    fail.

get_history(Hist):-
    findall_constraints(h(_,_),L),
    strip_chr_list(L,Hist).

strip_chr_list([],[]).
strip_chr_list([X#_|R],[X|T]):- strip_chr_list(R,T).

foreach([],_).
foreach([H|T],G):-
    copy_term(G,G1),
    G1 =.. [_,H|_],
    call(G1),
    foreach(T,G).

ground_history(Happ):- ground(Happ),!.
ground_history(Happ):- 
    var(Happ),!,
    rand_member(Happ,[1,2,3]).
ground_history([H|T]):- !,
    ground_history(H),
    ground_history(T).
ground_history(Happ):-
    Happ =.. [F|ArgH],
    ground_history(ArgH).


unify_with_pattern(Happ,_):- ground(Happ),!.
unify_with_pattern(Happ,Pattern):- 
    var(Happ),!,
    member(Happ,Pattern).
unify_with_pattern([H|T],[HP|TP]):-
    unify_with_pattern(H,HP),
    unify_with_pattern(T,TP).
unify_with_pattern(Happ,Pattern):-
    Happ =.. [F|ArgH],
    Pattern =.. [F|ArgP],!,
    unify_with_pattern(ArgH,ArgP).
unify_with_pattern(Happ,Patterns):-
    member(Pattern,Patterns),
    unify_with_pattern(Happ,Pattern).

rand_member(X,[X]):-!.
rand_member(X,[Y|T]):- 
    (random(1,4,1) ->
        X=Y ; (X=Y ; rand_member(X,T))
    ).

auction_history_save(TimeOut):-
    save_histories_timeout('hhh.txt',
        h(tell([f,taxi1,taxi2,taxi3],[f,taxi1,taxi2,taxi3],
            [openauction(taxi2station,[1,2,3,4,5,6,7,8,9,10],[1,2,3,4,5,6,7,8,9,10]),
             bid(taxi2station,[1,2,3,4,5,6,7,8,9,10]),
             answer([win,lose],taxi2station,[1,2,3,4,5,6,7,8,9,10])
            ],auction1
        ),[1,2,3,4,5,6,7,8,9,10]),
        TimeOut
    ).
    
iter_gsciff(Min,Max):-
    init_graph('proof.dot',_Stream),
    statistics(runtime,_),
    load_ics,
    society_goal,
    N in Min..Max,
    indomain(N), write('************************* new happen ******************************'),
        writeln(N),
    add_history_el(N),
    close_history,
    write('grounding\n'),
    make_choice,
    statistics(runtime,[_,Time]),
    writeln(runtime(Time)).

ground_domain_vars(Hist):-
    term_variables(Hist,Vars),
    extract_fd_vars(Vars,FDVars),
    domain(FDVars,0,10),
    labeling([],FDVars).
    
extract_fd_vars([],[]).
extract_fd_vars([H|T],[H|R]):-
    fd_var(H),!,
    extract_fd_vars(T,R).
extract_fd_vars([_|T],R):-
    extract_fd_vars(T,R).

save_failure(Stream):-
    failure_history(_,History1),
    renaming(History1,History),
    ground_history(History),
    write(Stream,History),
    nl(Stream),
    fail.

% Like copy_term, but copy_term maintains the constraints (and assert as well).
% Here I want to strip the constraints / domains
renaming(X,_):-  var(X),!.
renaming(A,B):- ground(A),!, A=B.
renaming([H|T],[RH|RT]):- !,
    renaming(H,RH), renaming(T,RT).
renaming(Term,Term1):-
    functor(Term,F,Arity),
    functor(Term1,F,Arity),
    Term =.. [F|Arg],
    Term1 =.. [F|Arg1],
    renaming(Arg,Arg1).
