package SOMA.agent;

import java.lang.reflect.*;
import java.io.*;

import SOMA.Environment;
import SOMA.network.connection.*;
import SOMA.agent.mobility.*;
import SOMA.naming.*;
import SOMA.mobilePlace.MobilePlaceID;

/** <P>Classe che rappresenta un agente in esecuzione.
* </P><P>
* Ad ogni agente <B>agente</B> il place associa un <B>worker</B>, un componente
* che effettua tutte le operazioni di base di un agente.
* Il worker possiede anche il thread di esecuzione dell'agente, che puo' essere avviato ed interrotto.
* </P><P>
* Il worker implementa l'interfaccia {@link SOMA.network.connection.Daemon}.
* Questa comprende i metodi per avviare ed interrompere il thread di un worker, ossia di un agente.
* L'interfaccia prevede anche uno stato del worker. Infine per gestire da menu gli agenti
* si puo' utilizzare la voce di menu {@link SOMA.network.connection.DaemonExplorerItem DaemonExplorerItem}
* utilizzata allo stesso modo in altri contesti.
*
* <P> Il worker ha uno stato che pu essere ON, OFF, GONE, IDLE, STOPPED, DEAD. Tutti
* i metodi che agiscono sullo stato sono sincronizzati.
*
* @see SOMA.agent.Agent
* @see SOMA.agent.AgentSystem
*
*
* @author Livio Profiri
*/

public class AgentWorker implements Runnable, Daemon
{
  /** Riferimento all'agente. */
  public final Agent agent;

    // Final perch l'agente di un worker non pu essere cambiato
    // OCCHIO: il suo stato s, lui pu essere cambiato!

  /** Tutti i threads creati dall'agente si troveranno in questo ThreadGroup. */
  ThreadGroup agentThreadGroup = null;

  Thread agentThread = null;
  Object status = OFF;

  /** Il worker e' nello stato IDLE dal momento in cui l'agente
  * ha richiesto di andare in Idle, al momento
  * in cui il thread dell'agente termina, a questo punto l'agente va nello stato OFF.
  */
  public static final Object IDLE = new Status( "IDLE" );

  /** Il worker e' nello stato GONE dal momento in cui l'agente
  * ha effettuato la migrazione, al momento in cui il thread dell'agente
  * termina, a questo punto il worker viene eliminato.
  */
  public static final Object GONE = new Status( "GONE" );

  /** Il worker e' nello stato DEAD dopo la morte,
  * ossia dopo che il metodo di avvio e' terminato.
  */
  public static final Object DEAD = new Status( "DEAD" );

  /** Il worker e' nello stato STOPPED dal momento in cui l'agente
  * ha terminato l'esecuzione del metodo stop() al momento in cui il suo thread principale termina.
  */
  public static final Object STOPPED = new Status( "STOPPED" );

 /** Il worker e' nello stato KILLED a seguito del metodo kill.
  */
  public static final Object KILLED = new Status( "KILLED" );

  Environment env;

  /** Costruttore.
  * @param agent L'agente
  * @param env Il riferimento all'Environment, accessibile al worker,
  * ma non all'agente.
  */
  public AgentWorker( Agent agent, Environment env )
  {
    this.agent = agent;   // Il worker ha un riferimento al suo agente
    agent.worker = this;  // L'agente al suo worker. OCCHIO: il riferimento ha visibilit di package,
                          // Quindi gli agenti creati dall'utentenon possono accedere al proprio worker.

    this.env = env;
  }

  /** Restituisce una stringa di descrizione del worker. */
  public String toString()
  {
    return "[Worker: " + agent + " Status: " + status + "]";
  }

  /** Restituisce lo stato del worker. */
  public Object getStatus()
  {
    return status;
  }

  /** <P>Metodo di avvio del worker.
  * <P>
  * Questo metodo mette in esecuzione l'agente gestito dal worker ed e' il complementare del metodo {@link #stop()}.
  * <P>
  */

  /* NON E' PIU VERO
  * <B>Attenzione:</B> i metodi {@link #start()} e {@link #stop()} sincronizzano la loro esecuzione
  * sul monitor dell'agente (= <CODE>synchronized( agent )</CODE>). Questo fornisce un semplice meccanismo
  * per interrompere e riavviare l'esecuzione di un agente senza correre il rischio che l'agente venga
  * a trovarsi in uno stato inconsistente. Infatti e' sufficiente, nel programmare un agente,
  * utilizzare il costrutto <CODE>synchronized</CODE> per isolare le parti di codice
  * che devono eseguire <I>operazioni atomiche</I> di modifica dello stato dell'agente.
  * </P>
  */
  public synchronized void start() throws AgentWorkerException
  {
    if( status == OFF || status == IDLE || status == STOPPED )
    {
      if( status == OFF )
      {
        // Il threadGroup va ricreato, perch lo stato va a OFF solo quando tutti
        //   i threads sono terminati ==> il guppo eliminato, perch daemon.
        //
        // Tutti i threads creati dall'agente si troveranno in questo ThreadGroup.
        agentThreadGroup = new ThreadGroup( env.threadGroup, agent.getID().toString() );
        agentThreadGroup.setDaemon( true ); // Distrutto dopo l'ultimo thread di questo gruppo
      }
      else // status == IDLE || STOPPED
      {
        env.out.println( "WARNING: The Agent was stil in " + status + " state." );
        env.out.println( "  There were still active threads!" );
      }

      agentThread = new Thread( agentThreadGroup, this, toString() );

      // Gli agenti sono demoni: la virtual machine si chiude anche se ci sono agenti attivi!
      agentThread.setDaemon( true );
      // Tutte le classi caricate dal thread dell'agente vengono
      // caricate dal suo ClassLoader.
      agentThread.setContextClassLoader( agent.getClass().getClassLoader() );

      status = ON;
      agentThread.start();
      //env.out.println( "AgentWorker " + toString() + " STARTED" );
    }
    else
      throw new AgentWorkerException( "AgentWorker " + toString() + " already " + status );
  }

  /** Metodo di arresto del worker.
    *
    * @returns L'agente congelato.
    *
    * @see #start()
    */
  //public synchronized Agent stop() throws AgentWorkerException
  public synchronized void stop() throws AgentWorkerException
  {
    // Non pi vero!!!!
    // Attenzione: mi sincronizzo sull'agente, quindi, se nella programmazione degli
    //   agenti le operazioni sullo stato dell'agente vengono fatte in modalit synchronized
    //   lo stato rimane consistente anche all'arresto del thread.
    //Agent freezedAgent = null;

    if( status == ON )
    {
      //agentThread.stop();
      //agentThread = null;

      try
      {
        //freezedAgent = agent.stop();
        agent.stop();
        status = STOPPED;
        //env.out.println( "AgentWorker " + toString() + " STOPPED" );
      }
      catch( UnsupportedOperationException e )
      {
        env.err.println( "Can't stop AgentWorker " + toString() + ": agent does't support the stop() operation." );

        throw new AgentWorkerException( e.toString() );
      }
      catch( Exception e )
      {
        env.err.println( "Can't stop AgentWorker " + toString() + ": " + e );
        e.printStackTrace( env.err );
        throw new AgentWorkerException( e.toString() );
      }
    }
    else
      throw new AgentWorkerException( "AgentWorker " + toString() + " already OFF" );


  }

  /** <P>Metodo di avvio del thread del worker.
  * </P><P>
  * Chiama il metodo dell'agente indicato dal campo {@link SOMA.agent.Agent#start start}.
  * <BR>
  * Terminata l'esecuzione dell'agente il worker viene eliminato.
  */
  public void run()
  {
    Class agentClass = agent.getClass();
    Method startMethod = null;

    try
    {
      startMethod = agentClass.getMethod( agent.start, new Class[0] );
    }
    catch( NoSuchMethodException e )
    {
      env.err.println( "ERROR: " + e );
    }
    catch( Exception e )
    {
      e.printStackTrace( env.err );
    }

    try
    {
      if( startMethod == null )
      {
        env.err.println( toString() + "ERROR: null start method --> run() method.");
        agent.run();
      }
      else
      {
        env.err.println( toString() + " calling method: " + startMethod.toString() );

        startMethod.invoke( agent, null );   // nessun argomento per comodita'
      }
    }
    catch( InvocationTargetException e )
    {
      env.err.println( toString() + " ERROR " + e );
      e.getTargetException().printStackTrace( env.err );
    }
    catch (Exception e)
    {
      e.printStackTrace( env.err );
      env.err.println( toString() + " Error " + e + " on method" + startMethod );
    }

    // se filo di esecuzione arriva qui significa che Agente ha terminato la
    // sua esistenza spontaneamente oppure e' avvenuto un errore

    //env.err.println( "Worker " + toString() + ": agente " + agent + " has done.");

    WaitForOtherThreads( env.err );

    synchronized( this )
    {
      if( status == ON )
        status = OFF;  // L'agente morir.

      try
      {
        remove();
      }
      catch( AgentWorkerException e )
      {
        // L'agente  in IDLE o STOPPED ==> va a OFF
        //env.out.println( "IDLE Worker: " + this + " ---> OFF" );
        status = OFF;
      }
    }
    /**
    synchronized( this )
    {
      if( status == GONE || status == ON )
      {
        // Cancello il worker.
        synchronized( env.agentManager.agentWorkerStore )
        {
          AgentWorker oldWorker = env.agentManager.agentWorkerStore.removeWorker( agent.getID() );
          // Questo accade se nel frattempo il worker era stato sostituito da uno nuovo.
          if( oldWorker != this && oldWorker != null )
          {
            env.err.println( toString() + " PROCEDURA INSOLITA: reinserisco il worker: " + oldWorker );
            env.agentManager.agentWorkerStore.putWorker( oldWorker );
          }
        }

        // Quando l'agente fa una GO lo stato diventa GONE.
        // Se fa una Idle diventa IDLE,
        // Se fa stop diventa STOPPED
        // Quindi, se  rimasto ON, vuol dire che l'agente  uscito naturalmente dal
        // metodo con cui era stato avviato ==> l'agente  morto ==> lo comunico al place di origine.
        if( status == ON )
        {
          status = DIED;
          env.networkManager.sendCommand( agent.getID().getHome(), new AgentDeathCommand( agent.getID() ) );
          env.err.println( toString() + " AGENT DIED." );
        }
      }
      else if( status == IDLE || status == STOPPED )
      {
        status = OFF;
        env.out.println( "IDLE Worker: " + this );
      }
      else
      {
        env.err.println( toString() + ".run() ERRORE STRANO stato non previsto!" );
      }
    }*/

    //env.err.println( "Worker " + toString() + "REMOVED");
  }

  public synchronized void remove() throws AgentWorkerException
  {
    if( status == OFF || status == GONE || status == KILLED ) // Effettuo la rimozione
    {
      synchronized( env.agentManager.agentWorkerStore )
      {
        AgentWorker oldWorker = env.agentManager.agentWorkerStore.removeWorker( agent.getID() );
        // Questo accade se nel frattempo il worker era stato sostituito da uno nuovo.
        if( oldWorker != this && oldWorker != null )
        {
          env.err.println( toString() + " PROCEDURA INSOLITA: reinserisco il worker: " + oldWorker );
           env.agentManager.agentWorkerStore.putWorker( oldWorker );
        }
      }

      // Rimuovo un agente OFF ==>  morto.
      if( status == OFF )
      {
        status = DEAD;
        //env.networkManager.sendCommand( agent.getID().getHome(), new AgentDeathCommand( agent.getID() ) );
        env.agentManager.agentDeath( agent.getID() );

        env.err.println( toString() + " AGENT DIED." );
      }
    }
    else
      throw new AgentWorkerException( "Can't remove worker " + this );
  }

  /** Metodo che effettua la migrazione dell'agente.
  *
  * <P>Questo metodo si interfaccia direttamente al {@link SOMA.network.NetworkManager}.
  *
  * <P>Il metodo e' sincronizzato per il cambiamento di stato.
  */
  public synchronized void go( PlaceID destination ) throws CantGoException
  {
    // Si potrebbe rivedere l'elenco degli stati da cui l'agente pu fare una go.
    if( status == DEAD || status == GONE || status == KILLED )
      throw new CantGoException( agent.toString() + " can't go to " + destination.toString() + " because his status is " + status  );
    if( !env.networkManager.sendCommand( destination, new AgentTransportCommand( env, agent, destination )))
    {
      // Se sono nella home del place mobile dove devo recarmi
      //   vuol dire che questo non  raggiungibile, quindi metto l'agente in attesa.
      if( env.mobilePlaceManager != null && env.mobilePlaceManager.checkWaitCondition( destination ) )
      {
        env.mobilePlaceManager.addWaitingAgent( agent.getID(), destination );
        if( status == ON )
          status = IDLE; // Se no il worker viene eliminato!
        else
          status = OFF;
      }
      else
        throw new CantGoException( agent.toString() + " can't go to " + destination.toString() );
    }
    else
    {
      env.out.println( "Worker " + toString() + ": agent " + agent + " sent to " + destination );
      if( status == ON )
        status = GONE;
      else
      {
        try
        {
          remove();
        }
        catch( AgentWorkerException e )
        {
          e.printStackTrace( env.out );
        }
      }
    }
  }

 /** L'agente va nello stato IDLE.
  */
  public synchronized void idle()
  {
    if( status == ON )
    {
      status = IDLE;
      //env.out.println( "Idle agent: " + this );
    }
  }

  /** Distrugge il worker senza farsi scrupoli di alcun genere.
  *
  * <P> Attenzione: fa uso di ThreadGroup.stop(), metodo deprecato.
  */
  public synchronized void kill()
  {
    // Si mette in un altro threadgroup, per non uccidersi da solo!
    Thread killer = new Thread( env.threadGroup, "Agent killer thread" )
      {
        public void run()
        {
          env.out.println( "Killing worker " + AgentWorker.this );

          status = KILLED;
          agentThreadGroup.stop(); // Uccide tutti i threads senza piet.

          try
          {
            remove();
          }
          catch( AgentWorkerException e )
          {
            env.out.println( "ERROR Removing worker: " + e.toString() );
          }
        }
      };

    // L'istruzione stop() potrebbe bloccare tutto, ad esempio se uno
    //   dei threads  in attesa di input. Quindi uso un altro thread!
    killer.start();

    try
    {
      killer.join( 10 * 1000 ); // Attende al massimo 10 secondi.
    }
    catch( InterruptedException e )
    {
      e.printStackTrace( env.out );
    }

    if( killer.isAlive() )
      env.out.println( "The killer thread has troubles in killing " + this + ": STILL WAITING!");
    else
      env.out.println( "Worker " + this + " killed." );
  }


  /*
  public void finalize()
  {
    env.err.println( "*** FINALIZE " + this + " ***" );
  }*/

  public static void WaitForOtherThreads( PrintStream out )
  {
    // Ho rimosso i messaggi di debug.

    Thread thisThread = Thread.currentThread();
    ThreadGroup threadGroup = thisThread.getThreadGroup();

    int count = threadGroup.activeCount();

    while( count > 1 )
    {
      Thread[] tarray = new Thread[count];

      threadGroup.enumerate( tarray, true );

      /*  Elenca i threads che deve attendere
      for( int i = 0; i < tarray.length; i++ )
      {
        if( tarray[i].equals( thisThread ) )
          out.print( " ME: " );

        out.println( " " + i + ") " + tarray[i] );
      }*/

      for( int i = 0; i < tarray.length; i++ )
      {
        if( !tarray[i].equals( thisThread ) && tarray[i].isAlive() )
        {
          //out.println( "Waiting for " + tarray[i] );
          try
          {
            tarray[i].join();
          }
          catch( InterruptedException e )
          {
            out.println( "AgentWorker.WaitForOtherThreads " + tarray[i] + " --> " + e );
          }
        }
      }

      count = threadGroup.activeCount();
    }

    //out.println( "I am the last thread!" );
  }

  public static class AgentWorkerException extends Exception
  {
    public AgentWorkerException( String  s )
    {
      super( s );
    }
  }

}
