package SOMA.network.connection;

import SOMA.Environment;

/** <P>Comando utilizzato per la comunicazione fra place diversi.
* </P><P>
* I comandi vengono spediti attraverso
* {@link SOMA.network.connection.Connection connessioni}. Ricevuto il conmando,
* il place avvia il metodo {@link #start( Connection ReturnConnection, Environment env )}.
* </P><P>
* Il metodo {@link #start( Connection ReturnConnection, Environment env ) start},
* effettuate le necessarie inizializzazioni, crea un nuovo thread ed avvia il metodo
* {@link #run()}, un metodo astratto che conterra' le operazioni svolte
* dal comando.
* </P><P>
* Per implementare un nuovo comando e' quindi sufficiente
* sottoclassare questa classe astratta e definire il metodo {@link #run()}.
* </P>
* @see SOMA.network.connection.QuickCommand
*
* @author Livio Profiri
*/

public abstract class Command implements java.io.Serializable, Runnable
{
  /** Connessione su cui spedire un eventuale comando di risposta.
  * @see #Return( Command ReturnCommand )
  * @serial*/
  protected transient Connection ReturnConnection = null;
  /** Riferimento all'Environment del place.
  * <BR>
  * E' necessario per poter accedere agli oggetti su cui
  * il comando deve operare.
  * @serial*/
  protected transient Environment env = null;

  // Shortcut che non lancia eccezioni!
  /** Spedisce un comando di risposta.
  * <BR>
  * <B>Attenzione:</B> si tratta di uno shortcut che non lancia eccezioni.
  * Per gestire le eccezioni utilizzare: <BR>
  * <CODE>ReturnConnection.send( ReturnCommand );</CODE>
  */
  public void Return( Command ReturnCommand )
  {
    try
    {
      ReturnConnection.send( ReturnCommand );
    }
    catch( Exception e )
    {
      e.printStackTrace( (env != null ? env.err : System.err) );
    }
  }

  /** <P>Esegue il comando.
  * </P><P>
  * @param ReturnConnection inizializza {@link #ReturnConnection}.
  * @param env inizializza {@link #env}.
  * </P><P>
  * Effettuate le inizializzazioni, crea un nuovo thread ed avvia il metodo
  * {@link #run()}, un metodo astratto che conterra' le operazioni svolte
  * dal comando.
  * </P>
  * @see SOMA.network.connection.QuickCommand
  */

  public void start( Connection ReturnConnection, Environment env )
  {
    this.ReturnConnection = ReturnConnection;
    this.env = env;
    Thread commandThread = new Thread( this, this.toString() );
    commandThread.setDaemon( true );  // Non voglio che un comando bloccato impedisca la terminazione della JVM
    //commandThread.setPriority( Thread.MIN_PRIORITY );
    commandThread.start();
  }

  /** <P>Se restituisce <CODE>true</CODE> il comando viene eseguito anche in caso di errore.
  * </P><P>
  * Quando un comando viene spedito lungo un cammino multiplo,
  * ad esempio un cammino che unisce place in domini diversi,
  *  possibile che ad uno dei passaggi si verifichi un errore.
  * In questo caso, se startOnError() restituisce <CODE>true</CODE>,
  * {@link SOMA.network.TransportCommand TransportCommand }
  * avvia il comando anche se non  giunto a destinazione.
  * </P><P>
  * Nell'implementazione del comando,
  * ridefinire questo metodo se si vuole che l'opzione sia attivata.
  * </P>
  */
  public boolean startOnError()
  {
    return false;
  }

  /** Contiene la computazione che il comando deve svolgere.
  * <BR>
  * Questo metodo astratto va definito nelle implementazioni concrete dei comandi.
  */
  public abstract void run();
}