package SOMA.naming.domain;

import java.io.*;
import java.util.*;
import java.net.InetAddress;

import SOMA.naming.*;
import SOMA.network.connection.*;
import SOMA.Environment;

/** Servizio di nomi di dominio, <B>DNS</B>.
*
* <P> Il servizio realizza la metafora di una tabella in cui vengono memorizzate
* le {@link DomainInfo}. La chiave di ricerca e' di tipo
* {@link SOMA.naming.PlaceID PlaceID}, identificatore del default place del dominio.
*
* <P> Questo servizio e' presente solo nel <B>default place</B> di ogni dominio.
* Infatti, gli altri place hanno informazioni sul solo dominio di appartenenza.
*
* <P> E' previsto un sistema gerarchico di scambio di informazioni fra DNS di domini diversi.
* Ogni DNS ha <B>1 DNS padre</B>, presso cui e' registrato, e una serie di <B>DNS figli</B>
* che si sono registrati presso di lui. In questo modo si realizza una <B>struttura ad albero</B>
* in cui le informazioni passano dai figli al padre e viceversa.
*
* @see DNSExplorerItem
* @see SOMA.naming.place.PlaceNameService
*
* @author Livio Profiri  (metodi aggiuntivi di Luigi Antenucci-gigi).
*/
public class DomainNameService implements Serializable
{
  /** @serial*/
  transient Environment env;
  /** @serial*/
  Hashtable domains = new Hashtable();

  /** @serial*/
  PlaceID fatherDNS = null;
  /** @serial*/
  Set childrenDNS = Collections.synchronizedSet( new HashSet() );

  /** Costruttore.
  * @param env L'environment del place.
  */
  public DomainNameService( SOMA.Environment env )
  {
    this.env = env;
  }

  /** Restituisce la <code>DomainInfo</code> corrispondente a <code>placeID</code>,
  * o <code>null</code> se non trovata.
  */
  synchronized public DomainInfo getDomain( PlaceID placeID )
  {
    return (DomainInfo)domains.get( placeID );
  }

  /** Inserisce una <code>DomainInfo</code>.
  *
  * <P> Se <code>DomainInfo</code> e' nuova aggiorna DNS padre e figli.
  * <BR>
  * @return Il vecchio valore memorizzato o <code>null</code> se si tratta di un nuovo dominio.
  */
  synchronized public DomainInfo putDomain( DomainInfo newDomainInfo )
  {
    DomainInfo oldDomainInfo = (DomainInfo)domains.put( newDomainInfo.placeID, newDomainInfo );

    if( ! newDomainInfo.equals( oldDomainInfo ) )
      sendToAllDomains( new PutDomainCommand( newDomainInfo ) );

    return oldDomainInfo;
  }

  /** Elimina la <code>DomainInfo</code> corrispondente a <code>aDomainID</code>.
  *
  * <P> Se <code>aDomainID</code> e' presente trasmette l'operazione a DNS padre e figli.
  * <BR>
  * @return Il vecchio valore memorizzato o <code>null</code> se si tratta di un nuovo dominio.
  */

  // Cancello il vecchio dominio anche dalla lista dei figli e dal padre, se necessario
  synchronized public DomainInfo removeDomain( PlaceID aDomainID )
  {
    DomainInfo oldDomainInfo = (DomainInfo)domains.remove( aDomainID );  // Restituisce il vecchio valore o null.
    childrenDNS.remove( aDomainID );
    if( aDomainID.equals( fatherDNS ) )
      fatherDNS = null;

    if( oldDomainInfo != null )
      sendToAllDomains( new RemoveDomainCommand( aDomainID ) );

    return oldDomainInfo;
  }

  /** Stampa la lista di tutti i domini su <code>out</code>.*/
  public void listDomains( PrintStream out )
  {
    out.println( "I know " + domains.size() + " domains:" );
    out.println();
    int i;
    Enumeration list;

    for( i = 1, list = domains.elements(); list.hasMoreElements(); )
      out.println( "    " + i++ + ") " + list.nextElement().toString() );
    out.println();

    if( fatherDNS != null )
      out.println( "  My father Domain Name Server is " + fatherDNS );

    if( childrenDNS.size() > 0 )
    {
      out.println( "  I have " + childrenDNS.size() + " children Domain Name Servers:"  );
      out.println();
      Iterator it;

      for( i = 1, it = childrenDNS.iterator(); it.hasNext(); )
        out.println( "    " + i++ + ") " + it.next().toString() );
      out.println();
    }
  }

  /** Registrazione presso il DNS padre.
  *
  * <P>Viene inviato un {@link DomainRegisterCommand} al DNS padre individuato dalla coppia
  * <B>host:port</B>.
  */
  public boolean register( InetAddress host, int port )
  {
    // Il cast viene fatto per sicurezza: non volgio registrare un place per errore.
    // a runtime viene lanciata un'eccezione

    return env.networkManager.sendCommand( host, port,
      new DomainRegisterCommand( (DomainInfo)env.networkManager.placeInfo ) );
  }

  /** Richiesta di refresh della tabella al DNS padre.
  *
  * <P>Viene inviato un {@link DomainRefreshCommand.Request} al DNS padre.
  */
  public boolean refresh()
  {
    boolean answer = false;
    
    DomainInfo di = getDomain( fatherDNS );

    if( di != null )
      answer = refresh( di.host, di.port );
    
    return answer;
    //return sendToFatherDomain( new DomainRefreshCommand.Request() );
  }

  /** Richiesta di refresh della tabella al DNS padre.
  *
  * <P>Viene inviato un {@link DomainRefreshCommand.Request} al DNS padre,
  * individuato dalla coppia <B>host:port</B>.
  *
  * <P> Attenzione: da usare solo quando il dominio padre cambia
  * <B>host:port</B>, altrimenti si viene registrati in due DNS diversi,
  * col risultato di avere due DNS padre che spediscono le informazioni di aggiornamento.
  */
  public boolean refresh( InetAddress host, int port )
  {
    return env.networkManager.sendCommand( host, port,
      new DomainRefreshCommand.Request( true ) );
  }

  /** Spedisce un comando al DNS padre.
  *
  * @return <code>true</code> se la spedizione ha avuto successo.
  */
  public boolean sendToFatherDomain( Command command )
  {
    boolean returnValue = false;

    if( fatherDNS != null )
    {
      returnValue = env.networkManager.sendCommand( fatherDNS, command );
      env.out.println( "  Sending command " + command + " to father " + fatherDNS + " result: " + returnValue );
      env.out.println();
    }

    return returnValue;
  }

  /** Spedisce un comando a tutti i DNS figli.
  */
  public void sendToChildrenDomains( Command command )
  {
    if( childrenDNS.size() > 0 )
    {
      env.out.println( "  Sending " + command + " to " + childrenDNS.size() + " children DNSs:"  );
      env.out.println();
      int i;
      Iterator it;

      for( i = 1, it = childrenDNS.iterator(); it.hasNext(); )
      {
        PlaceID dest = (PlaceID)it.next();
        env.out.println( "    " + i++ + ") " + dest + " result: " + env.networkManager.sendCommand( dest, command ));
      }
      env.out.println();
    }
  }

  /** Spedisce un comando al DNS padre e a tutti i DNS figli.
  */
  public void sendToAllDomains( Command command )
  {
    sendToFatherDomain( command );
    sendToChildrenDomains( command );
  }

  /** Restituisce un array contenente tutti gli identificatori di domino.
  *
  * Questa funzionalita' viene fornita agli agenti.
  */
  public PlaceID[] getDomainsArray()
  {
    return (PlaceID[])domains.keySet().toArray( new PlaceID[domains.size()] );
  }
  
    /** Aggiorna il riferimento all'Environment. Usato nel caricamento da disco. */
  public void setEnv( Environment env )
  {
    if( this.env == null )
      this.env = env;
  }

  public String toString()
  {
    return "[dns: " + domains.size() + " domains]";
  }

  /** 
   *  Rende il PlaceID del DNS padre.
   *  Questo metodo  stato aggiunto da Luigi Antenucci (gigi).
   */
  public PlaceID getParentDNS () {
    return fatherDNS;
  } //getParentDNS

  /** 
   *  Rende i PlaceID dei DNS figli (zero, uno o pi). 
   *  Rende un Vector contenente oggetti PlaceID!
   *  Questo metodo  stato aggiunto da Luigi Antenucci (gigi).
   */
  public Vector getChildrenDNS () {
    Vector reso = new Vector ();
    Iterator it = childrenDNS.iterator();
    while (it.hasNext()) {
      PlaceID bimbo = (PlaceID) it.next();
      reso.addElement (bimbo);
    }
    return reso;
  } //getChildrenDNS

}