package SOMA.mobilePlace;

import SOMA.naming.*;
import SOMA.explorer.*;
import SOMA.Environment;
import SOMA.agent.AgentWorker;

import java.util.*;
import java.io.*;

/** Gestore dei place mobili di un dominio.
*
* @author Livio Profiri
*/
public class MobilePlaceManager implements Serializable
{
  transient Environment env;
  Hashtable mobilePlacePositions = new Hashtable();
  Hashtable waitingAgents = new Hashtable();

  public final static PlaceID DISCONNECTED = new PlaceID( "", "" )
    {
      public String toString()
      {
        return "DISCONNECTED";
      }

      public boolean equals(Object obj)
      {
        return this == obj;
      }
    };

  /** Costruttore.
  * @param env L'environment del place.
  */
  public MobilePlaceManager( Environment env )
  {
    this.env = env;

    env.dir.addItem( "mpm", new MobilePlaceManagerExplorerItem( this ) );
  }

  public PlaceID getPosition( PlaceID mobilePlaceID )
  {
    return (PlaceID)mobilePlacePositions.get( mobilePlaceID );
  }

  public PlaceID putPosition( PlaceID mobilePlaceID, PlaceID position )
  {
    return (PlaceID)mobilePlacePositions.put( mobilePlaceID, position );
  }

  public PlaceID removePosition( PlaceID mobilePlaceID )
  {
    waitingAgents.remove( mobilePlaceID );

    return (PlaceID)mobilePlacePositions.remove( mobilePlaceID );
  }

  public void printStatus( PrintStream out )
  {
    printMobilePlacePositions( out );
    printWaitingAgents( out );
  }

  public void printMobilePlacePositions( PrintStream out )
  {
    out.println( "Mobile places positions: " );

    Iterator i = mobilePlacePositions.entrySet().iterator();

    for( int j = 1; i.hasNext(); j++ )
    {
      Map.Entry e = (Map.Entry)i.next();
      out.println( "  " + j + ") " + e.getKey() + " --> " + e.getValue() );
    }

    out.println();
  }

  /** Restituisce true se gli agenti destinati a destination devono attendere qui. */
  public boolean checkWaitCondition( PlaceID destination )
  {
    return destination instanceof MobilePlaceID
           && env.placeID.equals( destination.getHome() )
           && getPosition( destination ) == DISCONNECTED;
  }

  /** Aggiunge un agente alla liste degli agenti che attendono un place mobile.*/
  public void addWaitingAgent( AgentID agentID, PlaceID destination )
  {
    // Prendo il Set di agenti che aspettano
    Set agentIDSet = (Set)waitingAgents.get( destination );

    if( agentIDSet == null ) //Se non c' lo creo e lo inserisco!
    {
      // Creo una versione sincronizzata di set: nessun conflitto n duplicazione di indici
      agentIDSet = Collections.synchronizedSet( new HashSet() );
      waitingAgents.put( destination, agentIDSet );
    }

    // Ora posso inserire aID in agentIDSet!
    agentIDSet.add( agentID );

    env.out.println("MobilePlaceManager: Agent " + agentID + " is waiting for Mobile Place " + destination );
  }

  /** Spedisce ad un place mobile agenti che aspettavano
  * la sua connessione.
  */
  void sendWaitingAgents( PlaceID dest )
  {
    Set waitingSet = (Set)waitingAgents.get( dest );

    if( waitingSet == null )
    {
      env.out.println( "sendWaitingAgents: no agent is waiting for:" + dest );
    }
    else
    {
      env.out.println( "SENDING AGENTS to: " + dest );


      for( Iterator i = waitingSet.iterator(); i.hasNext(); )
      {
        AgentID AgID = (AgentID)i.next();
        i.remove();

        try
        {
          AgentWorker worker = env.agentManager.agentWorkerStore.getWorker( AgID );
          //env.out.println( "  agent " + AgID + " --> " + worker );
          worker.go( dest );
          //env.out.println( "    Sent Agent -->" + AgID );
        }
        catch( Exception e )
        {
          env.out.println( "Agent -->" + AgID + " ERROR: " + e );
        }
      }

      env.out.println( "  =================  DONE!!!" );
    }
  }

  public void printWaitingAgents( PrintStream out )
  {
    out.println( "List of agents waiting for Mobile Place connection:" );

    for (Iterator i = waitingAgents.entrySet().iterator(); i.hasNext(); )
    {
      Map.Entry  e = (Map.Entry)i.next();

      out.println( "  Place: " + e.getKey() + " agentIDs:" );

      for( Iterator it = ((Set)e.getValue()).iterator(); it.hasNext(); )
        out.println( "      " + it.next() );
    }

    out.println();
  }

  /** Aggiorna il riferimento all'Environment. Usato nel caricamento da disco. */
  public void setEnv( Environment env )
  {
    if( this.env == null )
      this.env = env;
  }

  public String toString()
  {
    return "[mpm: " + mobilePlacePositions.size() + " mobilePlaces]";
  }
}
