/**  FinAgentElencoPlace
 *     Finestra per mostrare un elenco di place.
 *     Utile per un agente che vuole mostrare l'elenco dei place che ha visitato
 *     @author     Luigi Antenucci
 *     @version    1.6
 *     @language   jdk 1.2.2
 */

package SOMA.gui;

import SOMA.gui.lingua.Lingua;
import SOMA.naming.PlaceID;
import SOMA.agent.Agent;

import java.util.Vector;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import javax.swing.table.TableModel;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumnModel;


public class FinAgentElencoPlace extends Finestra {

      /**
       *  Per memorizzare i parametri passati al costruttore,
       *  in modo che siano visibili anche agli altri metodi.
       */
  protected FinGoto1000Listener agente;
  protected boolean conBotGoto;
  protected String  fraseTesta;
  protected Vector  elencoPlace;
  protected PlaceID placeHome, 
                    placeAtt;

      /**
       *  Oggetti che mostrano frasi, coi che dipendono dalla lingua.
       */ 
  protected JTable      tabella;
  protected TableModel  modelloElencoPlace;
  protected JLabel      labTesta;
  protected JButton     botGoTo,
                        botClose;

      /**
       *  Costruzione della finestra per la visualizzazione dell'elenco dei place.
       *  L'elenco viene passato sottoforma di lista (Vector).
       *  Questo costruttore serve per avere la SOLA VISUALIZZAZIONE DELL'ELENCO.
       *  Viene mostrato il solo bottone per chiudere la finestra.
       *  Il parametro "fraseTesta" conterr la frase da stampare sopra la lista.
       *  I due parametri PlaceID aggiuntivi indicano il place di creazione dell'agente (home)
       *  e quello corrente (su cui si trova l'agente).
       */
  public FinAgentElencoPlace (Vector elencoPlace, 
                              String fraseTesta,
                              PlaceID placeHome, PlaceID placeAtt) {
    this (elencoPlace, null, fraseTesta, placeHome, placeAtt, false);
  } //costruttore

      /**
       *  Costruzione della finestra per la visualizzazione dell'elenco dei place.
       *  L'elenco viene passato sottoforma di lista (Vector).
       *  NB: Questo costruttore provveder a creare un bottone "GO-TO" il cui effetto sar quello di
       *  dire all'agente (passato) di migrare verso il place selezionato sulla lista.
       *  L'agente  passato non come un normale {@link SOMA.agent.Agent} ma come un
       *  {@link SOMA.gui.FinGoto1000Listener} perch si specifica non un agente in quanto tale,
       *  ma un qualcuno (agente o meno) a cui indicare il place che l'utente pu scegliere (tra 
       *  quelli mostrati nell'elenco).
       *  Invece "fraseTesta" conterr la frase da stampare sopra la lista.
       *  I due parametri PlaceID aggiuntivi indicano il place di creazione dell'agente (home)
       *  e quello corrente (su cui si trova l'agente).
       */
  public FinAgentElencoPlace (Vector elencoPlace, 
                              FinGoto1000Listener agente, 
                              String fraseTesta,
                              PlaceID placeHome, PlaceID placeAtt) {
    this (elencoPlace, agente, fraseTesta, placeHome, placeAtt, true);
  } //costruttore

      /**
       *  Costruzione interno, richiamato dai due precedenti.
       */
  protected FinAgentElencoPlace (Vector elencoPlace, 
                                 FinGoto1000Listener agente,
                                 String fraseTesta, 
                                 PlaceID placeHome, PlaceID placeAtt,
                                 boolean conBotGoto) {
    super (40, 20);        // CHIAMO IL COSTRUTTORE DELLA SUPERCLASSE

    this.elencoPlace = elencoPlace;
    this.agente      = agente;
    this.conBotGoto  = conBotGoto;
    this.fraseTesta  = fraseTesta;
    this.placeHome   = placeHome;
    this.placeAtt    = placeAtt;

    costruisciFinestra();
  } //costruttore

      /**
       *  Vedi in superclasse {@link SOMA.gui.Finestra} il metodo {@link SOMA.gui.Finestra#costruisciFinestra()}
       */
  public void costruisciFinestra () {
    InizioCostruisciFinestra ();       // PARTE INIZIALE DI COSTRUZIONE DELLA FINESTRA

    labTesta = new JLabel (fraseTesta);  // non modificabile dalla lingua, sigh!

    if (conBotGoto) {
      botGoTo = new JButton ();
      botGoTo.setCursor (cursoreBottone);
    }

    botClose = new JButton ();
    botClose.setCursor (cursoreBottone);

    modelloElencoPlace = new ModelloElencoPlace (elencoPlace, placeHome, placeAtt);
    tabella = new JTable (modelloElencoPlace);
    tabella.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);

    ListSelectionModel lsm = tabella.getSelectionModel();
    lsm.addListSelectionListener (new TabellaListener());

    TableColumnModel colonne = tabella.getColumnModel();
    colonne.getColumn(0).setPreferredWidth(50);       //la prima colonna sar pi piccola
    colonne.getColumn(1).setPreferredWidth(40);
    colonne.getColumn(2).setPreferredWidth(250-50-40);

    JScrollPane tabellaScroll = new JScrollPane (tabella);
    tabella.setPreferredScrollableViewportSize (new Dimension(250, 100));
    tabellaScroll.setMinimumSize (new Dimension(50,30));
    tabellaScroll.updateUI();

    impostaFrasi ();         // IMPOSTA LE FRASI (DIPENDENTI DALLA LINGUA)

    ListBottoni bottoniera = new ListBottoni(this, agente);
    if (conBotGoto) {
      botGoTo.addActionListener (bottoniera);
      botGoTo.setActionCommand("goto");
      botGoTo.setIcon (new ImageIcon(PICTURE_PATH+"AgentMigra.gif"));

      botGoTo.setEnabled (false);  // Per ora  disabilitato (devo selezionare una riga nella tabella)
    }

    botClose.addActionListener (bottoniera);
    botClose.setActionCommand("close");
    botClose.setIcon (new ImageIcon(PICTURE_PATH+"BotKO.gif"));
    botClose.setPressedIcon(new ImageIcon(PICTURE_PATH+"Zampina.gif"));

    // USO Window Listener
    mioFrame.addWindowListener (bottoniera);

    PanCentro.setLayout (new BorderLayout());

    // AGGIUNGO SPAZIO VUOTO ATTORNO AL PANNELLO
    PanCentro.setBorder (BorderFactory.createEmptyBorder(10, 5, 8, 5));   // alto,sx,basso,dx

    PanCentro.add (labTesta, BorderLayout.NORTH);

    PanCentro.add (tabellaScroll, BorderLayout.CENTER);

    JPanel panBott = new JPanel();
    GridBagLayout GBL3     = new GridBagLayout();
    GridBagConstraints CC3 = new GridBagConstraints();
    panBott.setLayout (GBL3);
      CC3.fill = GridBagConstraints.VERTICAL;
      if (conBotGoto) {
        CC3.gridx = 1;   CC3.gridy = 1;
        GBL3.setConstraints (botGoTo, CC3);
        panBott.add (botGoTo);
      }
  
      Component spazio = Box.createRigidArea (new Dimension(20, 4));
      CC3.gridx = 3;   CC3.gridy = 1;
      GBL3.setConstraints (spazio, CC3);
      panBott.add (spazio);

      CC3.gridx = 6;   CC3.gridy = 1;
      GBL3.setConstraints (botClose, CC3);
      panBott.add (botClose);
      panBott.setBorder (BorderFactory.createEmptyBorder(15, 10, 0, 10));   // alto,sx,basso,dx

    PanCentro.add (panBott, BorderLayout.SOUTH);

    mioFrame.setResizable (true);     // La finestra place e' ridimensionabile

    FineCostruisciFinestra ();       // PARTE FINALE DI COSTRUZIONE DELLA FINESTRA
  } //costruisciFinestra

      /**
       *  Vedi in superclasse {@link SOMA.gui.Finestra} il metodo {@link SOMA.gui.Finestra#creaBarraMenu()}
       */
  protected void creaBarraMenu () {
    creaBarraMenuDefault (MENU_CLOSE);
  } //creaBarraMenu

      /**
       *  Vedi in superclasse {@link SOMA.gui.Finestra} il metodo {@link SOMA.gui.Finestra#impostaFrasi()}
       *  Definisce/modifica tutte le frasi mostrate a video (nella GUI)
       */
  protected void impostaFrasi () {
    mioFrame.setTitle (Lingua.frase("FAEP_WIN_TITLE"));

    if (conBotGoto) {
      botGoTo.setText (Lingua.frase("FAEP_BOT_GOTO"));
      botGoTo.setToolTipText (Lingua.frase("FAEP_TIP_GOTO"));
    }
    botClose.setText (Lingua.frase("FAEP_BOT_CLOSE"));
    botClose.setToolTipText (Lingua.frase("FAEP_TIP_CLOSE"));

    tabella .setToolTipText (Lingua.frase("FAEP_TIP_TABLE"));

    FineImpostaFrasi();      // APPORTA ALLA FINESTRA LE MODIFICHE FATTE
  } //impostaFrasi


      /**
       *  CLASSE INTERNA PER ASCOLTARE LA PRESSIONE DEI BOTTONI DELLA FINESTRA.
       */
  class ListBottoni extends WindowAdapter implements ActionListener {
    protected FinAgentElencoPlace miaFinAgentElencoPlace;
    protected FinGoto1000Listener mioAgente;

    public ListBottoni (FinAgentElencoPlace miaFinAgentElencoPlace, FinGoto1000Listener mioAgente) {
      this.miaFinAgentElencoPlace = miaFinAgentElencoPlace;
      this.mioAgente = mioAgente;
    } //costruttore

    public void actionPerformed (ActionEvent e) {
      String azione = e.getActionCommand();
      if (azione.equals("goto")) {
        if (mioAgente != null) {  //per sicurezza

          int riga = miaFinAgentElencoPlace.tabella.getSelectedRow();
          PlaceID placeToGo = (PlaceID) modelloElencoPlace.getValueAt (riga, 2);
              // nota: 2 = terza colonna = colonna place
          try {

            agente.eseguiAzioneGoto (placeToGo);   // CHIAMATA SECONDO IL LISTENER "FinGoto1000Listener".

            miaFinAgentElencoPlace.distruggiFinestra();   // Se tutto bene, distruggo questa finestra

          }
          catch (Exception ecciu) {
            OiDialogoi.mostraErrore (ecciu, mioFrame);
          }
        }
      }
      else // caso: "Annulla"
        miaFinAgentElencoPlace.distruggiFinestra();   // In ogni caso distruggo questa finestra
    } //actionPerformed

    public void windowClosing (WindowEvent e)   {
      miaFinAgentElencoPlace.distruggiFinestra();   // distruggo questa finestra (torno a Inizio)
    } //windowClosing
  } //ListBottoni


      /**
       *  CLASSE INTERNA CHE FUNGE DA ASCOLTATORE DELLE SELEZIONI FATTE SULLA TABELLA.
       */
  public class TabellaListener implements ListSelectionListener {

    public void valueChanged (ListSelectionEvent e) {
      int riga = tabella.getSelectedRow();    // OTTENGO IL N.DELLA RIGA SELEZIONATA
      if (conBotGoto)
        botGoTo.setEnabled (riga >= 0);
    } //valueChanged
  } //TabellaListener


      /**
       *  CLASSE INTERNA CHE DEFINISCE IL "MODELLO" A CUI SI RIFERISCE LA TABELLA.
       *  La tabella si basa sull'elenco Vector passato al momento della costruzione.
       *  Vedi classe {@link javax.swing.table.AbstractTableModel}
       */
  public class ModelloElencoPlace extends AbstractTableModel {

    protected Vector  elencoPlace;
    protected PlaceID placeHome, 
                      placeAtt;

    protected ImageIcon iconaHomeOra = new ImageIcon(Finestra.PICTURE_PATH+"ElePlaceHomeOra.gif");
    protected ImageIcon iconaHome    = new ImageIcon(Finestra.PICTURE_PATH+"ElePlaceHome.gif");
    protected ImageIcon iconaOra     = new ImageIcon(Finestra.PICTURE_PATH+"ElePlaceOra.gif");
    protected ImageIcon iconaNull    = new ImageIcon(Finestra.PICTURE_PATH+"ElePlaceNull.gif");

      /**
       *  Costruisce il modello basandosi sull'elenco dei place passato.
       */
    public ModelloElencoPlace (Vector elencoPlace, 
                               PlaceID placeHome, PlaceID placeAtt) {
      this.elencoPlace = elencoPlace;
      this.placeHome   = placeHome;
      this.placeAtt    = placeAtt;
    } //costruttore

      /**
       *  Rende il numero di colonne della tabella.
       */
    public int getColumnCount () { 
      return 3;
    } //getColumnCount

      /**
       *  Rende il numero di righe della tabella.
       */
    public int getRowCount () {
      // Chiedo all'elenco la sua dimensione
      return elencoPlace.size();
    } //getRowCount

      /**
       *  Rende il valore (sottoforma di oggetto) contenuto nella tabella in posizione (row,col).
       */
    public Object getValueAt (int row, int col) {
      if ((row >= 0) && (row < elencoPlace.size())) {
        switch (col) {
          case  0: return new Integer(row);
          case  1: { PlaceID placeID = (PlaceID) elencoPlace.get(row);  // prendo l'elemento del Vector
                     if (placeID.equals(placeHome) && (placeID.equals(placeAtt)))
                       return iconaHomeOra;
                     else
                       if (placeID.equals(placeHome)) 
                         return iconaHome;
                       else
                         if (placeID.equals(placeAtt)) 
                           return iconaOra;
                         else
                           return iconaNull;
                   }
          case  2: return (PlaceID) elencoPlace.get(row);  // rendo l'elemento del Vector
          default: return null;
        }
      }
      else
        return null;
    } //getValueAt

      /**
       *  Rende il nome della colonna "col", cio l'informazione "intensiva".
       */
    public String getColumnName (int col) {
      switch (col) {
        case  0: return Lingua.frase("FAEP_NUMBER");
        case  1: return Lingua.frase("FAEP_TYPE");
        case  2: return Lingua.frase("FAEP_PLACEID");
        default: return null;
      }
    } //getColumnName

      /**
       *  Rende la classe degli oggetti relativi alla colonna "col".
       *  Necessaria per un buon "allineamento" delle stringhe a sinistra e dei numeri a destra!
       */
    public Class getColumnClass (int col) {
      return getValueAt(0,col).getClass();
    } //getColumnClass

      /**
       *  La cella in posione (row,col) pu essere modificata dall'utente?
       */
    public boolean isCellEditable (int row, int col) {
      return false;
    } //isCellEditable

  } //ModelloElencoPlace (classe)


/*
  public static void main (String[] args) {
    // PER DEBUGGING
    UIManager.installLookAndFeel ("Gigi", "SOMA.gui.gigi.GigiLookAndFeel");
    Vector elenco = new Vector();
    elenco.addElement (new PlaceID("Italia",""));
    elenco.addElement (new PlaceID("Italia","Bologna"));
    elenco.addElement (new PlaceID("America","Broadway"));
    new FinAgentElencoPlace (elenco, "Elenco:");
  } //main
*/

} //FinAgentElencoPlace