/**  Anagrafe
 *     Classe usata per mantenere traccia di tutti i place creati.
 *     In origine doveva essere una classe "statica" ma poi l'ho dovuta rendere un OGGETTO
 *     qualunque perch non voglio che sia accessibile da chiunque, ma solo dalla GUI DI SOMA.
 *     Se fosse stata statica anche gli agenti avrebbero potuto accedervi e ottenere gli
 *     oggetti "Environment" e gli "ActionPlace" dei place creati! E non  carino ci, visto
 *     che gli agenti devono passare SOLO attraverso l'"AgentSystem".
 *     Ci sar una sola istanza di questo oggetto e viene creato dalla classe "Inizio".
 *     Ha tre funzionalit base.
 *     La prima  l'aggiornamento della situazione corente (ci sono dei metodi 
 *     che occorre invocare ogni volta che viene creato e registrato un nuovo place).
 *     La seconda  il reperimento di informazioni utili avendo a disposizione il
 *     solo identificatore del place ({@link SOMA.naming.PlaceID}) o il suo ambiente
 *     ({@link SOMA.Environment}) (in particolare, si pu ottenere l'ActionPlace, la
 *     porta e l'host di registrazione, il percorso (directory) dove sono memorizzati
 *     gli agenti e quello per la cache degli agenti.
 *     L'ultima funzionalit  la costruzione di strutture dati a partire dai
 *     dati pervenuti fino a quel momento e dalla configurazione del sistema 
 *     (infatti si pu ottenere una lista con tutti i place, una con tutti i place 
 *     di default, una con tutti i place in un certo dominio, un albero con tutti
 *     i place secondo la relazione di appartenenza al rispettivo place di default,
 *     e infine un albero rappresentante tutta la gerarchia di registrazione dei 
 *     Domain-Name-Service dei place di default).
 *     Infine ha come funzionalit aggiuntiva la possibilit di "registrare" degli 
 *     oggetti (che implementano l'interfaccia "AnagrafeListener") per ascoltare
 *     gli eventi di "modifica dell'Anagrafe".
 *     @author     Luigi Antenucci
 *     @version    2.8
 *     @language   jdk 1.2.2
 */

package SOMA.gui;

import SOMA.Environment;
import SOMA.naming.PlaceID;
import SOMA.explorer.DirExplorerItem;
import SOMA.network.connection.Daemon;
import SOMA.network.connection.DaemonExplorerItem;

import java.util.Vector;
import java.util.Enumeration;
import java.net.InetAddress;
import javax.swing.event.EventListenerList;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;


public class Anagrafe {

  // Variabili protette:
      /**
       *  Qui ci memorizzo i riferimenti a tutti gli "Environment" dei place creati.
       */
  protected Vector libroMastro;

      /**
       *  Lista contenente tutti gli oggetti che si sono "registrati"
       *  per ascoltare un cambiamento di anagrafe (battesimo di un nuovo place)!
       */
  protected EventListenerList listenerList;
      // NON mi serve un "AnagrafeEvent". Se mi servisse scriverei:
      //    protected static AnagrafeEvent anagrafeEvent = null;

      /**
       *  CLASSE INTERNA, usata per tener traccia delle registrazioni dei "Domain Name Service".
       *  La lista "parentela" conterr oggetti di questo tipo.
       */
  protected class Cella {
            /**
             *  Identificazione dei place figlio e genitore.
             *  (i nomi - ovviamente - sono a discrezione del programmatore, non  vero?).
             */
          public PlaceID placeFigliuolo, placeBabbo;

            /**
             *  Il costruttore di una singola cella
             */
          public Cella (PlaceID figlio, PlaceID padre) {
            placeFigliuolo = figlio;
            placeBabbo     = padre;
          }
  } //Cella

      /**
       *  Qui ci memorizzo gli oggetti "Cella" per ricordarmi le registrazioni dei DNS.
       */
  protected Vector parentela;

      /**
       *  Costruzione di un oggetto Anagrafe.
       */
  public Anagrafe () {
    libroMastro  = new Vector ();
    listenerList = new EventListenerList ();
    parentela    = new Vector();
  } //costruttore

      /**
       *  Rende noto al mondo che  nato un nuovo place!
       *  Lo "battezza", cio lo scrive nell'elenco interno "libroMastro",
       *  dopodich genera un'evento "anagrafeCambiata".
       */
  public synchronized void battezzaPlace (Environment placeEnv) {

    aggiungiOrdinato (libroMastro, placeEnv);

    // NOTIFICO L'AVVENUTO CAMBIAMENTO A TUTTI GLI "AnagrafeListener" REGISTRATI
    // ("sparo" un "anagrafeCambiata" a tutti i listener
    fireAnagrafeCambiata ();

  } //battezzaPlace

      /**
       *  Questa funzione permette di "ricordarsi" una registrazione di DNS.
       *  Poich la classe "DomainNameService" NON permette la lettura dei sui attributi
       *  (non sono "public") l'unica strada  "memorizzarsi" la registrazione nello
       *  stesso momento in cui viene fatta. In genere questo metodo va invocato
       *  dopo la "battezzaPlace" (nel caso che il place sia un place di default). 
       */
  public synchronized void ricordaRegistrazioneLocaleDNS (PlaceID placeFiglio, int porta) {
    PlaceID placeGenitore = chePlaceLocaleSuPorta (porta);  // ottengo il place in base alla "porta"
    parentela.addElement (new Cella(placeFiglio, placeGenitore));
  } //ricordaRegistrazioneLocaleDNS


  // OPERAZIONI CHE RENDONO DATI "UTILI", CERCANDOLI NELLA CATENA DI
  // COLLEGAMENTI TRA OGGETTI (CHE INIZIANO DA UN OGGETTO "Environment")

      /**
       *  Dato un {@link SOMA.naming.PlaceID} ne rende il suo {@link SOMA.Environment}
       *  (precedentemente salvato in "libroMastro").
       */
  public synchronized Environment cheEnvironment (PlaceID placeID) {
    Enumeration enum = libroMastro.elements();
    Environment env = null;
    boolean trov = false;
    while (enum.hasMoreElements() && (! trov)) {
      env = (Environment) enum.nextElement();
      trov = env.placeID.equals(placeID);  // trovato, S/N?
    }
    return env;
  } //cheEnvironment

      /**
       *  Aggiunge il PlaceID all'elenco dato in maniera ORDINATA!
       *  (si basa sul "toString").
       */
  protected void aggiungiOrdinato (Vector elenco, Object nuovo) {
    Enumeration enum = elenco.elements();        // Scorre gli elementi gi presenti
    int i = 0;
    boolean cont = true;
    String nuovoStr = nuovo.toString();
    while (enum.hasMoreElements() && cont) {
      String presoStr = enum.nextElement().toString();
      if (nuovoStr.compareTo(presoStr) < 0)     // nuovo e' < di preso ?
        cont = false;
      else
        i++;
    }
    if (i <= elenco.size())
      try {
        elenco.insertElementAt (nuovo, i);   // Inserisco nella posizione trovata
      }
      catch (Exception ecc) { 
        elenco.addElement (nuovo); 
      }
    else
       elenco.addElement (nuovo);             // Oppure alla fine del vettore
  } //aggiungiOrdinato

      /**
       *  Dato un {@link SOMA.naming.PlaceID} ne rende il suo {@link SOMA.gui.ActionPlace}
       *  (ritrovandolo dalla catena di oggetti che partono dall'Environment).
       */
  public synchronized ActionPlace cheActionPlace (PlaceID placeID) {
    Environment env = cheEnvironment (placeID);
    return cheActionPlace (env);
  } //cheActionPlace

      /**
       *  Dato un {@link SOMA.Environment} ne rende il suo {@link SOMA.gui.ActionPlace}
       *  (ritrovandolo dall'oggetto DaemonExplorerItem contenuto nel MENU').
       */
  public synchronized ActionPlace cheActionPlace (Environment env) {
    return env.actionPlace;
/* PRECEDENTE VERSIONE - quando non c'era ancora il "campo" in "Environment" -
    if (env == null) return null;
    DirExplorerItem dir = env.dir;
    DaemonExplorerItem mioDaemon = (DaemonExplorerItem) dir.getItem("window");
    ActionPlace mioActionPlace = (ActionPlace) mioDaemon.daemon;
    return mioActionPlace;
*/
  } //cheActionPlace

      /**
       *  Dato un {@link SOMA.naming.PlaceID} ne rende il suo Numero di Porta
       *  (ritrovandolo dalla catena di oggetti che partono dall'Environment).
       */
  public int chePorta (PlaceID placeID) {
    // Dato un PlaceID ne rende il numero di porta a cui  stato registrato il networkManager
    Environment env = cheEnvironment (placeID);
    return chePorta (env);
  } //chePorta

      /**
       *  Dato un {@link SOMA.Environment} ne rende il suo Numero di Porta
       *  (ritrovandolo dalla catena di oggetti che partono dall'Environment).
       */
  public int chePorta (Environment env) {
    if (env == null) return 0;
    return env.networkManager.placeInfo.port;
  } //chePorta

      /**
       *  Dato un Numero di Porta rende il {@link SOMA.naming.PlaceID} che la sta usando
       *  (ritrovandolo dalla catena di oggetti che partono dall'Environment e dai place salvati in "libroMastro" ).
       */
  public PlaceID chePlaceLocaleSuPorta (int porta) {
    Enumeration enum = libroMastro.elements();
    Environment env;
    PlaceID place = null;
    boolean trov = false;
    while (enum.hasMoreElements() && (! trov)) {
      env = (Environment) enum.nextElement();
      if (porta == chePorta(env)) {
        place = env.placeID;
        trov = true;
      }
    }
    return place;
  } //chePlaceLocaleSuPorta

      /**
       *  Dato un {@link SOMA.naming.PlaceID} ne rende il suo Indirizzo di Host
       *  (ossia l'indirizzo a cui  stato registrato il networkManager)
       *  (ritrovandolo dalla catena di oggetti che partono dall'Environment).
       */
  public InetAddress cheHost (PlaceID placeID) {
    Environment env = cheEnvironment (placeID);
    return cheHost (env);
  } //cheHost

      /**
       *  Dato un {@link SOMA.Environment} ne rende il suo Indirizzo di Host
       *  (ossia l'indirizzo a cui  stato registrato il networkManager)
       *  (ritrovandolo dalla catena di oggetti che partono dall'Environment).
       */
  public InetAddress cheHost (Environment env) {
    if (env == null) return null;
    return env.networkManager.placeInfo.host;
  } //cheHost

      /**
       *  Rende il PlaceID del DNS GENITORE del PlaceID passato (ogni DNS ha al pi un solo padre).
       *  Viene reso "null" se non ha padre ( il DNS radice).
       *  NON SI ACCEDE AL "DNS" (poich ha variabili "protette") ma all'oggetto "Anagrafe".
       *  NB: il placeID passato DEVE essere un place di default!
       */
  public synchronized PlaceID rendiGenitoreDNS (PlaceID placeID) {
    Enumeration enum = parentela.elements();     // Scorro la lista delle REGISTRAZIONI
    Cella cella = null;
    boolean trov = false;
    while (enum.hasMoreElements() && (! trov)) {
      cella = (Cella) enum.nextElement();
      trov = placeID.equals(cella.placeFigliuolo);
    } //while
    if (trov)
      return cella.placeBabbo;
    else
      return null;
  } //rendiGenitoreDNS

      /**
       *  Rende l'elenco dei FIGLI DEL DNS PlaceID passato (ogni DNS pu avere zero, uno o pi figli).
       *  Viene reso una lista senza elementi se non ha nessun figlio ( un DNS foglia).
       *  Non si accede al "DNS" (poich ha variabili "protette") ma all'oggetto "Anagrafe".
       *  NB: il placeID passato DEVE essere un place di default!
       *  Il Vector reso contiene oggetti di classe PlaceID.
       */
  public synchronized Vector rendiFigliDNS (PlaceID placeID) {
    Vector elenco = new Vector();
    Enumeration enum = parentela.elements();     // Scorro la lista delle REGISTRAZIONI
    Cella cella;
    while (enum.hasMoreElements()) {
      cella = (Cella) enum.nextElement();
      if (placeID.equals(cella.placeBabbo))
        elenco.addElement (cella.placeFigliuolo);
    } //while
    return elenco;
  } //rendiFigliDNS


  // GENERAZIONE EVENTO DI MODIFICA ANAGRAFE

      /**
       *  Ci si registra come LISTENER dell'oggetto Anagrafe.
       *  Ogni listener deve implementare l'interfaccia "AnagrafeListener" 
       *  e sara' avvisato quando avviene una modifica all'Anagrafe (battesimo 
       *  di un nuovo place).
       *  Anche se l'accostamento delle parole inglesi "add" e "Listener" con
       *  l'italiano "Anagrafe" non  molto grazioso, l'ho fatto per mantenere
       *  la "compatibilit" di sintassi definita dal modello a eventi di Java.
       */
  public void addAnagrafeListener (AnagrafeListener cheListener) {

    listenerList.add (AnagrafeListener.class, cheListener);

  } //addAnagrafeListener

      /**
       *  Funzione inversa di "addAnagrafeListener"; serve per annullare
       *  una precedente registrazione di un Listener.
       */
  public void removeAnagrafeListener (AnagrafeListener cheListener) {

    listenerList.remove (AnagrafeListener.class, cheListener);

  } //removeAnagrafeListener

      /**
       *  Metodo INTERNO, usato per notificare a tutti i listener (che si sono 
       * registrati con "addAnagrafeListener") che  avvenuto un evento di "modifica d'Anagrafe".
       */
  protected void fireAnagrafeCambiata () {

    // Prelevo l'elenco dei listener e li metto in un array
    Object[] listeners = listenerList.getListenerList();

    // NON MI SERVE CREARE UN EVENTO DA DARE con "anagrafeCambiata(anagrafeEvent)"
    // Se mi servisse farei:
    //    if (anagrafeEvent == null)
    //      anagrafeEvent = new AnagrafeEvent(this);

    // Processo i listener in lista dall'ULTIMO al PRIMO e
    // gli notifico un cambio di anagrafe!
    int i = listeners.length-2;
    while (i >= 0) {
      if (listeners[i] == AnagrafeListener.class)
        ((AnagrafeListener)listeners[i+1]).anagrafeCambiata();
      i=i-2;
    }
  } //fireAnagrafeCambiata


  // COSTRUZIONE DINAMICA DI STRUTTURE DATI IN BASE ALLA CORRENTE CONFIGURAZIONE

      /**
       *  Rende un {@link java.util.Vector} contenente un elenco di {@link SOMA.naming.PlaceID}.
       *  che corrisponde a una lista di TUTTI I PLACE "battezzati" (sia dei place 
       *  di default sia di quelli "normali")
       */
  public synchronized Vector listaOgniPlace () {
    Vector reso = new Vector (libroMastro.size());
    PlaceID placeID;
    Enumeration enum = libroMastro.elements();
    while (enum.hasMoreElements()) {
      placeID = ( (Environment) enum.nextElement() ).placeID;
      reso.addElement (placeID);
    }
    return reso;
  } //listaOgniPlace

      /**
       *  Rende un {@link java.util.Vector} contenente un elenco di {@link SOMA.naming.PlaceID}.
       *  che corrisponde a una lista di TUTTI I PLACE DI DEFAULT "battezzati".
       */
  public synchronized Vector listaPlaceDefault () {
    Vector reso = new Vector (libroMastro.size());
    PlaceID placeID;
    Enumeration enum = libroMastro.elements();
    while (enum.hasMoreElements()) {
      placeID = ( (Environment) enum.nextElement() ).placeID;
      if (placeID.isDomain())          // Lo inserisco solo se  un dominio
        reso.addElement (placeID);
    }
    return reso;
  } //listaPlaceDefault

      /**
       *  Rende un {@link java.util.Vector} contenente un elenco di {@link SOMA.naming.PlaceID}.
       *  che corrisponde a una lista di TUTTI I PLACE ("battezzati") CHE SONO 
       *  CONTENUTI NEL DOMINIO PASSATO.
       */
  public synchronized Vector listaPlaceInDominio (PlaceID dominio) {
    Vector reso = new Vector (libroMastro.size());
    PlaceID placeID;
    Enumeration enum = libroMastro.elements();
    while (enum.hasMoreElements()) {
      placeID = ( (Environment) enum.nextElement() ).placeID;
      if (placeID.sameDomain(dominio))   // Lo inserisco solo se  nello stesso dominio
        reso.addElement (placeID);
    }
    return reso;
  } //listaPlaceInDominio

      /**
       *  Rende un modello di un albero ({@link javax.swing.tree.DefaultTreeModel})
       *  il cui albero  costituito da nodi di classe {@link javax.swing.tree.DefaultMutableTreeNode}
       *  in cui  incapsulato un oggetto di classe {@link SOMA.naming.PlaceID}.
       *  Tale albero ha due soli livelli; a profondit "uno" ci sono tutti i place
       *  di default mentre a profondit "due" ci sono tutti gli altri place,
       *  ognuno dei quali "discende" dal place di default del proprio dominio.
       *  In pratica  la relazione di "appartenenza" dei place ai place di default.
       *  Nota: la radice dell'albero non  rilevante e perci non verr visualizzata.
       */
  public synchronized DefaultTreeModel alberoPlaceNeiDominii () {

    DefaultMutableTreeNode radice = new DefaultMutableTreeNode("(radice)");

    // SCORRO TUTTI I PLACE E CONSIDERO SOLO QUELLI DI DEFAULT
    PlaceID  dominioID, placeID;
    DefaultMutableTreeNode  nodoDominio, nodoPlace;
    Enumeration enum = libroMastro.elements();
    while (enum.hasMoreElements()) {
      dominioID = ( (Environment) enum.nextElement() ).placeID;
      if (dominioID.isDomain()) {         // Va bene solo se  un dominio

        nodoDominio = new DefaultMutableTreeNode (dominioID);
        radice.add (nodoDominio);

        // ORA RISCORRO TUTTO L'ELENCO IN CERCA DEI SOTTODOMINII DEL DOM.TROVATO
        Enumeration enum2 = libroMastro.elements();
        while (enum2.hasMoreElements()) {
          placeID = ( (Environment) enum2.nextElement() ).placeID;
          if (placeID.sameDomain(dominioID) &&       // Lo inserisco se  nello STESSO DOMINIO
              (! placeID.equals(dominioID))    ) {   // MA NON E' il default-place!
                 nodoPlace = new DefaultMutableTreeNode (placeID);
                 nodoDominio.add (nodoPlace);
          } //if
        } //while
      } //if
    } //while
 
    DefaultTreeModel modello = new DefaultTreeModel(radice);
    return modello;
  } //alberoPlaceNeiDominii

      /**
       *  Rende un modello di un albero ({@link javax.swing.tree.DefaultTreeModel})
       *  il cui albero  costituito da nodi di classe {@link javax.swing.tree.DefaultMutableTreeNode}
       *  in cui  incapsulato un oggetto di classe {@link SOMA.naming.PlaceID}.
       *  Tale albero rappresenta la relazione di "gerarchia" tra i Domain-Name-Service 
       *  dei place di default.
       *  Perci pu avere una profondit non predicibile.
       */
  public synchronized DefaultTreeModel alberoGerarchiaDNS () {

    DefaultMutableTreeNode radice = CreaSottoAlberoGerarchiaDNS (null, 0);   // null = nessun place (padre)!
    DefaultTreeModel modello = new DefaultTreeModel(radice);

    return modello;
  } //alberoGerarchiaDNS

      /**
       *  Metodo interno, usato per creare "ricorsivamente" l'albero di gerarchia dei DNS.
       *  Nota: per non "ciclare" all'infinito in caso di una possibile creazione
       *  errata della gerarchia (da parte dell'utente) si  scelto di limitare la
       *  profondit massima dell'albero a trenta (che poi, per pura coincidenza,  
       *  uguale al voto massimo a cui uno studente universitario pu aspirare).
       */
  protected synchronized DefaultMutableTreeNode CreaSottoAlberoGerarchiaDNS (PlaceID placeAtt, int livello) {

    DefaultMutableTreeNode nodoAtt;
    if (placeAtt == null)
      nodoAtt = new DefaultMutableTreeNode ("root");
                // radice fittizia (NB: esiste il caso in cui posso avere pi DNS radice (alberi separati)
    else
      nodoAtt = new DefaultMutableTreeNode (placeAtt);


    livello++;
    if (livello < 30) {       // Vorrei evitare LOOP INFINITI (in caso di errore di creazione gerarchia)

      Enumeration enum = parentela.elements();     // Scorro la lista delle REGISTRAZIONI
      while (enum.hasMoreElements()) {
        Cella cella = (Cella) enum.nextElement();

        boolean vaBene;
        if (placeAtt == null)                           // Sto cercando i nodi di DNS-RADICE  
          vaBene = cella.placeBabbo == null;            // Se non ha nessun padre,  un DNS radice (l'ho trovato!)
        else
          vaBene = placeAtt.equals(cella.placeBabbo);   // Se no, va bene se  figlio del "nodo attuale"
        
        if (vaBene) {                                   // Se il nodo attuale "va bene",

          DefaultMutableTreeNode sottoAlberoFiglio = CreaSottoAlberoGerarchiaDNS (cella.placeFigliuolo, livello);
          // RICORSIOVAMENTE inserisce tutti i figli del figlio trovato!

          nodoAtt.add (sottoAlberoFiglio);
        } //if
      } //while

    } //if

    return nodoAtt;       // Rende il nodo contenente placeAtt e tutta la sua sottostruttura
  } //CreaSottoAlberoGerarchiaDNS


/*
  public static void main (String[] args) {
    // PER DEBUGGING
    DirExplorerItem root = new DirExplorerItem( "root" );
    root.addItem( "newDomain", new SOMA.NewDomainExplorerItem( root ) );
    root.addItem ("window", new DaemonExplorerItem( new SOMA.gui.ActionPlace (null) ));
    root.addItem( "newPlace", new SOMA.NewPlaceExplorerItem( root ) );
    DaemonExplorerItem mioDaemon = (DaemonExplorerItem) root.getItem("window");
    System.out.println (mioDaemon.getClass());
    ActionPlace mioActionPlace = (ActionPlace) mioDaemon.daemon;
    System.out.println (mioActionPlace.getClass());
  }
*/

} //Anagrafe