package SOMA.agent;

import SOMA.naming.*;

/**
* <P>Classe astratta che rappresenta un agente.
* </P>
* <P> Per creare un agente e' necessario sottoclassare la classe Agent
* ridefinendo i metodi {@link #run()} e,
* se necessario, {@link #putArgument( Object obj )}.
* </P>
* <P> Il place mette a disposizione dell'agente un insieme di risorse,
* i cui riferimenti si trovano nel campo {@link #agentSystem}.
* </P>
* <P> Il place associa ad ogni agente un {@link SOMA.agent.AgentWorker worker}
* che permette di mandare in esecuzione l'agente stesso.
* </P>
* <BR>
* @see SOMA.agent.AgentSystem
* @see SOMA.agent.AgentWorker
*
* @author Livio Profiri
*/

public abstract class Agent implements java.io.Serializable
{
  // Vedere le public

  /** @serial*/
  private AgentID myID = null;  // Attenzione: non voglio che sia modificato direttamente

  /** @serial
  * Metodo che verra' eseguito alla prossima attivazione dell'agente.
  */
  public String start = "run";
  //public boolean Traceable = false;
  //public Mailbox Mail = null;

  /** <P>Interfaccia fra agente e sistema.
  * </P><P> <code>agentSystem</code> contiene i riferimenti agli oggetti cui
  * l'agente ha accesso. In questo modo e' garantita la protezione del sistema da
  * possibili attacchi di agenti ostili, a cui non verranno forniti riferimenti
  * a risorse critiche.
  * </P><P>  Questo campo e' <code>transient</code> perche' viene asegnato all'agente
  * dal place in cui si trova.
  * </P>
  */
  public transient AgentSystem agentSystem = null;

  public Mailbox mailbox = null;
  /** @see #setTraceable( boolean ) */
  public boolean traceable = false;

  /** Riferiomento al worker dell'agente.
  * <BR><BR><B>Attenzione:</B> il campo worker ha visibilit di package perch
  * non deve essere visto dalle sottoclassi di Agent che si trovano in un altro package,
  * ossia dagli agenti creati dall'utente.
  * <BR>  Questo campo e' <code>transient</code> perche' viene asegnato all'agente
  * dal place in cui si trova.
  */
  transient AgentWorker worker = null;

  /** Costruttore vuoto. */
  public Agent()
  {
  }

  /** <P>Permette di specificare l'ID di un agente.
  *
  * <P> L'implementazione permette di specificare l'ID di un agente una sola volta.
  * Quest'operazione  fatta dal sistema al momento della creazione di un agente:
  * {@link SOMA.agent.mobility.AgentManager#createAgent( String agentName, Object argument )}.
  * Se si fosse usato un costruttore, anche tutte le sottoclassi lo avrebbero dovuto fornire.
  * </P>
  */
  // una volta fissato l'agentID non si pu pi cambiare!
  // non lo metto nel costruttore per non complicare troppo la procedura di creazione agenti.
  public final void setID( AgentID myID )
  {
    if( this.myID == null )
      this.myID = myID;
    else
      throw new RuntimeException( "Agent.setID: trying to change agentID!!!" );
  }

  /** Restituisce l'identificatore dell'agente.*/
  public final AgentID getID()
  {
    return myID;
  }

  /** Rende rintracciabile o meno l'agente.
  *
  * <P> Se l'agente e' rintracciabile e non ha ancora una mailbox, viene creata.
  *
  * <P> Attenzione: {@link #traceable} e' <code>public</code>, quindi e' comunque
  * possibile cambiarne il valore.
  */
  public final void setTraceable( boolean traceable )
  {
    this.traceable = traceable;

    if( traceable && (mailbox == null) )
        mailbox = new Mailbox();
  }

  /** Permette di definire lo stato iniziale dell'agente.
  * <BR> Questo metodo  vuoto in nella classe Agent e
  * deve essere ridefinito dalle sottoclassi che implementano agenti,
  * in maniera analoga a {@link #run()}.
  * @param obj Un oggetto contenente informazioni di inizializzazione.
  * Ovviamente l'oggetto pu anche contenere una struttura dati complessa.
  */
  public void putArgument( Object obj )
  {}

  /** Metodo mandato in esecuzione all'avvio di un agente.
  * <BR> Questo metodo astratto va ridefinito nell'implementazione di
  * ogni agente.
  */
  public abstract void run();

  /** <P>Metodo di migrazione verso un altro place.
  * </P><P>
  * Viene invocato il metodo {@link SOMA.agent.AgentSystem#go( Agent agent, PlaceID destination )}.
  * </P>
  * @param destination Il place di destinazione, che pu appartenere ad un dominio qualsiasi.
  * @param method Il metodo con cui verra' riattivato l'agente, vedi {@link #start}
  * <BR><BR>
  * <P>
  * <B>Attenzione:</B> se un agente ha il riferimento ad un altro agente pu chiamarne il metodo di
  * migrazione, come ogni altro metodo. In sostanza ne pu assumere il controllo completo.
  * E' quindi estremamente opportuno mantenere nascosti ad ogni agente i
  * riferiemnti ad altri agenti.
  * </P>
  */
  public final void go( PlaceID destination, String method ) throws CantGoException
  {
    start = method;
    agentSystem.go( this, destination );
  }

  /** <P>Metodo di messa in attesa di un agente.
  */
  public final void idle( String method )
  {
    start = method;
    // Passo attraverso agentSystem per coerenza:
    //   voglio che tutte le chiamate facciano riferimento a quella classe,
    //   cosicch  sufficiente ridefinirla per cambiare il comportamento del sistema.
    agentSystem.idle( this );
  }

  /**
  * Ridefinire questo metodo per liberare le risorse dell'agente e
  * lasciarlo in uno stato consistente, nel caso sia necessario interromperlo.
  *
  * <P> Ricordarsi aggiornare la variabile start con il nome del metodo da lanciare
  * al prossimo avvio dell'agente.
  */
  public void stop()
  {
    throw( new UnsupportedOperationException( "Abstract Agent doesn't support this operation." ) );
  }


  /** Fornisce una rappresentazione in forma di stringa dell'agente.
  */
  public String toString()
  {
    return "[Agent: " + myID + (mailbox == null ? " not traceable" : " " + mailbox) + "]";
  }
}
