package SOMA.agent.classLoading;

import java.io.*;

/** <P>Permette di caricare e salvare classi dal file system.
*
* <P> Le classi non sono caricate in formato <code>.class</code>,
* ma sotto forma di <code>byte[]</code> in modo da essere direttamente
* utilizzabili dai classloader per definire classi.
*
* @author Livio Profiri
*/
public class ClassManager
{
  /** Punto di partenza per la ricerca delle classi.
  * Voglio che termini con un FILE.separator, in modo da poter
  * subito concatenare il nome della classe cercata.
  */
  File classPathDirectory;
  final boolean createDirectory;

  /** Costruttore.
  * Equivale all'altro costruttore con <code>createDirectory</code> a <code>false</code>
  */
  public ClassManager( String classPath ) throws ClassManagerException
  {
    this( classPath, false );
  }

  /** Costruttore.
  * @param classPath Il direttorio in cui ricercare le classi.
  * @createDirectory Se createDirectory e a <code>true</code> creo il direttorio.
  */
  public ClassManager( String classPath, boolean createDirectory ) throws ClassManagerException
  {
    this.createDirectory = createDirectory;
    classPathDirectory = new File( classPath );

    if( !classPathDirectory.isDirectory() )
      if( createDirectory )
      {
        if( !classPathDirectory.mkdirs() )
          throw new ClassManagerException( "Can't create dir: " + classPathDirectory );
      }
      //else
        //throw new ClassManagerException( "Dir: " + classPathDirectory + " does not exist." );
  }

  /** Restituisce {@link #classPathDirectory}. */
  public String getClassPathDirectory()
  {
    return classPathDirectory.getAbsolutePath();
  }

  /** Restituisce {@link #createDirectory}. */
  public boolean getCreateDirectory()
  {
    return createDirectory;
  }

  /** Carica un file ".class" da disco
  * andandolo a cercare nella directory specificata da {@link #classPathDirectory}.
  *
  * @param className Il nome della classe
  * @return I byte del file.
  */
	public byte[] loadClassFile( String className ) throws IOException
	{
    byte[] returnData = null;

	  BufferedInputStream myBIS =
	    new BufferedInputStream(
	      new FileInputStream(
	        new File( classPathDirectory,
	                  className.replace( '.', '/' ) + ".class" ) ) );

	  int fileLen = myBIS.available(),
	      readBytes = 0,
	      readNow;

		returnData = new byte[fileLen];

		while (readBytes < fileLen)
		{
			readNow = myBIS.read(returnData, readBytes, fileLen-readBytes);
			readBytes += readNow;
		}

		myBIS.close();

		return returnData;
	}

	/** Salva un file ".class" su disco
  * andandolo a cercare nella directory specificata da {@link #classPathDirectory}.
  *
  * @param className Il nome della classe
  * @param data      I byte del file.
  */

	void saveClassFile( String className, byte[] data) throws ClassManagerException,
	                                                          IOException
	{
	  File fileToSave = new File( classPathDirectory,
	                 className.replace( '.', '/' ) + ".class" );

	  //System.out.println( "Saving file: " + fileToSave );

	  // Creo la directory se non esiste!
	  File parentDir = fileToSave.getParentFile();
	  if( parentDir != null && !parentDir.isDirectory() && !parentDir.mkdirs() )
	    throw new ClassManagerException( "Can't create dir: " + classPathDirectory );


	  BufferedOutputStream myBOS =
	    new BufferedOutputStream(
	      new FileOutputStream( fileToSave ) );

    myBOS.write(data);
		myBOS.flush();
		myBOS.close();

		//System.out.println( "SAVED FILE: " + fileToSave );
	}

  /** Eccezione del ClassManager.
  *
  * @author Livio Profiri
  */
  public static class ClassManagerException extends Exception
  {
    public ClassManagerException( String s )
    {
      super( s );
    }
  }
}
