/**  AcceptPlace
 *     Thread che esegue la sequenza di operazioni (comandi) che arrivano da un'applet remota 
 *     per un certo Place (dato dall'ActionPlace).
 *     Questo thread viene creato e lanciato da "SportelloRichieste", un nuovo "accettatore"
 *     per ogni connessione che viene accettata.
 *     @author     Luigi Antenucci
 *     @version    1.6
 *     @language   jdk 1.2.2
 */

package SOMA.gui.remotegui;

import SOMA.gui.ActionPlace;
import SOMA.gui.Debug;

import java.lang.Thread;
import java.net.Socket;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;


public class AcceptPlace extends Thread {

      /**
       *  Parametri passati al costruttore.
       */
  protected Socket cliSock;

      /**
       *  RIFERIMENTO AL PROPRIO "ActionPlace" (passato al costruttore).
       */
  protected ActionPlace actionPlace;

      /**
       *  Variabile di classe per distinguere con un numero progressivo i vari thread concorrenti
       */
  protected static int numIstanza = 0;

      /**
       *  Nome dell'oggetto, con contatore progressivo d'istanza incorporato.
       */
  protected String mioNome;

      /**
       *  Variabile di classe per contare il numero di thread attualmente in esecuzione.
       */
  protected static int inEsecuzione = 0;


      /**
       *  Costruttore dell'accettatore.
       *  "cliSock" e' la socket che viene resa dopo un "accept"; 
       *  da essa questo thread legger i dati.
       */
  public AcceptPlace (Socket cliSock, ActionPlace actionPlace) {
    numIstanza++;
    String O;
    if (numIstanza <= 9) O = "0";
    else                 O = "";
    mioNome = "AccettPlace"+O+String.valueOf(numIstanza);;
    setName (mioNome);

    this.cliSock     = cliSock;
    this.actionPlace = actionPlace;
  } // costruttore

      /**
       *  Metodo di esecuzione del thread!
       */
  public void run () {

    Debug.outln (mioNome+" - lanciato");

    inEsecuzione++;

    try {

      ObjectInputStream  objIn  = new ObjectInputStream  (cliSock.getInputStream());
      ObjectOutputStream objOut = new ObjectOutputStream (cliSock.getOutputStream());

      Mess mess = riceviMessaggio (objIn);

      if (mess instanceof MessDelLancia)
        esegui_lanciaAgente ((MessDelLancia)mess, objOut);

      else if (mess instanceof MessDelDammi)
        esegui_DammiQualcosa ((MessDelDammi)mess, objOut);

      else if (mess instanceof MessDelAYA)
        rispondi_AreYouAlive (objOut);
      
      else Debug.outln (mioNome+" - RICEVUTO UN MESSAGGIO SCONOSCIUTO!");

    }
    catch (Exception err) {
      Debug.outln (mioNome+" - ERRORE: "+err);
    }
    finally {

      chiudiConnessione (cliSock);    // CHIUDO QUI LA CONNESSIONE

      inEsecuzione--;

      Debug.outln (mioNome+" - terminato");
    }

  } //run

      /**
       *  Rende il numero di istanze di questo oggetto che sono attualmente in esecuzione.
       *  Utile per negare nuove creazioni di oggetti di questo tipo in caso si superi
       *  un limite prefissato.
       */
  public static int quantiInEsecuzione () {
    return inEsecuzione;
  } //quantiInEsecuzione

      /**
       *  Interna, chiude la comunicazione con la socket passata.
       */
  protected void chiudiConnessione (Socket sock) {

    Debug.outln ("chiudendo la connessione");

    try {
    	sock.close();
    }
    catch (Exception e) { }
  } //chiudiConnessione

      /**
       *  Interna, invia il messaggi passato (un oggetto "polimorfico" di classe base "Mess") sullo
       *  stream di oggetti passato (in genere  quello generato tramite l'uso della socket resa 
       *  dopo una "creaConnessione").
       *  Pu generare un'eccezione in caso di errore di spedizione (si usa il supporto di TCP).
       */
  protected void inviaMessaggio (ObjectOutputStream objOut, Mess mess) throws Exception {

    Debug.outln ("inviando il messaggio "+mess.toString());

    objOut.writeObject (mess);
    objOut.flush();
    
  } //inviaMessaggio

      /**
       *  Interna, riceve un messaggio dallo stream di oggetti passato (in genere  quello generato 
       *  tramite l'uso della socket resa dopo una "creaConnessione").
       *  Pu generare un'eccezione in caso di errore di ricezione (si usa il supporto di TCP).
       *  Nota: sar necessario un "casting" sull'oggetto "Mess" reso.
       */
  protected Mess riceviMessaggio (ObjectInputStream objIn) throws Exception {

    Mess mess = (Mess) objIn.readObject ();

    Debug.outln ("ricevuto il messaggio "+mess.toString());

    return mess;

  } //riceviMessaggio

  protected void esegui_lanciaAgente (MessDelLancia mess, ObjectOutputStream objOut) {
    MessBid messRisp;
    try {
      // LANCIO L'AGENTE COI DATI CONTENUTI NEL MESSAGGIO
      actionPlace.lanciaAgente (mess.nomeAgente, mess.arrParam,
                                mess.usaSysClassLoader, mess.rintracciabile,
                                mess.nonFarloPartire);

      messRisp = new MessBidMess ("Fatto");

    }
    catch (Exception eccez) {
      messRisp = new MessBidErrore (eccez);
    }

    try {
      inviaMessaggio (objOut, messRisp);
    }
    catch (Exception e) { }

  } //esegui_lanciaAgente

  protected void esegui_DammiQualcosa (MessDelDammi mess, ObjectOutputStream objOut) {
    Object oggettoDaRendere;

    switch (mess.dammiChe) {  // TUTTI I CASI POSSIBILI
      case MessDelDammi.DAMMI_PLACE_ID  : { oggettoDaRendere = actionPlace.chePlaceID();         break; }
      case MessDelDammi.DAMMI_AGENTI    : { oggettoDaRendere = actionPlace.cheElencoAgenti();    break; }
      case MessDelDammi.DAMMI_DOMINII   : { oggettoDaRendere = actionPlace.elencoDominiiInDNS(); break; }
      case MessDelDammi.DAMMI_PLACE     : { oggettoDaRendere = actionPlace.elencoPlaceInPNS();   break; }
      case MessDelDammi.DAMMI_ALB_THREAD: { oggettoDaRendere = actionPlace.alberoPlaceThread();  break; }
      case MessDelDammi.DAMMI_AGENT_EXE : { oggettoDaRendere = actionPlace.elencoAgentiInEsecuzione(); break; }
      case MessDelDammi.DAMMI_AGENT_POS : { oggettoDaRendere = actionPlace.elencoPosizioneAgenti(); break; }
      default:  oggettoDaRendere = null;
    }
    
    if (oggettoDaRendere != null) {
      Debug.outln ("inviando l'oggetto di classe "+oggettoDaRendere.getClass().getName());
      try {

        objOut.writeObject (oggettoDaRendere);
        objOut.flush();      	

      }
      catch (Exception e) { }
    }
    else
      Debug.outln ("inviando un oggetto NULL (non rendo nulla, chiudo la connessione)");

  } //esegui_DammiQualcosa

  protected void rispondi_AreYouAlive (ObjectOutputStream objOut) {

    MessAccIAA messIAA = new MessAccIAA ();

    try {
      inviaMessaggio (objOut, messIAA);
    }
    catch (Exception e) { }

  } //rispondi_AreYouAlive

} //AcceptPlace