/**  ModelloPNS
 *     @author     Luigi Antenucci
 *     @version    1.0
 *     @language   jdk 1.2.2
 */

package SOMA.gui;

import SOMA.gui.lingua.Lingua;
import SOMA.Environment;
import SOMA.naming.PlaceID;
import SOMA.naming.place.PlaceInfo;
import SOMA.naming.place.PlaceNameService;

import java.net.InetAddress;
import javax.swing.table.TableModel;
import javax.swing.table.AbstractTableModel;


public class ModelloPNS implements ModelloXNS {

      /**
       *  L'Environment del place; in questo oggetto  contenuto il riferimento al PNS (vedi dopo).
       */
  protected Environment mioEnv;

      /**
       *  Il Place Name Service a cui fare riferimento.
       *  Lo "tiro fuori" da "mioEnv" per un accesso diretto.
       */
  protected PlaceNameService mioPNS;

      /**
       *  La finestra a cui (eventualmente) fa riferimento questo modello.
       */
  protected FinestraXNS miaFinestraXNS = null;

      /**
       *  Costruisce il Modello per il PNS dell'Environment passato come parametro
       *  e ne apre la finestra.
       */
  public ModelloPNS (Environment env) {
    this (env, true);
  } //costruttore

      /**
       *  Costruisce il Modello per il PNS dell'Environment passato come parametro
       *  e ne apre la finestra (con tutti i bottoni di modifica) a seconda del 
       *  valore di "apriFin".
       */
  public ModelloPNS (Environment env, boolean apriFin) {
    Lingua.caricaFileDiLingua (ModelloPNS.class);  // Carico file di lingua per questa classe
    mioEnv = env;
    mioPNS = env.placeNameService;
    if (apriFin) {
      miaFinestraXNS = new FinestraXNS (this);
    }
  } //costruttore

      /**
       *  Costruisce il Modello per il PNS dell'Environment passato come parametro
       *  e ne apre la finestra relativa (in ogni caso).
       *  La finestra avr o meno i bottoni di modifica a seconda del valore di 
       *  "finXNSModif" (vedi costanti in {@link SOMA.gui.FinestraXNS}).
       */
  public ModelloPNS (Environment env, int finXNSModif) {
    Lingua.caricaFileDiLingua (ModelloPNS.class);  // Carico file di lingua per questa classe
    mioEnv = env;
    mioPNS = env.placeNameService;
    miaFinestraXNS = new FinestraXNS (this, finXNSModif);
  } //costruttore

      /**
       *  Rende una stringa contenente il tipo di Name-Service a cui il modello fa riferimento
       */
  public String cheTipo () {
    return (Lingua.frase("MP_WIN_TITLE"));
  } //cheTipo

      /**
       *  Rende il PlaceID del place a cui appartiene il Name-Service del modello
       */
  public PlaceID perChePlace () {
    return mioEnv.placeID;
  } //perChePlace

      /**
       *  Rende una tabella che rappresenta la stessa tabella contenuta nel Name-Service.
       *  Il TableModel conterr tutti oggetti di classe PlaceInfo.
       *  Vedi {@link SOMA.naming.place.PlaceInfo}
       */
  public TableModel cheTabella () {
    return new TabellaPNS();
  } //cheTabella

      /**
       *  Rende un "Pannello" contenente informazioni aggiuntive sul Name-Service.
       */
  public Pannello pannelloInfoInPiu () {
    return null;  // NESSUN PANNELLO PER IL PNS
  } //pannelloInfoInPiu

      /**
       *  Invocazione del metodo "put" del PNS.
       *  Tutte le informazioni sono contenute nell'oggetto passato, che sar della classe opportuna.
       */
  public void invocaPut (Object informazioni) {
    mioPNS.putPlace ((PlaceInfo)informazioni);
  } //invocaPut

      /**
       *  Invocazione del metodo "del" del PNS.
       *  Viene eseguito sul PlaceID relativo alla colonna "col" passata.
       */
  public void invocaDel (PlaceID placeID) {
    mioPNS.removePlace (placeID);
  } //invocaDel

      /**
       *  Invocazione del metodo "refresh" del PNS.
       */
  public void invocaRefresh () {
    mioPNS.refresh ();
  } //invocaRefresh

      /**
       *  Rende true se si pu invocare il "refresh", ossia se il place corrente NON  un 
       *  place di default.
       */
  public boolean refreshInvocabile () {
    return ( ! mioEnv.placeID.isDomain() );
  } //refreshInvocabile


      /**
       *  CLASSE INTERNA CHE DEFINISCE IL "MODELLO" A CUI SI RIFERISCE LA TABELLA DEL PNS.
       *  L'istanza di questa classe  resa dal metodo "cheTabella".
       *  Vedi classe {@link javax.swing.table.AbstractTableModel}
       *  Nota Bene: c' un side-effect sulla variabile "mioPNS" della classe contenitrice!
       */
  protected class TabellaPNS extends AbstractTableModel {

      /**
       *  Rende il numero di colonne della tabella.
       */
    public int getColumnCount () { 
      return 3;
    } //getColumnCount

      /**
       *  Rende il numero di righe della tabella.
       */
    public int getRowCount () { 
      // Chiedo al PNS l'elenco di tutti i place
      PlaceID[] arrPlaceID = mioPNS.getPlacesArray();
      return arrPlaceID.length;
    } //getRowCount

      /**
       *  Rende il valore (sottoforma di oggetto) contenuto nella tabella in posizione (row,col).
       */
    public Object getValueAt (int row, int col) {
      // Chiedo al PNS l'elenco di tutti i place
      PlaceID[] arrPlaceID = mioPNS.getPlacesArray();

      if ((row >= 0) && (row < arrPlaceID.length)) {
        // Prelevo il PlaceInfo ralativo alla riga in esame
        PlaceInfo plaAtt = mioPNS.getPlace (arrPlaceID[row]);

        // E rendo l'oggetto relativo alla colonna
        switch (col) {
          case  0: return plaAtt.placeID;
          case  1: return plaAtt.host.getHostAddress();
          case  2: return new Integer(plaAtt.port);
          default: return null;
        }
      }
      else
        return null;
    } //getValueAt

      /**
       *  Rende il nome della colonna "col", cio l'informazione "intensiva".
       */
    public String getColumnName (int col) {
      switch (col) {
        case  0: return Lingua.frase("MP_PLACE_ID");
        case  1: return Lingua.frase("MP_ADDRESS");
        case  2: return Lingua.frase("MP_PORT");
        default: return null;
      }
    } //getColumnName

      /**
       *  Rende la classe degli oggetti relativi alla colonna "col".
       *  Necessaria per un buon "allineamento" delle stringhe a sinistra e dei numeri a destra!
       */
    public Class getColumnClass(int col) {
      return getValueAt(0,col).getClass();
    } //getColumnClass

      /**
       *  La cella in posione (row,col) pu essere modificata dall'utente?
       */
    public boolean isCellEditable (int row, int col) {
      return false;
    } //isCellEditable

  } //TabellaPNS (classe)
  
} //ModelloPNS