/**  Finestra
 *     Gestione di una generica finestra per l'interfaccia grafica.
 *     Ogni "finestra" della GUI sar sottoclasse di questa e dovr rendere
 *     "concreti" tutti i metodi che sono lasciati "astratti" in questa classe!
 *     E' importante che in ogni sottoclasse il COSTRUTTORE e "costruisciFinestra" 
 *     invochino i metodi di questa SUPERCLASSE.
 *     Questo oggetto svolge le funzioni base per la creazione e distruzione della
 *     finestra (come JFrame), la registrazione per ascoltare "cambi di lingua" e i
 *     cambi di apparenza (Look & Feel) e la crazione di un men standard.
 *     Inoltre usa l'oggetto {@link SOMA.gui.Config} per memorizzare la posizione
 *     e dimensione della finestra in modo da recuperarli al momento di una nuova
 *     creazione (anche in successive esecuzioni del programma)!
 *     @author     Luigi Antenucci
 *     @version    2.0
 *     @language   jdk 1.2.2
 */

package SOMA.gui;

import SOMA.gui.lingua.*;

import java.io.File;
import java.io.Serializable;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;


public abstract class Finestra implements LinguaListener, PropertyChangeListener, Serializable {

      /**
       *  Il percorso del file system da cui caricare le immagini.
       *  NON E' VOLUTAMENTE IMPOSTO A "final"!
       *  Pu essere necessario modificarlo poich  un percorso "relativo" a programmi
       *  lanciati nel direttorio principale di SOMA.
       */
  public static String PICTURE_PATH = "SOMA"+File.separator+"gui"+File.separator+"picture"+File.separator;

      /**
       *  La forma del cursore quando  sopra un bottone.
       */
  public static final  Cursor cursoreBottone = new Cursor (Cursor.HAND_CURSOR);

      /**
       *  Costanti per la scelta del tipo di men da creare (con l'opzione "esci" o "chiudi".
       */
  public static final  int  MENU_EXIT  = 0;
  public static final  int  MENU_CLOSE = 1;

      /**
       *  ICONE CARICATE UNA VOLTA SOLA ALLA CREAZIONE DELLA CLASSE!
       */
  protected static final ImageIcon IMG_ICO_OPZIO   = new ImageIcon(PICTURE_PATH+"Opzioni.gif");
  protected static final ImageIcon IMG_ICO_EXIT    = new ImageIcon(PICTURE_PATH+"ExitIco.gif");
  protected static final ImageIcon IMG_ICO_CLOSE   = new ImageIcon(PICTURE_PATH+"CloseIco.gif");
  protected static final ImageIcon IMG_ICO_HELP    = new ImageIcon(PICTURE_PATH+"HelpIco.gif");

  // VARIABILI DI CLASSE:
      /**
       *  Tengo il conto del numero di finestre aperte
       */
  protected static int numFinAperte = 0;

  // Variabili d'istanza:

      /**
       *  Oggetti passati al costruttore.
       */
  protected  int  X, Y;
      /**
       *  Oggetti inizializzati da "costruisciFinestra"
       *  Possono (e devono) essere usati dalle sottoclassi!
       */
  protected JFrame    mioFrame     = null;
  protected JPanel    PanCentro    = null;
  protected Container contentPane  = null;

  protected String    nomeFinestra = null;


      /**
       *  Costruisce una finestra in alto a sinistra nel video.
       *  Il titolo della finestra sar il NOME DELLA CLASSE
       *  Vedi il costruttore successivo.
       */
  public Finestra () {
    this (0, 0,  "");
  } //costruttore

      /**
       *  Costruisce una finestra alla posizione passata (X=orizzontale; Y=verticale).
       *  Il titolo della finestra sar il NOME DELLA CLASSE
       *  Vedi il costruttore successivo.
       */
  public Finestra (int X, int Y) {
    this (X, Y,  "");
  } //costruttore

      /**
       *  Costruisce una finestra alla posizione passata (X=orizzontale; Y=verticale)
       *  col titolo passato.
       *  Le indicazioni di posizione passate e la dimensione della finestra "finale"
       *  vengono usate A MENO CHE non sia gi stata aperta in passato (nel corso
       *  dell'esecuzione del programma o in esecuzioni precedenti all'attuale) una
       *  finestra con lo STESSO TITOLO.  In tal caso si useranno le vecchie impostazioni!
       *  Questo metodo chima automaticamente la {@link SOMA.gui.lingua.Lingua#caricaFileDiLingua()}
       *  sia per la classe "Finestra" sia per la sua sottoclasse!
       */
  public Finestra (int X, int Y, String nomeFin) {
    // NB nomeFin  sia il nome interno (per salvare la posizione in Config) 
    // sia il titolo della finestra
    if (nomeFin.equals(""))
      nomeFinestra = this.getClass().getName();
    else
      nomeFinestra = nomeFin;
    Debug.outln (nomeFinestra+" - NOME FIN = "+nomeFinestra);

    this.X = X;
    this.Y = Y;

    // CARICA LE FRASI IN LINGUA DA USARE
    Lingua.caricaFileDiLingua (Finestra.class);  // Carico file di lingua per questa classe
    Lingua.caricaFileDiLingua (this);            // Mi serve anche quello per la FINESTRA FIGLIA
  } //costruttore

      /**
       *  ASTRATTA! Parte di costruzione della finestra;
       *  Ogni implementazione di questo metodo deve innanzitutto chiamare 
       *  la "InizioCostruisciFinestra", quindi comporre la finestra inserendovi
       *  tutti gli oggetti che servono e al termine di tutto invocare il metodo
       *  "FineCostruisciFinestra".
       *  All'interno DEV'ESSERE invocata la "impostaFrasi" (dopo aver creato gli
       *  oggetti con le frasi nulle - per esempio i JLabel e i JButton).
       */
  public abstract void costruisciFinestra ();

      /**
       *  Inizio della costruzione della finestra.
       *  Crea fisicamente il "mioFrame" e inizializza "contentPane" e "PanCentro"
       *  Vedi {@link SOMA.gui.Finestra#costruisciFinestra()}
       */
  protected void InizioCostruisciFinestra () {
    // CREO LA FINESTRA INIZIALE (e ne prendo il contentPane)
    mioFrame     = new JFrame ();
    mioFrame.setTitle (nomeFinestra);

    mioFrame.setResizable (false);

    contentPane = mioFrame.getContentPane();
    // contentPane.setLayout (new BorderLayout());    --> E' gia' di default sul "contentPane"!

    PanCentro = new JPanel ();

    // USO Window Listener
    // Ecco una fregatura rispetto all' AWT del JDK 1.1:
    // per default la finestra viene NASCOSTA, anziche' non fare nulla!
    mioFrame.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);

  } //costruisciFinestra

      /**
       *  Fine della costruzione della finestra.
       *  Dev'essere SEMPRE invocata alla fine della propria "costruisciFinestra".
       *  Aggiunge il "PanCentro" al "contentPane", invoca la "creaBarraMenu",
       *  si mette in ascolto di eventi (cambio lingua e apparenza), definisce la
       *  vera posizione e dimensione della finestra e infine la rende visibile!
       *  Vedi {@link SOMA.gui.Finestra#costruisciFinestra()}
       */
  protected void FineCostruisciFinestra () {
    contentPane.add (PanCentro, BorderLayout.CENTER);

    creaBarraMenu ();

    // SI REGISTRA SUL CAMBIO DI LINGUA
    Lingua.addLinguaListener (this);

    // SI REGISTRA SUL CAMBIO DI LOOK AND FEEL
    UIManager.addPropertyChangeListener (this);

    mioFrame.setLocation  (X, Y);     // DEFINISCO LA LOCAZIONE
    mioFrame.pack();                  // COMPATTO I COMPONENTI DELLA FINESTRA

    Config.defPropComponent (nomeFinestra, mioFrame);

    try {
      Rectangle rettangolo = Config.chePropComponent (nomeFinestra, mioFrame);

      mioFrame.setLocation (rettangolo.x, 
                            rettangolo.y);

      if (mioFrame.isResizable()) {
        // RIPRISTINO LA DIMENSIONE PRECEDENTE
        mioFrame.setSize (rettangolo.width,
                            rettangolo.height);
      } //if

    }
    catch (Exception E) { }

    numFinAperte++;                 // Sto aprendo una nuova finestra
  
    mioFrame.setVisible (true);     // INFINE RENDO VISIBILE LA FINESTRA
  
  } //FineCostruisciFinestra

      /**
       *  Distrugge la finestra (salvando eventualmente la configurazione)
       */
  public void distruggiFinestra () {
    Debug.outln (nomeFinestra+" - distruggendo la finestra "+this.getClass());
    try {
      Config.modifPropComponent (nomeFinestra, mioFrame);
    }
    catch (Exception E) { }

    // Salva le modifiche alla configurazione
    try {
      Config.salvaPropUtente();
    }
    catch (java.security.AccessControlException errApp) { }  // nulla, posso essere in un'applet!
    catch (Exception E) { OiDialogoi.mostraErrore (E); }

    // Si rimuove dai listener di lingua
    Lingua.removeLinguaListener (this);

    // Si rimuove dai listener di apparenza
    UIManager.removePropertyChangeListener (this);

    // Rimuove fisicamente la finestra
    mioFrame.dispose();

    numFinAperte--;                                      // Sto chiudendo una finestra:
    if ((numFinAperte == 0) &&                           // Se non ce ne sono rimaste pi, termino!
        (Config.chePropBool("ContinuaDopoUltima") != true)) { // (ma solo se  stato stabilito ci tramite "opzioni")
          // Nota: tengo il "!= true" perch la cheProp potrebbe rendere "null" se non ho mai fatto
          // la "defProp" - e in genere gi accade nei "main" (delle classi) usati a scopo di debug!
          System.out.println (Lingua.frase("F_EXITING_SOMA"));
          try {
            System.exit (0);                    // USCITA BRUTALE DAL SISTEMA!          	
          }
          catch (Exception e) { }    // Ho eccezione se sono in un APPLET!
    }
  } //distruggiFinestra

      /**
       *  Disabilita la finestra
       */
  public void disabilitaFinestra () {
    mioFrame.setEnabled (false);
  } //disabilitaFinestra

      /**
       *  Abilita la finestra (precedentemente disabilitata)
       */
  public void abilitaFinestra () {
    mioFrame.setEnabled (true);
  } //abilitaFinestra

      /**
       *  Mette in primo piano la finestra
       */
  public void inPrimoPiano () {
    mioFrame.setState(Frame.NORMAL);
    mioFrame.toFront();
  } //inPrimoPiano

      /**
       *  Mette in primo piano la finestra
       */
  public void RiduciAIcona () {
    mioFrame.setState(Frame.ICONIFIED);
  } //RiduciAIcona

      /**
       *  ASTRATTA! 
       *  Deve creare una barra di menu' e aggiungerla a "mioFrame".
       *  Per una barra dei menu' di default si pu invocare la "creaBarraMenuDefault"
       */
  protected abstract void creaBarraMenu ();

      /**
       *  Crea una barra di menu' di default e l'aggiunge al "mioFrame"
       */
  protected void creaBarraMenuDefault (int ExitClose) {
    JMenuItem menuItem;
    JMenu     menuVoce;
    JMenuBar  menuBarra = new JMenuBar();

    String mnemonici = Lingua.frase("F_MENU_MNEMONICS") + "       ";  // = "FECHA"

    menuVoce = new JMenu (Lingua.frase("F_MENU_FILE"));   // men "file"
    menuVoce.setMnemonic (mnemonici.charAt(0));
    menuBarra.add (menuVoce);

      menuItem = new JMenuItem (Lingua.frase("F_MENU_F_OPTION"));
      menuItem.setIcon (IMG_ICO_OPZIO);
      menuItem.addActionListener (new ActionListener() {
                                    public void actionPerformed (ActionEvent e) {
                                      FinOpzioni fin = new FinOpzioni();
                                    }
                                  });
      menuVoce.add (menuItem);

      if (ExitClose == MENU_EXIT) {
        menuItem = new JMenuItem (Lingua.frase("F_MENU_F_EXIT"));
        menuItem.setIcon (IMG_ICO_EXIT);
        menuItem.setMnemonic (mnemonici.charAt(1));
        menuItem.setAccelerator (KeyStroke.getKeyStroke(KeyEvent.VK_X, ActionEvent.CTRL_MASK));  //imposta "Ctrl-X"
        menuItem.addActionListener (new ListExit(mioFrame, this, ListExit.EXIT_SOMA));
      }
      else {
        menuItem = new JMenuItem (Lingua.frase("F_MENU_F_CLOSE"));
        menuItem.setIcon (IMG_ICO_CLOSE);
        menuItem.setMnemonic (mnemonici.charAt(2));
        menuItem.addActionListener (new ActionListener() {
                                          public void actionPerformed(ActionEvent e) {
                                            distruggiFinestra();       // Distruggo questa finestra!
                                          }
                                    });
      } //else
      menuVoce.add (menuItem);

    menuVoce = LinguaGui.menuSceltaLingua(mioFrame);      // men "lingua"
    menuBarra.add (menuVoce);

    menuVoce = Apparenza.menuSceltaApparenza(mioFrame);   // men "apparenza"
    menuBarra.add (menuVoce);

    menuBarra.add (Box.createHorizontalGlue());           // aggiunge della colla

    menuVoce = new JMenu (Lingua.frase("F_MENU_HELP"));   // men "HELP"
    menuVoce.setMnemonic (mnemonici.charAt(3));
    menuBarra.add (menuVoce);

      menuItem = new JMenuItem (Lingua.frase("F_MENU_H_ABOUT"));
      menuItem.setIcon (IMG_ICO_HELP);
      menuItem.setMnemonic (mnemonici.charAt(4));
      menuItem.addActionListener ( new ActionListener() {
                                     public void actionPerformed(ActionEvent e) {
                                       FinAbout fin = new FinAbout(mioFrame);
                                   }} );
      menuVoce.add (menuItem);

    // Definisco la barra (ATTENZIONE: al JFrame!)
    mioFrame.setJMenuBar (menuBarra);
    
  } //creaBarraMenuDefault

      /**
       *  ASTRATTA! 
       *  Deve definire/modificare tutte le frasi mostrate a video (nella GUI).
       *  Dev'essere invocata dalla "costruisciFinestra" (per assegnare una frase
       *  agli oggetti GIA' CREATI che tengono una stringa).
       *  In essa si fanno invocazioni del tipo:
       *      <code>  OGGETTO.setText (Lingua.frase("IDENTIFICATORE"));  </code>
       *  Al termine si deve invocare la "FineImpostaFrasi" per apportare le modifiche.
       */  
  protected abstract void impostaFrasi ();

      /**
       *  Parte finale della "impostaFrasi" (dev'essere invocata al termine di essa)
       *  Ricrea la barra del men e invalida il JFrame (in modo che ne venga rifatto
       *  il layout da parte del gestore della grafica).
       */
  protected void FineImpostaFrasi () {
    creaBarraMenu ();           // Ricrea la barra del men

    mioFrame.invalidate();      // RIVALIDAZIONE DELLA FINESTRA
    mioFrame.pack();
  } //FineImpostaFrasi

      /**
       *  INTERNA!
       *  Aggiorna il "Look & Feel" del JFrame "mioFrame": ritraccia l'UI di 
       *  tutti i suoi componenti e invalida il JFrame (in modo che ne venga 
       *  rifatto il layout da parte del gestore della grafica).
       *  Viene chiamata al termine di "propertyChange".
       */
  protected void AggiornaApparenzaFinestra () {
    SwingUtilities.updateComponentTreeUI (mioFrame);  // RIDEFINIZIONE UI
    mioFrame.invalidate();                            // RIVALIDAZIONE DELLA FINESTRA
    mioFrame.pack();
  } //AggiornaApparenzaFinestra

      /**
       *  METODO RICHIESTO DALL'INTERFACCIA "LinguaListener".
       *  Viene invocato automaticamente quando qualcuno invoca la "Lingua.defLingua()"
       *  Ovviamente chiama "impostaFrasi()".
       */
  public void linguaCambiata () {
    Debug.outln(nomeFinestra+" - ho ascoltato un cambio di lingua");

    impostaFrasi ();

    // Aggiorna la propriet di configurazione
    Config.modifPropInt ("Lingua", Lingua.cheLinguaAttuale());

  } //linguaCambiata

      /**
       *  METODO RICHIESTO DALL'INTERFACCIA "PropertyChangeListener".
       *  Viene invocato automaticamente quando qualcuno imposta un nuovo
       *  "Look & Feel" tramite un'invocazione a "UIManager.setLookAndFeel()"
       *  Chiamer la "AggiornaApparenzaFinestra()".
       */  
  public void propertyChange (PropertyChangeEvent e) {
    String lookName = UIManager.getLookAndFeel().getClass().getName();
    Config.modifProp ("Apparenza", lookName);
    AggiornaApparenzaFinestra ();
  } //propertyChange

} //Finestra