/**  Config
 *     Classe "statica" usata per mantenere la configurazione di un programma
 *     attraverso la definizione di "identicatori" associati a ogni propriet
 *     della configurazione.
 *     Ricorda che ogni propriet pu essere: definita, modificata, letta!
 *     Con "DEFINITA" intendo il valore che il programmatore stabilisce per
 *     una certa propriet e che sar usato in mancanza di una ridefinizione
 *     dall'utente (che pu sempre essere fatta e dovr poi essere mantenuta).
 *     Quindi con "MODIFICATA" intendo il valore che l'utente vuole per una certa
 *     propriet (e che "scavalcher" il valore predefinito dal programmatore).
 *     Con "LETTA" si intende l'operazione di lettura del valore della propriet.
 *     Ogni classe deve innanzitutto definire le proprie propriet con la "defProp";
 *     se l'utente impone una modifica occorre invece usare la "modifProp"!
 *     Su disco verranno salvate solo le modifiche apportate dall'utente.
 *     Ovviamente, al termine del programma occorre ricordarsi di salvarle queste
 *     benedette modifiche.
 *     @author     Luigi Antenucci
 *     @version    5.5
 *     @language   jdk 1.2
 */

package SOMA.gui;

import java.util.Properties;
import java.util.Enumeration;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Rectangle;



public class Config {

  // COSTANTI (protette):
      /**
       *  File su cui memorizzare le modifiche di utente.
       */
  protected static final String FILE_CONFIG = "CONFIG.GG";

  // Variabili di classe (protette):
      /**
       *  Qui ci memorizzo tutte le "propriet" MODIFICATE da utente!
       */
  protected static Properties propUtente  = new Properties();

      /**
       *  Qui invece tutte le "propriet" DEFINITE dal programmatore!
       */
  protected static Properties propDefault = new Properties();

      /**
       *  Devo sapere se  la prima volta che eseguo un qualsiasi metodo di questa 
       *  classe, perch devo (ovviamente) caricare dal disco tutte le "propriet" 
       *  modificate da utente (nella precedente esecuzione del programma).
       *  Lo faccio in automatico, senza che il programmatore se ne debba interessare.
       */
  protected static boolean primaVolta  = true;
  // Questo mi dice se le tabelle sono ancora da riempire

      /**
       *  Questa variabile mi permetter di ricordarmi (al momento del salvataggio)
       *  se l'utente ha fatto una MODIFICA delle propriet.
       *  Ho voluto "ottimizzare": se viene richiesto un salvataggio delle propriet
       *  (modificate), se non  stata fatta nessuna modifica dall'ultimo salvataggio
       *   inutile andarle a salvare nuovamente!
       */
  protected static boolean modificate = false;
  // L'utente ha modificato una propriet?

      /**
       *  Carico tutte la propriet che sono state modificate dall'utente.
       *  Viene usato il file specifico definito in "FILE_CONFIG".
       */
  public static void caricaPropUtente () throws Exception {
    caricaPropUtente (FILE_CONFIG);
  } //caricaPropUtente

      /**
       *  Come la precedente, ma NON rende l'eccezione se non c' il file.
       */
  public static void caricaPropUtenteSeCiSono () {
    try {
      caricaPropUtente(FILE_CONFIG);
    }
    catch (Exception E) { }
  } //caricaPropUtenteSeCiSono

      /**
       *  Carico tutte la propriet che sono state modificate dall'utente.
       *  Non si usa il file definito in "FILE_CONFIG" ma il file il cui nome
       *   passato nella stringa nomeFile.
       *  In tal modo si possono "personalizzare" il file di salvataggio.
       */
  public static void caricaPropUtente (String nomeFile) throws Exception {
    if (modificate || primaVolta) {
      Debug.outln("Config - Caricando la configurazione da disco..");
      primaVolta = false;  // Non e' piu' la prima volta!
      propUtente.load (new FileInputStream(nomeFile));  // CARICO IL FILE UTENTE (se c'e')
      // L'operazione di "load" puo' generare l'eccezione.
      modificate = false;   // Ora non c' pi nessuna modifica rispetto alla config. salvata
    }
  } //caricaPropUtente

      /**
       *  Salva su disco tutte le modifiche fatte dall'utente alle propriet.
       *  Viene usato il file specifico definito in "FILE_CONFIG".
       *  NB: il salvataggio avviene solo se non  stata fatta nessuna modifica 
       *  delle propriet relativamente all'ultimo salvataggio.
       *  Infatti sarebbe inutile andare a salvare una configurazione che 
       *  perfettamente identica a quella che  gi presente su disco!
       */
  public static void salvaPropUtente () throws Exception {
    salvaPropUtente (FILE_CONFIG);
  } //salvaPropUtente

      /**
       *  Salva su disco tutte le modifiche fatte dall'utente alle propriet.
       *  Non si usa il file definito in "FILE_CONFIG" ma il file il cui nome
       *   passato nella stringa nomeFile.
       *  In tal modo si possono "personalizzare" il file di salvataggio.
       *  NB: il salvataggio avviene solo se non  stata fatta nessuna modifica 
       *  delle propriet relativamente all'ultimo salvataggio.
       */
  public static void salvaPropUtente (String nomeFile) throws Exception {
    if (modificate) {
      // SALVO solo se c' stata una modifica dall'ultimo salvataggio!
      Debug.outln("Config - Salvando la configurazione su disco..");
      propUtente.store (new FileOutputStream(nomeFile), "");  // SALVO IL FILE UTENTE (se c'e')
      modificate = false;   // Ora non c' pi nessuna modifica rispetto alla config. salvata
    }
  } //salvaPropUtente

      /**
       *  DEFINIZIONE di una propriet - si impone l'assegnazione di un valore di DEFAULT.
       *  Deve essere fatta dal programmatore all'interno del programma.
       *  La propriet  espressa dall'identificatore "chiave" a cui viene
       *  assegnato il valore "valore".
       *  "chiave" e "valore" sono entrambi delle stringhe!
       *  Se la chiave non  gi in UTENTE, la si assegna anche a esso col valore di default.
       *  Attenzione: non  ammessa la definizione di una propriet pi di una
       *  volta nel corso dell'esecuzione del programma, a meno che non venga 
       *  definita sempre con lo stesso valore.
       */
  public static void defProp (String chiave, String valore) {
    if (primaVolta)                // LA PRIMA VOLTA CARICA IL FILE EVENTUALMENTE SALVATO
      caricaPropUtenteSeCiSono();  // (CONTENENTE LE PROPRIETA' DEFINITE DA UTENTE), SE C'E`.

    // Salva nel DEFAULT (in ogni caso)
    Object Occupato = propDefault.setProperty (chiave, valore);
    if (Occupato != null) 
      if (! valore.equals( (String)Occupato )) {      // Avviso solo se hanno una definizione differente!
        System.out.println ("Config: the Key '"+chiave+"' Already Exists!");
        System.out.println ("        Please, change its name!");
      }

    if (propUtente.getProperty(chiave) == null)   // Se NON c'e' gia' nelle propr.UTENTE
      propUtente.setProperty (chiave, valore);    // lo salva anche li'
  } //defProp


      /**
       *  Come la "defProp", ma non controlla se la propriet  gi stata definita (con diverso valore).
       *  Utile per propriet che possono essere ridefinite pi volte e ogni volta con valore diverso,
       *  per es. la dimensione di una finestra - che dipende dai componenti all'interno.
       */
  public static void defPropVariabile (String chiave, String valore) {
    if (primaVolta)                // LA PRIMA VOLTA CARICA IL FILE EVENTUALMENTE SALVATO
      caricaPropUtenteSeCiSono();  // (CONTENENTE LE PROPRIETA' DEFINITE DA UTENTE), SE C'E`.

    // Salva nel DEFAULT (in ogni caso)
    Object Occupato = propDefault.setProperty (chiave, valore);

    if (propUtente.getProperty(chiave) == null)   // Se NON c'e' gia' nelle propr.UTENTE
      propUtente.setProperty (chiave, valore);    // lo salva anche li'
  } //defPropVariabile

      /**
       *  MODIFICA di una propriet - si fa una modifica del valore di DEFAULT.
       *  Deve essere fatta dall'utente (o da azioni dell'utente che si ripercuotono
       *  sulle propriet di configurazione).
       * NON va a modificare il default, ma sar salvata su disco!
       *  La propriet  espressa dall'identificatore "chiave" a cui viene
       *  modificato il valore esistente con il nuovo valore "valore".
       *  "chiave" e "valore" sono entrambi delle stringhe!
       *  Attenzione: la modifica di una propriet pu essere fatta quante volte si
       *  vuole, anche nessuna!
       */
  public static void modifProp (String chiave, String valore) {
    if (primaVolta)                // LA PRIMA VOLTA CARICA IL FILE EVENTUALMENTE SALVATO
      caricaPropUtenteSeCiSono();  // (CONTENENTE LE PROPRIETA' DEFINITE DA UTENTE), SE C'E`.

    // nessuna modifica a propDefault!

    // lo salva in prop.utente IN OGNI CASO (sovrascrive il vecchio valore)
    Object Occupato = propUtente.setProperty (chiave, valore);

    // L'utente ha modificato almeno una propriet?
    // Ho modificato una propriet solo se ne ho effettivamente modificato il valore!
    // E (ovviamente) erano gi state precedentemente modificate (e non ancora salvate)
    if (Occupato != null)
        modificate = modificate || (! valore.equals( (String)Occupato ));
    if (Occupato == null)
      System.out.println ("Config: the Key '"+chiave+"' Hasn't Ever Been Defined!");
  } //modifProp

      /**
       *  Rende il valore della propriet "chiave".
       *  Tale valore sar quello MODIFICATO dall'utente.
       *  Se l'utente non ha fatto alcuna modifica della propriet, verr reso
       *  il valore DEFINITO dal programmatore.
       *  "chiave" e "valore" sono entrambi delle stringhe!
       */
  public static String cheProp (String chiave) {
    if (primaVolta)                // LA PRIMA VOLTA CARICA IL FILE EVENTUALMENTE SALVATO
      caricaPropUtenteSeCiSono();  // (CONTENENTE LE PROPRIETA' DEFINITE DA UTENTE), SE C'E`.

    String chevalore = propUtente.getProperty (chiave);
    if (chevalore != null)
      return chevalore;                          // PRIMA guarda in prop. di UTENTE, 
    else                                         // Se non la trova, guarda nel DEFAULT
      return propDefault.getProperty (chiave);
  } //cheProp

      /**
       *  Rende il valore di default della propriet "chiave".
       *  Il valore reso  il valore predefinito dal programmatore, anche se l'utente
       *  l'ha "modificato".
       *  "chiave" e "valore" sono entrambi delle stringhe!
       */
  public static String chePropDefault (String chiave) {
    return propDefault.getProperty (chiave);
  } //chePropDefault

      /**
       *  Ripristina al valore di default la propriet passata.
       *  Toglie l'eventuale modifica fatta da utente.
       *  "chiave" e "valore" sono entrambi delle stringhe!
       */
  public static void resetProp (String chiave) {
    modifProp(chiave, chePropDefault(chiave));
  } //resetProp

      /**
       *  Ripristina tutte le propriet al proprio valore di default!
       *  Toglie tutte le eventuali modifiche fatta da utente.
       */
  public static void resetConfig () {
    String chiave;
    Enumeration enum = propDefault.propertyNames();  // Enumero le propriet
    while (enum.hasMoreElements()) {
      chiave = (String) enum.nextElement();
      modifProp(chiave, chePropDefault(chiave));
    }
    File F = new File (FILE_CONFIG);
    F.delete ();
  } //resetConfig

  // VERSIONI SPECIALIZZATE PER ALCUNI TIPI DI DATO (MOLTO USATI)

      /**
       *  Come "defProp" ma specifica per valori di tipo intero.
       */
  public static void defPropInt (String chiave, int valore) {
    defProp (chiave, String.valueOf(valore));
  }
      /**
       *  Come "modifProp" ma specifica per valori di tipo intero.
       */
  public static void modifPropInt (String chiave, int valore) {
    modifProp (chiave, String.valueOf(valore));
  }
      /**
       *  Come "cheProp" ma specifica per valori di tipo intero.
       */
  public static int chePropInt (String chiave) {
    return Integer.valueOf( cheProp(chiave) ).intValue();
  }

      /**
       *  Come "defProp" ma specifica per valori di tipo boolean.
       */
  public static void defPropBool (String chiave, boolean valore) {
    defProp (chiave, String.valueOf(valore));
  }
      /**
       *  Come "modifProp" ma specifica per valori di tipo boolean.
       */
  public static void modifPropBool (String chiave, boolean valore) {
    modifProp (chiave, String.valueOf(valore));
  }
      /**
       *  Come "cheProp" ma specifica per valori di tipo boolean.
       */
  public static boolean chePropBool (String chiave) {
    return Boolean.valueOf( cheProp(chiave) ).booleanValue();
  }
      
      /**
       *  Come "defProp" ma specifica per valori di classe Color.
       */
  public static void defPropCol (String chiave, Color valore) {
    defProp (chiave, String.valueOf(valore.getRGB()));   // prendo RGB (di tipo int)
  }
      /**
       *  Come "modifProp" ma specifica per valori di classe Color.
       */
  public static void modifPropCol (String chiave, Color valore) {
    modifProp (chiave, String.valueOf(valore.getRGB()));
  }
      /**
       *  Come "cheProp" ma specifica per valori di classe Color.
       */
  public static Color chePropCol (String chiave) {
    return Color.decode( cheProp(chiave) );
  }

      /**
       *  Come "defProp" ma specifica per valori di classe Dimension.
       */
  public static void defPropDim (String chiave, Dimension valore) {
    defPropInt (chiave+"DimX", valore.width);
    defPropInt (chiave+"DimY", valore.height);
  }
      /**
       *  Come "modifProp" ma specifica per valori di classe Color.
       */
  public static void modifPropDim (String chiave, Dimension valore) {
    modifPropInt (chiave+"DimX", valore.width);
    modifPropInt (chiave+"DimY", valore.height);
  }
      /**
       *  Come "cheProp" ma specifica per valori di classe Color.
       */
  public static Dimension chePropDim (String chiave) {
    return new Dimension (chePropInt(chiave+"DimX"),
                          chePropInt(chiave+"DimY"));
  }

      /**
       *  Come "defProp" ma specifica per memorizzare "locazione" e "dimensione" di una finestra
       *  (passata come un generico componente - su cui si pu fare la "getBounds()").
       *  La chiave viene definita da utente (possono esserci pi di una finestra) e i "valori" 
       *  sono all'interno della finestra passata (posizione X-Y e dimensione W-H).
       */
  public static void defPropComponent (String chiave, Component componente) {
    Rectangle valori = componente.getBounds();
    defPropInt (chiave+"-PosX", valori.x);
    defPropInt (chiave+"-PosY", valori.y);
    defPropVariabile (chiave+"-DimX", String.valueOf(valori.width));  // NON E' UN VALORE FISSO OGNI VOLTA!
    defPropVariabile (chiave+"-DimY", String.valueOf(valori.height));
  }
      /**
       *  Come "modifProp" ma specifica per un "componente".
       */
  public static void modifPropComponent (String chiave, Component componente) {
    Rectangle valori = componente.getBounds();
    modifPropInt (chiave+"-PosX", valori.x);
    modifPropInt (chiave+"-PosY", valori.y);
    modifPropInt (chiave+"-DimX", valori.width);
    modifPropInt (chiave+"-DimY", valori.height);
  }
      /**
       *  Come "cheProp" ma specifica per un "componente".
       *  Rende il rettangolo contenente "posizione" e "dimensione"
       *  Serve lo stesso la chiave!
       */
  public static Rectangle chePropComponent (String chiave, Component componente) {
    return new Rectangle (chePropInt(chiave+"-PosX"),
                          chePropInt(chiave+"-PosY"),
                          chePropInt(chiave+"-DimX"),
                          chePropInt(chiave+"-DimY"));
  }

      /**
       *  Rende la rappresentazione sottoforma di stringa di TUTTO l'elenco di
       *  propriet definite dal programmatore e modificate da utente.
       *  NB: NON posso chiamarla "toString" perch questa  STATICA!
       */
  public static String inStringa () {
    return "Default="+propDefault+" \n Utente="+propUtente;
  } //inStringa

} //Config