/**  ActionPlace
 *     Esecutore delle azioni invocabili da una finestra di place.
 *     Viene creato dall'oggetto {@link SOMA.Environment} al momento della
 *     costruzione dell'ambiente del place.
 *     Contiene un riferimento alla sua {@link SOMA.gui.FinestraPlace}, che non
 *      necessariamente mostrata a video.
 *     E' fondamentalmente un oggetto {@link SOMA.network.connection.Daemon}.
 *     Sono stato "obbligato" a fare ci per mantenere una certa compatibilit
 *     con la precedente versione della finestra place, in modo da non amdare a
 *     modificare troppo il men a linea di comando in SOMA (gi disponibile).
 *     Implementando l'interfaccia "Daemon" posso RIUSARE il "DaemonExplorerItem"
 *     per ridirezionare la voce "window" del men sul mio "ActionPlace"!
 *     Racchiude il codice di tutti i comandi eseguibili da un place.
 *     @author     Luigi Antenucci
 *     @version    1.8
 *     @language   jdk 1.2.2
 */

package SOMA.gui;

import SOMA.Environment;
import SOMA.naming.PlaceID;
import SOMA.naming.AgentID;
import SOMA.agent.AgentWorker;
import SOMA.network.connection.Daemon;
import SOMA.agent.classLoading.ClassManager;
import SOMA.agent.mobility.AgentWorkerStore;
import SOMA.agent.mobility.AgentPositionStore;
import SOMA.mobilePlace.MobileEnvironment;
import SOMA.mobilePlace.MobileSecurityEnvironment;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Vector;
import java.util.Properties;
import java.net.InetAddress;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;


public class ActionPlace implements Daemon, ActionPlaceInterface {

  // VARIABILI PRIVATE:
      /**
       *  Riferimento all'{@link SOMA.Environment} del place a cui  associato questo "ActionPlace".
       */
  protected  Environment env;         // IO HO L' "ENV", NON LA MIA FINESTRA!

      /**
       *  Riferimento alla propria {@link SOMA.gui.FinestraPlace}
       */
  protected FinestraPlace finestraPlace = null;

      /**
       *  Stato della finestra: ON=esistente, OFF=non ancora creata.
       *  Non l'ho definito io, lo si eredita dall'interfaccia {@link SOMA.network.connection.Daemon}.
       */
  protected Object statoFin = OFF;


      /**
       *  Costruttore di un ActionPlace per l'{@link SOMA.Environment} env passato.
       */
  public ActionPlace (Environment env) {
    this (env, false);
  } //costruttore

      /**
       *  Costruttore di un ActionPlace per l'{@link SOMA.Environment} env passato.
       *  Se apriFin=true, verr automaticamente creata la {@link SOMA.gui.FinestraPlace} associata.
       */
  public ActionPlace (Environment env, boolean apriFin) {
    Debug.outln ("ActionPlace - in creazione per l'ENVIRONMENT "+env.toString());
    this.env = env;     // Salva il riferimento al proprio ambiente
    if (apriFin)
      try {
        start();
      }
      catch (Exception E) { System.out.println(E); }
  } //costruttore

      /**
       *  Rende l'{@link SOMA.Environment} del place associato all'ActionPlace
       */
  public Environment cheEnv () {
    return env;
  } //cheEnv

      /**
       *  Rende il {@link SOMA.naming.PlaceID} del place associato all'ActionPlace
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public PlaceID chePlaceID () {
    return env.placeID;
  } //chePlaceID

      /**
       *  Viene creata la finestra {@link SOMA.gui.FinestraPlace} del place
       *  associato a questo ActionPlace.
       *  Nota: se la finestra  gi stata creata viene portata in "primo piano".
       *  Metodo richiesto dall'interfaccia {@link SOMA.network.connection.Daemon}.
       */
  public void start () throws Exception {
    if (statoFin == OFF) {
      finestraPlace = new FinestraPlace (this);

      statoFin = ON;
    }
    else
      finestraPlace.inPrimoPiano();
  } //start

      /**
       *  Viene distrutta la finestra del place.
       *  Nota bene: NON viene distrutto questo "ActionPlace" ma solo la sua finestra;
       *  quindi si pu successivamente richiamare il metodo "start()" per riaprire
       *  la finestra.
       *  Metodo richiesto dall'interfaccia {@link SOMA.network.connection.Daemon}.
       */
  public void stop () throws Exception {
    if (statoFin == ON) {
      finestraPlace.distruggiFinestra ();
      statoFin = OFF;
    }
  } //stop

      /**
       *  Rende lo stato della finestra (ON o OFF).
       *  Lo stato  un oggetto, come richiesto in {@link SOMA.network.connection.Daemon}
       *  e lo stato pu essere {@link SOMA.network.connection.Daemon#ON} o {@link SOMA.network.connection.Daemon#OFF}
       *  Metodo richiesto dall'interfaccia {@link SOMA.network.connection.Daemon}.
       */
  public Object getStatus () {
    return statoFin;
  } //getStatus

      /**
       *  Mostra l'oggetto in una stringa.
       */
  public String toString() {
   return "["+ActionPlace.class.toString()+", status=" + statoFin + ", env="+env.toString()+"]";
  } //toString

      /**
       *  "Connessione" per un Place Mobile.
       */
  public void mobileConnect (PlaceID placeID) {
    if  (env instanceof MobileEnvironment) {
       ((MobileEnvironment)env).connect(placeID); }
      else
         if (env instanceof MobileSecurityEnvironment) {
           ((MobileSecurityEnvironment)env).connect(placeID); }

  } //mobileConnect

      /**
       *  "Disconnessione" per un Place Mobile.
       */
  public void mobileDisconnect () {
     if  (env instanceof MobileEnvironment) {
          ((MobileEnvironment)env).disconnect();  }
      else
         if (env instanceof MobileSecurityEnvironment) {
          ((MobileSecurityEnvironment)env).disconnect();  }

  } //mobileDisconnect

      /**
       *  Rende il PlaceID (di dominio) a cui  connesso al momento il Place Mobile
       *  Se non  connesso render "MobilePlaceManager.DISCONNECTED".
       */
  public PlaceID mobileCurrentDomainID () {
    if  (env instanceof MobileEnvironment) {
    return ((MobileEnvironment)env).currentDomainID; }
      else
         if (env instanceof MobileSecurityEnvironment) {
             return ((MobileSecurityEnvironment)env).currentDomainID; }
         else return null;
  } //mobileCurrentDomainID

      /**
       *  Rende l'indirizzo IP del computer locale.
       *  Se rende "null", il computer NON ha un indirizzo IP.
       */
  public InetAddress indirizzoLocale () {
    InetAddress indLocale;
    try {
      indLocale = InetAddress.getLocalHost();
    }
    catch (java.net.UnknownHostException E) {
      indLocale = null;
    }
    return indLocale;
  } //indirizzoLocale

      /**
       *  Rende la quantit di memoria totale.
       */
  public long memTotale () {
    return Runtime.getRuntime().totalMemory();
  } //memTotale

      /**
       *  Rende la quantit di memoria libera.
       */
  public long memLibera () {
    return Runtime.getRuntime().freeMemory();
  } //memLibera

      /**
       *  Rende la quantit di memoria occupata (pari al totale meno la libera).
       */
  public long memUsata () {
    return memTotale() - memLibera();
  } //memUsata

      /**
       *  Rende un modello di un albero ({@link javax.swing.tree.DefaultTreeModel})
       *  il cui albero  costituito da nodi di classe {@link javax.swing.tree.DefaultMutableTreeNode}.
       *  Tale albero contiene in ogni nodo una STRINGA (e non un oggetto Thread!) che rappresenta la
       *  DESCRIZIONE del relativo thread nel threadgroup.
       */
  public DefaultTreeModel alberoAllThread () {

    ThreadGroup threadGroup = Thread.currentThread().getThreadGroup();  //Prendo il gruppo del mio thread!
    return alberoThreadGroup (threadGroup);

  } //alberoAllThread

      /**
       *  Rende un modello di un albero ({@link javax.swing.tree.DefaultTreeModel})
       *  il cui albero  costituito da nodi di classe {@link javax.swing.tree.DefaultMutableTreeNode}.
       *  Tale albero contiene tutti i thread attivi all'interno del Place di riferimento.
       *  Nota: ogni nodo contiene una STRINGA (e non un oggetto Thread!) che rappresenta la
       *  DESCRIZIONE del reltivo thread nel threadgroup.
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public DefaultTreeModel alberoPlaceThread () {

    ThreadGroup threadGroup = env.threadGroup;      //Prendo il gruppo definito nell'ENVIRONMENT!
    return alberoThreadGroup (threadGroup);

  } //alberoPlaceThread

      /**
       *  Rende un modello di un albero ({@link javax.swing.tree.DefaultTreeModel})
       *  il cui albero  costituito da nodi di classe {@link javax.swing.tree.DefaultMutableTreeNode}
       *  Tale albero contiene tutti i thread del gruppo relativo al thread-group passato.
       *  Nota: ogni nodo contiene una STRINGA (e non un oggetto Thread!) che rappresenta la
       *  DESCRIZIONE del reltivo thread nel threadgroup.
       */
  public DefaultTreeModel alberoThreadGroup (ThreadGroup threadGroup) {

    DefaultMutableTreeNode radice = alberoThreadGroupRic (threadGroup);

    DefaultTreeModel modello = new DefaultTreeModel(radice);
    return modello;

  } //alberoThreadGroup

      /**
       *  Metodo interno, usato per creare "ricorsivamente" l'albero dei thread-group.
       *  Preleva tutti i thread del Thread-Group passato e crea per ognuno un nuovo nodo che
       *  viene aggiunto al Nodo del threadgruop (che sar il nodo che viene reso).
       *  Per ognuno dei nuovi nodi esegue la chiamata ricorsiva!
       *  Nota: ogni nodo contiene una STRINGA (e non un oggetto Thread!) che rappresenta la
       *  DESCRIZIONE del reltivo thread nel threadgroup.
       */
  protected DefaultMutableTreeNode alberoThreadGroupRic (ThreadGroup threadGroup) {

    // CREO IL NODO RELATIVO AL GRUPPO DI "threadGroup".
    DefaultMutableTreeNode nodoGruppo = new DefaultMutableTreeNode (threadGroup.toString());

    DefaultMutableTreeNode nodoNuovo;
    // PRELEVO I THREAD ATTIVI DEL GRUPPO "threadGroup" passato.
    int count = threadGroup.activeCount();
    Thread[] arrThread = new Thread[count];
    count = threadGroup.enumerate (arrThread, false);   // Cos ho tutti i thread del gruppo

    String Testo;
    ClassLoader classLoader;
    for (int i=0; i<count; i++) {
      classLoader = arrThread[i].getContextClassLoader();
      Testo = arrThread[i].toString()+" class-loader:";
      if (classLoader == null)
        Testo = Testo + "null";
      else
        Testo = Testo + classLoader.toString();
      nodoNuovo = new DefaultMutableTreeNode (Testo);
      nodoGruppo.add (nodoNuovo);
    }

    // PRELEVO I SOTTO-GRUPPI DEL GRUPPO "threadGroup" passato.
    int countGroups = threadGroup.activeGroupCount();
    ThreadGroup[] arrThreadGroup = new ThreadGroup[countGroups];
    countGroups = threadGroup.enumerate (arrThreadGroup, false);  // Cos ho tutti i gruppi del gruppo

    for (int i=0; i<countGroups; i++) {

      DefaultMutableTreeNode sottoGruppo = alberoThreadGroupRic (arrThreadGroup[i]);  // RICORSIONE!
      nodoGruppo.add (sottoGruppo);
    }

    return nodoGruppo;     // Rende il nodo contenente il gruppo intero
  } //alberoThreadGroupRic

      /**
       *  Rende le propriet di sistema definite.
       */
  public Properties cheProprieta () {
    return System.getProperties();
  } //cheProprieta

      /**
       *  Chiamata al Garbage-Collector.
       */
  public void chiamaGarbageCollector () {
    System.gc();
  } //chiamaGarbageCollector

      /**
       *  Rende il Percorso degli Agenti del place a cui  associato questo ActionPlace
       *  (ritrovandolo dalla catena di oggetti che partono dall'Environment).
       */
  public String cheAgentPath () {
    return env.agentManager.agentClassManager.getClassPathDirectory();
  } //cheAgentPath

      /**
       *  Imposta il Percorso degli Agenti al percorso contenuto nella stringa passata.
       */
  public void defAgentPath (String newPath) throws Exception {
    env.agentManager.agentClassManager = new ClassManager (newPath, true);
  } //defAgentPath

      /**
       *  Rende il Percorso di Cache degli Agenti del place a cui  associato questo ActionPlace
       *  (ritrovandolo dalla catena di oggetti che partono dall'Environment).
       */
  public String cheCachePath () {
    return env.agentManager.cacheClassManager.getClassPathDirectory();
  } //cheCachePath

      /**
       *  Imposta il Percorso di Cache degli Agenti al percorso contenuto nella stringa passata.
       *  Dato un {@link SOMA.naming.PlaceID} ne imposta il suo Percorso di Cache degli Agenti
       */
  public void defCachePath (String newPath) throws Exception {
    env.agentManager.cacheClassManager = new ClassManager (newPath, true);
  } //defCachePath

      /**
       *  Cancella tutto il contenuto della "cache-agent".
       *  Ricorda: quando si invia un agente su un Place esso viene per prima cosa cercato nella
       *  cache e, se trovato l, viene usata quella copia.
       *  E viene usata anche nel caso che quella sia una "versione precedente" (e ci causa un errore
       *  di "serializzazione" dell'agente.
       */
  public void cancellaCache () {
    String dirCacheStr = cheCachePath();
    File dirCache = new File (dirCacheStr);
    File[] elencoFile = dirCache.listFiles ();  // Prende elenco file+dir
    for (int i=0; i<elencoFile.length; i++) {
      if (elencoFile[i].isFile())
        elencoFile[i].delete();
    }
  } //cancellaCache

      /**
       *  Lancia un agente, in base ai dati passati.
       *  Il Place in cui viene lanciato  quello associato a questo ActionPlace,
       *  Non fa altro che invocare l'oggetto "statico" {@link SOMA.gui.Creatore}.
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public void lanciaAgente (String nomeAgente, String[] arrParam,
                            boolean usaSysClassLoader, boolean rintracciabile,
                            boolean nonFarloPartire) throws Exception {

    Creatore.lanciaAgente (env,                       // L'ambiente del place associato.
                           nomeAgente, arrParam,
                           usaSysClassLoader, rintracciabile, nonFarloPartire,
                           null, null);
  } //lanciaAgente

      /**
       *  Lancia un agente, in base ai dati passati.
       *  In pi, ci sono i parametri di sicurezza (profilo e password)
       */
  public void lanciaAgente (String nomeAgente, String[] arrParam,
                            boolean usaSysClassLoader, boolean rintracciabile,
                            boolean nonFarloPartire,
                            String profileFileName,
                            String profilePassword) throws Exception {

    Creatore.lanciaAgente (env,                       // L'ambiente del place associato.
                           nomeAgente, arrParam,
                           usaSysClassLoader, rintracciabile, nonFarloPartire,
                           profileFileName, profilePassword);
  } //lanciaAgente

      /**
       *  Rende la lista di tutti i nomi (sottoforma di stringa) degli agenti contenuti
       *  nel Percorso degli Agenti del place relativo all'ActionPlace.
       *  (usa la cheAgentPath e preleva tutti i nomi di file "giusti").
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector cheElencoAgenti () {
    String percAgenti = cheAgentPath ();
    File dirAgenti = new File (percAgenti);
    File[] elencoAgenti = dirAgenti.listFiles (
       new FilenameFilter () {
         public boolean accept (File dir, String name) {
           if (name.endsWith (".java")) {
             String nomeClass = name.substring (0, (name.length()-5));  // tolgo estensione ".java"
             File tmpF = new File (dir, nomeClass+".class");
             // Se il file  un file JAVA, puo' andare bene;
             // a patto che ci sia il suo rispettivo file .CLASS!
             return tmpF.exists();  // Ok se esiste il file di classe (compilato)
           }
           else
             return false;
         } //accept
       });
    Vector elenco = new Vector(elencoAgenti.length);
    for (int i=0; i<elencoAgenti.length; i++) {
      String nome = elencoAgenti[i].getName();
      nome = nome.substring (0, (nome.length()-5));   // tolgo estensione ".java" (5 caratteri)
      elenco.addElement (nome);
    }
    return elenco;
  } //cheElencoAgenti

      /**
       *  Rende la lista di tutti i PlaceID dei place contenuti nel DNS del place associato
       *  all'ActionPlace (il quale place deve essere un place di default per poter avere il DNS;
       *  in caso contrario la lista resa sar vuota).
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector elencoDominiiInDNS () {
    PlaceID[] arrPlace = env.domainNameService.getDomainsArray();
    Vector elenco = new Vector();
    for (int i=0; i<arrPlace.length; i++) {
      elenco.addElement (arrPlace[i]);
    }
    return elenco;
  } //elencoDominiiInDNS

      /**
       *  Rende la lista di tutti i PlaceID dei place contenuti nel PNS del place associato
       *  all'ActionPlace.
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector elencoPlaceInPNS () {
    PlaceID[] arrPlace = env.placeNameService.getPlacesArray();
    Vector elenco = new Vector();
    for (int i=0; i<arrPlace.length; i++) {
      elenco.addElement (arrPlace[i]);
    }
    return elenco;
  } //elencoPlaceInPNS

      /**
       *  Rende una lista contenente una descrizione di tutti gli agenti
       *  che sono al momento in esecuzione.
       *  Ogni elemento della lista  a sua volta una lista, cio una terna:
       *    (AgentID, NomeClasse, Stato)
       *  Utile per il costruttore di JTable().
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector elencoAgentiInEsecuzione () {
    Vector rowGruppo = new Vector();
    AgentWorkerStore mioAgentWorkerStore = env.agentManager.agentWorkerStore;
    AgentID[] arrAgentID = mioAgentWorkerStore.cheElencoAgenti();
    // Se la precedente linea d "errore" significa che manca il
    // metodo "cheElencoAgenti()" aggiunto da Luigi all'AgentWorkerStore

    AgentWorker agentWorker;
    Vector rowSingola;
    for (int row=0; row<arrAgentID.length; row++) {
      rowSingola = new Vector();

      agentWorker = mioAgentWorkerStore.getWorker (arrAgentID[row]);
      rowSingola.addElement (arrAgentID[row]);
      rowSingola.addElement (agentWorker.agent.getClass().getName());
      rowSingola.addElement (agentWorker.getStatus().toString());

      rowGruppo.addElement(rowSingola);
    }
    return rowGruppo;
  } //elencoAgentiInEsecuzione

      /**
       *  Rende una lista contenente la posizione di tutti gli agenti che sono
       *  stati creati nel Place.
       *  Ogni elemento della lista  a sua volta una lista, cio una coppia:
       *    (AgentID, PlaceID)
       *  Utile per il costruttore di JTable().
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector elencoPosizioneAgenti () {

    Vector rowGruppo = new Vector();
    AgentPositionStore mioAgentPositionStore = env.agentManager.agentPositionStore;
    AgentID[] arrAgentID = mioAgentPositionStore.cheElencoAgenti();
    // Se la precedente linea d "errore" significa che manca il
    // metodo "cheElencoAgenti()" aggiunto da Luigi all'AgentPositionStore

    Vector rowSingola;
    for (int row=0; row<arrAgentID.length; row++) {
      rowSingola = new Vector();

      rowSingola.addElement (arrAgentID[row]);
      rowSingola.addElement (mioAgentPositionStore.get(arrAgentID[row]));

      rowGruppo.addElement(rowSingola);
    }
    return rowGruppo;

  } //elencoPosizioneAgenti

} //ActionPlace