package SOMA.agent;

import SOMA.Environment;
import SOMA.naming.*;
import SOMA.agent.mobility.PlaceAccessPermission;
//import SOMA.utility.PrintProtectionDomain;
import java.security.*;
import java.io.*;
import SOMA.utility.IndexHashtable;

/** <P>Classe di interfaccia fra agenti e sistema.
* </P>
* <P> Si tratta dell'unico riferimento fornito ad un agente al momento
* dell'attivazione in un place. Quindi, per interagire con il mondo esterno
* l'agente dovra' necessariamente utilizzare riferimenti e metodi
* dell'<code>AgentSystem</code> che gli e' stato associato.
* </P>
* <P> L'AgentSystem, se necessario, effettuer i dovuti controlli di sicurezza
* prima di consentire ad un agente l'accesso ad una determinata risorsa.
* </P>
* <P> E' possibile costruire sottoclassi di <code>AgentSystem</code> in modo
* da fornire interfacce di tipo diverso ad agenti diversi per motivi di
* flessibilit, efficienza o compatibilit.
* </P>
* <BR>
* @see SOMA.agent.Agent
* @see SOMA.agent.AgentWorker
*
* @author Livio Profiri
*/

public class AgentSystem
{
  private Environment env;

  /** Contenitore per oggetti condivisi fra agenti.
  * <P> Ad ogni oggetto e' associato un indice di tipo int.
  */
  public IndexHashtable sharedObjects = new IndexHashtable();

  //public ObjectProxy proxy = new ObjectProxy;

  /** Costruttore. */
  public AgentSystem( Environment env )
  {
    this.env = env;
  }

  /** Metodo di migrazione.
  * <P> Effettua i controlli di sicurezza, quindi
  * invoca il metodo {@link SOMA.agent.AgentWorker#go( PlaceID destination )}.
  */
  protected void go( final Agent agent, final PlaceID destination ) throws CantGoException
  {
    try
    {
      //PrintProtectionDomain.Do( agent );

      if( System.getSecurityManager() != null )
        AccessController.checkPermission( new PlaceAccessPermission( destination ) );

      AccessController.doPrivileged(
        new PrivilegedExceptionAction()
        {
          public Object run() throws CantGoException
          {
            agent.worker.go( destination );

            return null;
          }
        });
    }
    catch( AccessControlException e )
    {
      throw new CantGoException( "Place Access Permission Denied." );
    }
    catch( PrivilegedActionException e )
    {
      throw (CantGoException)e.getException();
    }
	}

  protected void idle( Agent agent )
  {
    // Potrei aggiungere un controllo di sicurezza o lanciare una
    // throw( new UnsupportedOperationException( ... ) );

    agent.worker.idle();
  }

	public PlaceID getPlaceID()
	{
	  return env.placeID;
	}


	public InputStream getIn()
	{
	  return env.in;
	}

  public PrintStream getOut()
	{
	  return env.out;
	}

	public PrintStream getErr()
	{
	  return env.err;
	}

  public Environment getEnvironment()
	{
	  if( System.getSecurityManager() != null )
      AccessController.checkPermission( new AgentPermission( "Environment" ) );

	  return env;
	}

	/** Restituisce l'elenco degli identificatori dei place di questo dominio.*/
	public PlaceID[] getPlaces()
	{
	  return env.placeNameService.getPlacesArray();
	}

  /** Restituisce l'elenco degli identificatori dei domini, o
  * un array vuoto se non e' presente un Domain Name Service, perche'
  * non siamo in un default place.
  */
  public PlaceID[] getDomains()
	{
	  if( env.domainNameService != null )
	    return env.domainNameService.getDomainsArray();
	  else
	    return new PlaceID[0];
	}

  /** Spedizione di un messaggio
  *
  * <P> Non e' possibile statilire se il messaggio sara' correttamente recapitato.
  *
  * <P> Questo metodo richiama {@AgentWorker.sendMessage( Message )
  */
  public void sendMessage( final Message message )
  {
    AccessController.doPrivileged(
        new PrivilegedAction()
        {
          public Object run()
          {
            env.agentManager.sendMessage( message );
            return null;
          }
        });
  }

  /** Restituisce il numero di worker e quindi di agenti del place. */
  public int agentsNumber()
  {
    return env.agentManager.agentsNumber();
  }
}



  /*
  void kill(AgentID id)
  {
    Main.agentManager.dispose(id,"");
  }*/



