/** 
 * Classe che rappresenta l'astrazione di place.
 * <BR>Contiene i riferimenti a tutti gli oggetti che compongono un <I>place</I> o un <I>default place</I>,
 * che e' il place che gestisce un dominio.
 * <BR>Il costruttore di Environment avvia il place.
 *
 * @author Livio Profiri  (aggiunte di Luigi Antenucci)
 */

package SOMA;

// Ecco dove trovare le classi degli oggetti che compongono il sistema
import SOMA.network.NetworkManager;
import SOMA.explorer.DirExplorerItem;
import SOMA.naming.*;
import SOMA.naming.domain.*;
import SOMA.naming.place.*;
import SOMA.agent.mobility.AgentManager;
import SOMA.mobilePlace.*;
import SOMA.output.PlaceWindow;
import SOMA.agent.AgentWorker;
import SOMA.gui.ActionPlace;                  // Aggiunta da Gigi!
import SOMA.gui.remotegui.SportelloRichieste; // Aggiunta da Gigi!

import SOMA.explorer.*;
import SOMA.network.connection.*;
import SOMA.naming.*;
import SOMA.telnet.*;

import java.io.*;
import java.util.*;


public class Environment {

  /** Identificatore del place.
  * <code>final</code> impedisce ad un place di alterare il suo identificatore.*/
  public final PlaceID placeID;
  /** Gestore delle comunicazioni fra place.*/
  public NetworkManager networkManager = null;
  /** Menu di gestione del place.*/
  public DirExplorerItem dir;
  /** Servizio di nomi di dominio.
  * E' presente solo in un <I>default place</I>*/
  public DomainNameService domainNameService;
  /** Servizio di nomi di place.*/
  public PlaceNameService placeNameService;
  /** Gestore degli agenti.*/
  public AgentManager agentManager;
  /** Gestore dei place mobili.*/
  public MobilePlaceManager mobilePlaceManager;

  /** Gestore della finestra del Place (package gui); aggiunto da Gigi */
  public ActionPlace actionPlace;                // Aggiunta da Gigi!
  /** Thread di attesa delle richieste da parte delle applet (package gui.remoteapplet); aggiunto da Gigi */
  public SportelloRichieste sportelloRichieste;  // Aggiunta da Gigi!

  /** L'InputStream del place.*/
  public InputStream in;

  /** Il MultiOutputStream out. */
  public MultiOutputStream multiOut = new MultiOutputStream();
  /** L'OututStream del place.*/
  public PrintStream out = new PrintStream( multiOut );

  /** Il MultiOutputStream err. */
  public MultiOutputStream multiErr = new MultiOutputStream();
  /** L'ErrorStream del place.*/
  public PrintStream err = new PrintStream( multiErr );

  public ThreadGroup threadGroup = null;

  private Exception creationException = null; // Un po' sporco, ma funziona...


  /** Costruttore usato per avere un Environment "vuoto". */
  public Environment( DirExplorerItem dir, InputStream in, PrintStream out, PrintStream  err )
  {
    this.dir = dir;
    this.in = in;
    multiOut.add( out );
    multiErr.add( err );
    placeID = new PlaceID( "???", "???" );
  }

  /** Costruttore usato per avere un Environment "vuoto". */
  public Environment( PlaceID placeID)
  {
    this.placeID = placeID;
  }

  /** Costruttore: avvia il place chiamando i costruttori di tutti i suoi componenti.
  *
  * <P> I final servono solo per poter utilizzare delle classi anonime.
  */
  public Environment( final PlaceID placeID, final DirExplorerItem dir, final int port,
                      final InputStream in, final PrintStream out, final PrintStream  err )
    //throws IOException, ConnectionServer.ConnectionServerException, NameException
    throws Exception
           // Eccezioni di NetworkManager:

  {
    this.placeID = placeID;
    this.dir = dir;

    this.in = in;
    //this.out = out;
    //this.err = err;
    multiOut.add( out );
    multiErr.add( err );


    dir.addItem( "env", new ObjectExplorerItem( this ) );

    aggiungiVociDellaGUI(dir);   // AGGIUNTA DA Luigi Antenucci (per la GUI)

    createNameServices();

    if( placeID.isDomain() )
      mobilePlaceManager = new MobilePlaceManager( this );

    // Tutti i threads creati dall'environment, ossia, facenti parte del place si
    //   troveranno in questo ThreadGroup.
    threadGroup = new ThreadGroup( "[Environment:" + placeID + "]" );
    threadGroup.setDaemon( true ); // Distrutto dopo l'ultimo thread di questo gruppo

    Thread initThread =
                 new Thread(  threadGroup, toString() + " Initialization Thread." )
                    {
                      public void run()
                      {
                        try
                        {
                          //out.println( "The initialization thread: " + Thread.currentThread() );
                          //out.println( "Threadgroup: " + Thread.currentThread().getThreadGroup() );

                          networkManager = new NetworkManager( Environment.this, port );
                          out.println( "NewtworkManager created!" );

                          agentManager = new AgentManager( Environment.this );
                          out.println( "AgentManager created!" );

                          dir.addItem( "threads", new SOMA.utility.ThreadsExplorerItem() );
                        }
                        catch( Exception e )
                        {
                          creationException = e;
                        }
                      }
                    };

    initThread.setDaemon( true );
    initThread.start();
    initThread.join();

    if( creationException != null )
      throw( creationException );

    // Questo ha solo finalit di debugging.
    dir.addItem( "text", new ExplorerItem( "Writes text to 3 different PrintStreams." )
      {
        public Object Execute( java.util.Collection Parameters, PrintStream out )
        {
          out.println( "********** Testo di Explorer! *********" );
          Environment.this.out.println( "----------- Testo di PlaceWindow! -----------" );
          System.out.println( "############## Testo di System! ##############" );

          return null;
        }
      } );

    dir.addItem( "save", new ExplorerItem( "Save place info to disk." )
      {
        public Object Execute( java.util.Collection Parameters, PrintStream out )
        {
          try
          {
            save();
            out.println( "Saved!" );
          }
          catch( Exception e )
          {
            e.printStackTrace( out );
          }

          return null;
        }
      } );

    dir.addItem( "load", new ExplorerItem( "Load place info from disk." )
      {
        public Object Execute( java.util.Collection Parameters, PrintStream out )
        {
          try
          {
            load();
            out.println( "Loaded!" );
          }
          catch( Exception e )
          {
            e.printStackTrace( out );
          }

          return null;
        }
      } );

    // Questo ha solo finalit di debugging.
    /*
    (new Thread( threadGroup, "END Waiter" )
                {
                  public void run()
                  {
                    out.println( "ENVIRONMENT: I am waiting for the last thread of this place." );

                    AgentWorker.WaitForOtherThreads( out );

                    out.println( "ENVIRONMENT: this is the end!" );
                  }
                }).start();
    */
    /*
    try
    {
      initThread.join();
    }
    catch( InterruptedException e )
    {
      e.printStackTrace( out );
    }*/
  }

  /** Cambia in MobileEnvironment. */
  protected void createNameServices()
  {
    if( placeID.isDomain() )
    {
      domainNameService = new DomainNameService( this );
      dir.addItem( "dns", new DNSExplorerItem( domainNameService ) );
    }

    placeNameService = new PlaceNameService( this );
    dir.addItem( "pns", new PNSExplorerItem( placeNameService ) );
  }

  public void load() throws IOException, ClassNotFoundException
  {
    load( placeID.toString() + ".dat" );
  }

  public void save() throws IOException
  {
    save( placeID.toString() + ".dat" );
  }

  public void save( String fileName ) throws IOException
  {
    ObjectOutputStream oos = new ObjectOutputStream( new FileOutputStream( new File( fileName ) ) );

    Hashtable table = new Hashtable();
    table.put( "pns", placeNameService );
    out.println( "pns:" + placeNameService );

    table.put( "dns", domainNameService );
    out.println( "dns:" + domainNameService );

    table.put( "mpm", mobilePlaceManager );
    out.println( "mpm:" + mobilePlaceManager );

    table.put( "count", new Integer( agentManager.AgentIDCounter ) );
    out.println( "agent counter:" + agentManager.AgentIDCounter );

    oos.writeObject( table );
    oos.close();

    out.println( "Status saved!" );
  }

  public void load( String fileName ) throws IOException, ClassNotFoundException
  {
    ObjectInputStream ois = new ObjectInputStream( new FileInputStream( new File( fileName ) ) );

    Hashtable table = new Hashtable();

    table = (Hashtable)ois.readObject();
    ois.close();

    //Attenzione: bisognerebbe ripensare il tutto:
    //  vengono caricati gli oggetti e poi sostituiti gli ExplorerItem
    //  codice duplicato!

    placeNameService = (PlaceNameService)table.get( "pns" );
    if( placeNameService != null )
    {
      placeNameService.setEnv( this );
      out.println( "pns:" + placeNameService );
      dir.addItem( "pns", new PNSExplorerItem( placeNameService ) );
    }

    domainNameService = (DomainNameService)table.get( "dns" );
    if( domainNameService != null )
    {
      domainNameService.setEnv( this );
      out.println( "dns:" + domainNameService );
      dir.addItem( "dns", new DNSExplorerItem( domainNameService ) );
    }

    mobilePlaceManager = (MobilePlaceManager)table.get( "mpm" );
    if( mobilePlaceManager != null )
    {
      mobilePlaceManager.setEnv( this );
      dir.addItem( "mpm", new MobilePlaceManagerExplorerItem( mobilePlaceManager ) );
      out.println( "mpm:" + mobilePlaceManager );
    }

    int newCount = ((Integer)table.get( "count" )).intValue();
    int oldCount = agentManager.AgentIDCounter;

    if( newCount > oldCount )
    {
      agentManager.AgentIDCounter = newCount;
      out.println( "agent counter:" + oldCount + " -->" + newCount );
    }

    out.println( "Status loaded!" );
  }

      /**
       *  Metodo aggiunto da Luigi Antenucci per aggiungere le voci della GUI.
       */
  protected void aggiungiVociDellaGUI (DirExplorerItem dir) {
    // GIGI: ho modificato la seguente linea :
    //       dir.addItem( "window", new DaemonExplorerItem( new PlaceWindow( this ) ) );
    //       con la nuova, per usare la mia GUI:
    actionPlace = new ActionPlace (this);                          // Aggiunta   da Gigi !
    dir.addItem ("window", new DaemonExplorerItem(actionPlace));   // Modificata da Gigi !

    // GIGI: inoltre aggiungo la voce per interrogare il demone delle connessioni con APPLET REMOTE:
    //       NB: Usa la porta di default
    sportelloRichieste = new SportelloRichieste (actionPlace, SportelloRichieste.PORTA_DEFAULT);  // Aggiunta da Gigi!
    dir.addItem ("remoteApplet", new DaemonExplorerItem(sportelloRichieste));                     // Aggiunta da Gigi !
  } //aggiungiVociDellaGUI


  public String toString()
  {
    return "[Environment " + placeID + " " + networkManager + " " + domainNameService + " " + placeNameService + " " + agentManager + " " + mobilePlaceManager +"]";
  }
}