function temaA()
	% Flusso magnetico per unità di superficie all'air-gap
	B0 = 70;
	% Raggio di un magnete
	R = 1e-2;
	% Altezza di un magnete
	t = 1e-3;
	% Permeabilita' del vuoto
	mu0 = 4 .* pi * 1e-7;
	% Magnetizzazione
	M = B0 ./ mu0;
	% Accelerazione di gravita'
	g = 9.81;
	% Massa di un modulo del trenino
	Mt = 0.100;

	% ============================================================
	% Q1
	% ============================================================

	fz = @(x) magforce(x, R, t, mu0, M) - Mt .* g;
	[q1x, q1fval] = fsolve(fz, 0.10)

	% ============================================================
	% Q2
	% ============================================================

	q2f = @(T, X) dstate(X, R, t, mu0, M, Mt, g).';
	tspan = [0, 10];
	X0 = [0.13, 0];
	[T, X] = ode45(q2f, tspan, X0);

	q2x = X(:, 1);
	q2v = X(:, 2);

	q2xmin = min(q2x)

	figure()
	plot(T, q2x, 'linewidth', 2);
	xlabel('Tempo', 'fontsize', 14);
	ylabel('Posizione', 'fontsize', 14);
	grid();

	% ============================================================
	% Q3
	% ============================================================

	q3f = @(Mt) get_xmin(R, t, mu0, M, Mt, g) - 0.09;
	[q3Mt, q3fval] = fsolve(q3f, Mt)

	% ============================================================
	% Q4
	% ============================================================

	fz = @(x) magforce(x, R, t, mu0, M) - Mt .* g;
	[q4x, q4fval] = my_bisection(fz, 0.05, 0.15)
end


function Fm = magforce(x, R, t, mu0, M)
	% Termine di distanza nel calcolo della forza di repulsione magnetica
	dterm = (1./x.^2 + 1./(x + 2.*t).^2 - 2 ./(x + t).^2);
	Fm = (pi .* mu0 ./ 4) .* M^2 .* R^4 .* dterm;
end

function dX = dstate(X, R, t, mu0, M, Mt, g)
	% X(1) = posizione (i.e. x), X(2) = velocita'
	% Calcolo il valore delle forze in atto
	Fm = magforce(X(1), R, t, mu0, M);
	Fg = - g .* Mt;
	% Calcolo le derivate
	dX(1) = X(2);
	dX(2) = (1 ./ Mt) .* (Fm + Fg);
end

function xmin = get_xmin(R, t, mu0, M, Mt, g)
	f = @(T, X) dstate(X, R, t, mu0, M, Mt, g).';
	tspan = [0, 10];
	X0 = [0.13, 0];
	[T, X] = ode45(f, tspan, X0);
	x = X(:, 1);
	xmin = min(x);
end

function [x, fval] = my_bisection(f, x0, x1)
    tol = 1e-9;
    while true
        % Calcolo il prossimo valore di x
        x = 0.5 * (x0 + x1);
        fval = f(x);
        % Controllo se non sia il momento di terminare
        if abs(x - x0) <= tol
            break;
        end
        % Rimpiazzo x0 o x1
        if f(x0) * f(x) < 0
            x1 = x;
        else
            x0 = x;
        end
    end
end
