/*
 * EService.java
 *
 * 
 */

package appss.services.errorservice;

import java.util.*;
import java.io.*;


import appss.services.logservice.*;


/** 
 * Contiene i metodi per la gestione delle eccezioni.
 *
 * Ciascun tipo di eccezione (FatalError,GeneralError,GeneralException,
 * Warning) deve essere gestita in base alla sua tipologia.
 * <p>
 * Ogni volta che viene scatenata una eccezione, nel blocco catch, prima
 * di effettuare tutte le operazioni necessarie in locale,  necessario chiamare
 * i metodi di log e recover dell'eccezione. E' presente un metodo per ogni tipo
 * di eccezione. Il metodo ha lo stesso nome e riconosce automaticamente 
 * l'eccezione in base al suo tipo. 
 *
 * @author  webeng
 * 
 * @see FatalError
 * @see GeneralError
 * @see GeneralException
 * @see Warning
 *
 */

public class EService {
  
  /**
   * Rappresenta un errore ingestibile.
   * <p>
   * Ogni volta che nel flusso del sito si verifica una eccezione
   * che rappresenta una anormale condizione del flusso tale
   * da non poter essere gestita, l'errore  classificato con
   * questa costante affinch il flusso venga redirezionato
   * verso una pagina di errore.
   *
   */  
  
  public static final int UNRECOVERABLE_ERROR = -1; 
  
  /** 
   * Rappresenta un errore gestibile.
   * <p>
   * Ogni volta che nel flusso del sito si verifica una eccezione
   * che rappresenta una anormale condizione del flusso tale
   * da poter essere gestita, l'errore  classificato con
   * questa costante affinch il flusso lo possa gestire.
   *
   */  
      
  public static final int RECOVERABLE_ERROR = -2; 

  /** 
   * Class Constructor.
   * <p>
   * Non utilizzato in quanto i metodi della classe sono statici.
   *
   */
  
  public EService() {
  }
  
  /** 
   * Gestisce il recover di un Fatal Error. 
   * <p>
   * Gestisce il recover di un Fatal Error facendo la RollBack,
   * inviando una mail di notifica al responsabile e loggando l'errore
   * sia sul log dei fatal error che su quello della sezione di codice
   * relativa all'errore.
   * <p>
   * @param fatalError L'errore da gestire
   *
   * @see FatalError
   *
   */      
  
  public static void logAndRecover(FatalError fatalError) {
 
    fatalError.makeRollBack();

    fatalError.log();
    
    ErrorLog.fatalErrorLog(fatalError.getLogMessage());    
        
  }
  
  /** 
   * Gestisce il recover di un General Error. 
   * <p>
   * Gestisce il recover di un General Error facendo la RollBack e 
   * loggando l'errore sia sul log dei General Error che su quello
   * della sezione di codice relativa all'errore.
   * <p>
   * @param generalError L'errore da gestire
   *
   * @see GeneralError
   *
   */        
  
  public static void logAndRecover(GeneralError generalError) {
 
    generalError.makeRollBack();
    generalError.log();
    
    ErrorLog.generalErrorLog(generalError.getLogMessage());      
    
  }  

  
  /** 
   * Gestisce il recover di un Warning. 
   * <p>
   * Gestisce il recover di un Warning  
   * loggando l'errore sia sul log dei Warning che su quello
   * della sezione di codice relativa all'errore.
   * <p>
   * @param warning L'errore da gestire
   *
   * @see Warning
   *
   */            
  
  public static void logAndRecover(Warning warning) {
 
    warning.log();
    
    ErrorLog.warningLog(warning.getLogMessage());    
        
  }
  




  
  /**
   * Gestisce il recover di un Warning. 
   * <p>
   * Gestisce il recover di un Warning  
   * loggando l'errore sia sul log dei Warning che su quello
   * della sezione di codice relativa all'errore.
   * <p>
   * @param warning L'errore da gestire
   *
   * @see Warning
   *
   */            
  
  public static void logFrontendException(Throwable exception,Hashtable info,Vector parameters) {
    
    StringBuffer parametersView=new StringBuffer();
    int i;
    
    String message=exception.getMessage();
    
    ByteArrayOutputStream stackTrace=new ByteArrayOutputStream();
    exception.printStackTrace(new PrintWriter(stackTrace,true));              
    
    for (i=0;i<parameters.size();i++) {
      parametersView.append( parameters.elementAt(i)+"\n" );
    }    
    
    ErrorLog.FrontendErrorLog(message+"\n\n"+stackTrace.toString()+"\n\n"+info.toString()+"\n\n"+parametersView.toString());  
       
    
  }  
  
  public static void logFrontend(String sMesg,Hashtable info,Vector parameters) {
    
    StringBuffer parametersView=new StringBuffer();
    int i;
    
    String message=sMesg;
    
    for (i=0;i<parameters.size();i++) {
      parametersView.append( parameters.elementAt(i)+"\n" );
    }    
    
    ErrorLog.FrontendErrorLog(message+"\n\n"+info.toString()+"\n\n"+parametersView.toString());  
       
    
  }  
  
  // End of Class
  
}