/*
 * DataBase.java
 * 
 */

package appss.services.databaseservice;

import java.sql.*;
import java.io.*;
import javax.sql.*;
import java.net.URL;


import appss.services.databaseservice.exception.*;
import appss.services.logservice.*;
import appss.util.*;


/**
 * DataBase Class represents a connection with Database, it allows
 * to open connection with database, execute queries to read or modify
 * database content. This class represents a transaction and permit to commit
 * or rollback chnages and close connection.
 * @see DBService
 */
public class DataBase {
  
  private Connection oConnection;
  private Statement oStatement;
  
/**
 * Create a logic connection to DataBase.
 *
 * @param oC The physical connection with database
 * @throws NotFoundDBException Throw when it finds problems with database connection. The problem type is specified by exception message.
 */
  public DataBase(Connection oC) throws NotFoundDBException {
    
    try{
      oConnection=oC;
      if (oConnection==null) 
        throw new NotFoundDBException("DataBase: DataBase(): Impossibile Aprire la Connessione Logica");
  
      oConnection.setAutoCommit(false);
      oStatement=oConnection.createStatement();
      // sonda per il monitoraggio delle connessioni a database
      DBService.nConnections++;
      Debug.println(">: new connection:"+DBService.nConnections);      
      if (DBService.nConnections>DBService.maxConnections) 
            DBService.maxConnections=DBService.nConnections;
      // sonda per il monitoraggio delle connessioni a database
      
    } catch (Exception ex) {
      ByteArrayOutputStream stackTrace=new ByteArrayOutputStream();
      ex.printStackTrace(new PrintWriter(stackTrace,true));      
      throw new NotFoundDBException("DataBase: DataBase(): Impossibile Aprire la Connessione col DataBase: \n"+stackTrace);
    }
    
  }


  
  /**
   * This method executes the read only query sSQL. It returns ResultSet object 
   * with the query results.
   * @param sSql The query to be executed, sintax is Database vendor dependent.
   * @return resultset with query results
   * @throws NotFoundDBException Throw when it finds problems with database connection. The problem type is specified by exception message.
 */
  public ResultSet select(String sSql) throws NotFoundDBException {

    Statement oStat;
    ResultSet rs;

    try {
      rs=oStatement.executeQuery(sSql);
      return rs;
    } catch (SQLException ex) {
      throw new NotFoundDBException("DataBase: select(): Impossibile eseguire la query sul DB. Eccezione: "+ex+ "\n " + sSql ,this);
    }
  }


  
  /**
   * Execute query modifying database content.
   * This method executes query and returns the number of records modified.
   * To complete transaction and confirm changes the commit method needs to be called.
   * @param sSql The query to be executed, sintax is Database vendor dependent.
   * @return returns the number of records modified by the query.
   * @throws NotFoundDBException Throw when it finds problems with database connection. The problem type is specified by exception message.
 */  
  public int modify(String sSql) throws NotFoundDBException {   
      
    int nRecs=-2;
    try {
      nRecs=oStatement.executeUpdate(sSql);
    } catch (SQLException ex){
      
      throw new NotFoundDBException("DataBase: modify(): Impossibile eseguire la update sul DB. Eccezione: "+ex+ "\n " + sSql,this);
      
    }
    
    return nRecs;
    
  }
  
  /**
   * Commit the transaction.
   * @throws NotFoundDBException Throw when it finds problems with database connection. The problem type is specified by exception message.
 */  
  
  public void commit() throws NotFoundDBException {
    
    try{
      
      oConnection.commit();
      oStatement.close();
      oStatement=oConnection.createStatement();
      return;
    
    } catch (SQLException ex){
    
      throw new NotFoundDBException("DataBase: commit(): Impossibile eseguire la commit sul DB. Eccezione: "+ex,this);
      
    }
    
}

/**
 * RollBack the transaction.
 */

 public void rollBack() {
    
  try{
    oConnection.rollback();
    oStatement.close();
    oStatement=oConnection.createStatement();    
    return;
   } catch (SQLException ex){
     new NotFoundDBException("DataBase: rollback(): Impossibile eseguire il RollBack sul DB. Eccezione: "+ex);
   }
 }

  /**
   * Explicitly close database connection setting resources free.
   *
   * @throws NotFoundDBException Throw when it finds problems with database connection. The problem type is specified by exception message.
 */
  public void close() throws NotFoundDBException {
    try{
     
      oConnection.close();
      
     } catch (SQLException ex){
       throw new NotFoundDBException("DataBase: close(): Impossibile chiudere il DB. Eccezione: "+ex);
     }
  }
  
/**
 * Close database connection when garbage collector finalize object instance.
 * @throws Throwable Garbage Collector exception
*/
  protected void finalize() throws Throwable {
    this.close();
    DBService.nConnections--;
    Debug.println(">: connection close:"+DBService.nConnections);
  }
                   
  
}
