//******************************************************************
 //******************************************************************
  //**********          ANts Peer To Peer Sources        *************
   //
   // ANts P2P realizes a third generation P2P net. It protects your
   // privacy while you are connected and makes you not trackable, hiding
   // your identity (ip) and crypting everything you are sending/receiving
   // from others.

   // Copyright (C) 2004  Roberto Rossi

   // This program is free software; you can redistribute it and/or
   // modify it under the terms of the GNU General Public License
   // as published by the Free Software Foundation; either version 2
   // of the License, or (at your option) any later version.

   // This program is distributed in the hope that it will be useful,
   // but WITHOUT ANY WARRANTY; without even the implied warranty of
   // MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   // GNU General Public License for more details.

   // You should have received a copy of the GNU General Public License
   // along with this program; if not, write to the Free Software
   // Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

package ants.p2p.utils;

import java.util.*;
import java.io.*;
import java.beans.*;

import ants.p2p.*;
import ants.p2p.query.*;

import org.apache.log4j.*;

/**
 * File indexer engine: keep a table with infos about indexed files
 * hash
 * dimension
 * name
 * location
 */
public class BackgroundEngine
    extends Thread {
  Hashtable partialFiles = new Hashtable();
  public Hashtable sharedFiles = new Hashtable();
  public ArrayList sharedDirectories = new ArrayList();
  static BackgroundEngine instance;
  public static int refreshRate = 30000;
  File store = new File("sharedFiles.ant");
  private PropertyChangeSupport propertyChangeSupport = new
      PropertyChangeSupport(this);

  static Logger _logger = Logger.getLogger(BackgroundEngine.class.getName());

  public static BackgroundEngine getInstance(File store) {
    try {
      if (instance == null) {
        instance = new BackgroundEngine(store);
        instance.setPriority(Thread.MIN_PRIORITY);
        instance.start();
        return instance;
      }
      else {
        return instance;
      }
    }
    catch (Exception ex) {
      _logger.error("", ex);
      return null;
    }
  }

  public void addPropertyChangeListener(PropertyChangeListener pcl){
    this.propertyChangeSupport.addPropertyChangeListener(pcl);
    DigestManager.addPropertyChangeListener(pcl);
  }

  public void removePropertyChangeListener(PropertyChangeListener pcl){
    this.propertyChangeSupport.removePropertyChangeListener(pcl);
    DigestManager.removePropertyChangeListener(pcl);
  }

  public void addPartialFile(MultipleSourcesDownloadManager msdm) {
    this.partialFiles.put(msdm.getFileHash(), msdm);
  }

  public void addPartialFile(InterruptedDownload id) {
    this.partialFiles.put(id.getFileHash(), id);
  }

  public void removePartialFile(String hash) {
    this.partialFiles.remove(hash);
  }

  public Enumeration getPartialFiles() {
    return this.partialFiles.keys();
  }

  public QueryPartialFileTuple getPartialFileTuple(String sessionKey,
      String fileHash, String ownerID,
      Integer freeSlots, String connectionType) {
    Object partialFile = this.partialFiles.get(fileHash);
    String fileName = null;
    Long fileLength = null;
    Integer blockSize = null;
    Integer blocksPerSource = null;
    String percentage = null;
    boolean[] downloadedBlockGroups = null;
    if (partialFile instanceof MultipleSourcesDownloadManager) {
      MultipleSourcesDownloadManager msdmPartialFile = (
          MultipleSourcesDownloadManager) partialFile;
      fileName = msdmPartialFile.getFileName();
      fileLength = new Long(msdmPartialFile.getFileSize());
      blockSize = new Integer(msdmPartialFile.getBlockSize());
      blocksPerSource = new Integer(MultipleSourcesDownloadManager.
                                    blocksPerSource);
      downloadedBlockGroups = msdmPartialFile.getDownloadedBlockGroups();
      percentage = msdmPartialFile.getPercentage();
    }
    else if (partialFile instanceof InterruptedDownload) {
      InterruptedDownload idPartialFile = (InterruptedDownload) partialFile;
      fileName = idPartialFile.getFileName();
      fileLength = new Long(idPartialFile.getFileSize());
      blockSize = new Integer(idPartialFile.getBlockSize());
      blocksPerSource = new Integer(MultipleSourcesDownloadManager.
                                    blocksPerSource);
      downloadedBlockGroups = idPartialFile.getDownloadedBlockGroups();
      percentage = idPartialFile.getPercentage();
    }
    if (fileName != null && fileLength != null && blockSize != null &&
        downloadedBlockGroups != null && percentage != null) {
      return new QueryPartialFileTuple(sessionKey, fileName, fileHash,
                                       fileLength, blockSize, blocksPerSource,
                                       downloadedBlockGroups, ownerID,
                                       freeSlots, connectionType, percentage);
    }
    else {
      throw new NullPointerException(
          "Null pointer in QueryPartialFileTuple parameters");
    }
  }

  public static BackgroundEngine getInstance() {
    try {
      if (instance == null) {
        instance = new BackgroundEngine();
        instance.start();
        return instance;
      }
      else {
        return instance;
      }
    }
    catch (Exception ex) {
      _logger.error("", ex);
      return null;
    }
  }

  BackgroundEngine() throws IOException, ClassNotFoundException {
    if (store.exists()) {
      load(store);
    }
  }

  BackgroundEngine(File f) throws IOException, ClassNotFoundException {
    store = f;
    load(f);
  }

  public void forceExternalUpdate() {
    this.propertyChangeSupport.firePropertyChange(
        "SharedDirectoriesModification", null, this);
  }

  public void setStoreFile(File store) {
    this.store = store;
  }

  private void load(File source) {
    try{
      ObjectInputStream ois = new ObjectInputStream(new FileInputStream(
          source));
      sharedFiles = (Hashtable) ois.readObject();
      sharedDirectories = (ArrayList) ois.readObject();
      ois.close();
    }catch(Exception e){
      _logger.info("Error loading shared files infos");
      _logger.debug("TRACE",e);
    }
  }

  private void store(File dest) {
    try {
      ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(
          dest));
      oos.writeObject(sharedFiles);
      oos.writeObject(sharedDirectories);
      oos.flush();
      oos.close();
    }
    catch (Exception e) {
      _logger.info("Error loading shared files infos");
      _logger.debug("TRACE", e);
    }
  }

  public synchronized void addDirectory(File directory) {
    if (directory.isDirectory()) {
      if(!this.sharedDirectories.contains(directory)){
        this.sharedDirectories.add(directory);
        this.propertyChangeSupport.firePropertyChange(
            "SharedDirectoriesModification", null, this);
      }
    }
  }

  public synchronized void removeDirectory(File directory) {
    this.sharedDirectories.remove(directory);
    this.propertyChangeSupport.firePropertyChange(
        "SharedDirectoriesModification", null, this);
  }

  public void run() {
    while (true) {
      try {
        boolean changes = false;
        this.propertyChangeSupport.firePropertyChange("fileIndexingCompleted", null, null);
        Thread.sleep(BackgroundEngine.refreshRate);
        this.propertyChangeSupport.firePropertyChange("fileIndexingInit", null, null);
        ArrayList fileList = new ArrayList();
        Hashtable tempHashtable = (Hashtable) sharedFiles.clone();
        for (int x = 0; x < this.sharedDirectories.size(); x++) {
          File[] files = ( (File)this.sharedDirectories.get(x)).listFiles();
          if (files != null) {
            for (int y = 0; y < files.length; y++) {
              if (files[y].isFile()) {
                fileList.add(files[y].getAbsolutePath());
                tempHashtable.remove(files[y].getAbsolutePath());
              }
            }
          }
        }
        Enumeration toBeRemoved = tempHashtable.keys();
        while (toBeRemoved.hasMoreElements()) {
          changes = true;
          sharedFiles.remove(toBeRemoved.nextElement());
        }
        for (int y = fileList.size() - 1; y >= 0; y--) {
          if (!sharedFiles.containsKey(fileList.get(y))) {
            changes = true;
            sharedFiles.put(fileList.get(y),
                          new FileInfos(new File( (String) fileList.get(y))));
          }
          else {
            if (! ( ( (FileInfos) sharedFiles.get(fileList.get(y))).
                   getLastModified() ==
                   (new File( (String) fileList.get(y))).lastModified())) {
              changes = true;
              sharedFiles.put(fileList.get(y),
                              new FileInfos(new File( (String) fileList.get(y))));
            }
          }
          this.propertyChangeSupport.firePropertyChange("fileIndexed", new Integer(fileList.size()), new Integer(y));
        }
        if (changes) {
          this.store(store);
          Enumeration visualizeKeys = sharedFiles.keys();
          Enumeration visualizeValues = sharedFiles.elements();
          while (visualizeKeys.hasMoreElements()) {
            _logger.info(visualizeKeys.nextElement() + ".....");
            FileInfos fi = (FileInfos) visualizeValues.nextElement();
            _logger.info(fi.getHash() + "....." + fi.getLastModified() +
                         "\n");
          }
          this.propertyChangeSupport.firePropertyChange(
              "SharedDirectoriesModification", null, this);
        }
      }
      catch (Exception e) {
        _logger.error("Background Indexer Error", e);
        this.propertyChangeSupport.firePropertyChange("fileIndexingCompleted", null, null);
      }
    }
  }
}