//******************************************************************
//******************************************************************
//**********          ANts Peer To Peer Sources        *************
//
// ANts P2P realizes a third generation P2P net. It protects your
// privacy while you are connected and makes you not trackable, hiding
// your identity (ip) and crypting everything you are sending/receiving
// from others.

// Copyright (C) 2004  Roberto Rossi

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

package ants.p2p;

import java.io.*;
import java.util.*;
import java.beans.*;
import javax.crypto.*;
import java.net.*;
import ants.p2p.messages.*;
import ants.p2p.security.*;
import ants.p2p.query.*;
import ants.p2p.query.security.*;
import ants.p2p.utils.*;

import org.apache.log4j.*;

/**
 * This class extends the base Ant and realizes a service provider for
 * Transport/Session and Application Layers.
 * It manages:
 * transport/session services.... secure connections between nodes
 * node addresses lookup........
 * queries......................
 * file part transfers........... x bytes at given offset
 * message processing............
 */
public class WarriorAnt extends Ant {
  java.util.List partialFiles = Collections.synchronizedList(new ArrayList());
  public java.util.List outputSecureConnections = Collections.synchronizedList(new ArrayList());
  public java.util.List inputSecureConnections = Collections.synchronizedList(new ArrayList());
  public java.util.List pendingSecureRequest = Collections.synchronizedList(new ArrayList());
  public java.util.List inServiceFilePullRequests = Collections.synchronizedList(new ArrayList());
  public java.util.List inServiceFilePushRequests = Collections.synchronizedList(new ArrayList());
  public java.util.List pendingFilePullRequests = Collections.synchronizedList(new ArrayList());

  public java.util.List scheduledDownloads = Collections.synchronizedList(new ArrayList());
  public static int maxScheduledDownloads = 1000;

  public static String downloadPath = "./";
  public static String chunksPath = "./chunks/";
  public static int maxOwnPullRequestsToTrace = 1000;
  public static int maxPullRequestsToServe = 100;
  public static int maxSecureConnections = 1000;
  public static int maxPartialFiles = 500;
  public static int blockSizeInDownload = (int)Math.pow(2,15);
  public static String ConnectionType = "56K";
  public static int maxQueryTimeToLive = 5000;
  public static int minQueryTimeToLive = 200;
  public static long queryTimeToLive = 1000;
  public static boolean isServer = true;
  public static String password = "";
  public static int processProbability = 50;

  static Logger _logger = Logger.getLogger(WarriorAnt.class.getName());

  AddressServer addressServer = null;

  public WarriorAnt(String id, int maxNeighbours, int serverPort){
    super(id,maxNeighbours, serverPort);
    WarriorAnt.processProbability += (int)(System.currentTimeMillis() % 25);
    WarriorAnt.checkChunksPath();
  }

  public void startAddressServer(){
    if(isServer && this.sss!=null){
      addressServer = new AddressServer(this);
    }
  }

  public void disconnectWarrior(){
    if(addressServer!=null)
      addressServer.terminate();
    if(InetAddressEngine.getInstance()!=null)
      InetAddressEngine.getInstance().terminate();
    this.disconnect();
  }

  public void scheduleDownload(QueryFileTuple qft){
    if(this.scheduledDownloads.size() > WarriorAnt.maxScheduledDownloads){
      this.scheduledDownloads.remove(0);
    }
    this.scheduledDownloads.add(qft);
  }

  void checkScheduledDownloads(EndpointSecurityManager esm){
    for(int x = 0; x < this.scheduledDownloads.size(); x++){
      QueryFileTuple qft = (QueryFileTuple)this.scheduledDownloads.get(x);
      if(esm.getPeerId().equals(qft.getOwnerID())){
        this.scheduledDownloads.remove(x);
        this.propertyChangeSupport.firePropertyChange("addSourcePeer",null,qft);
      }
    }
  }

  public void createSecureConnection(String peerId){
    try{
      if(this.outputSecureConnections.size() >= maxSecureConnections)
        return;

      _logger.info(this.getId() + " Creating secure connection");

      for (int x = this.outputSecureConnections.size() - 1; x >= 0; x--) {
        EndpointSecurityManager esm = (EndpointSecurityManager) this.
            outputSecureConnections.get(x);
        if (esm.getPeerId().equals(peerId)) {
          this.myMessages.remove(esm.getSecurityMessage());
          this.outputSecureConnections.remove(esm);
        }
      }
      for (int x = this.pendingSecureRequest.size() - 1; x >= 0; x--) {
        EndpointSecurityManager esm = (EndpointSecurityManager) this.
            pendingSecureRequest.get(x);
        if (esm.getPeerId().equals(peerId)) {
          this.myMessages.remove(esm.getSecurityMessage());
          this.pendingSecureRequest.remove(esm);
        }
      }

      if (this.pendingSecureRequest.size() >= maxSecureConnections) {
        EndpointSecurityManager esm = (EndpointSecurityManager)this.
            pendingSecureRequest.get(this.pendingSecureRequest.size() - 1);
        if (System.currentTimeMillis() - esm.getLastTimeUsed() >
            Ant.messageTimeout * Ant.maxRetransmissions)
          this.pendingSecureRequest.remove(this.inputSecureConnections.size() -
                                             1);
      }


      ants.p2p.security.EndpointSecurityManager esm = new
          ants.p2p.security.EndpointSecurityManager(this, peerId);
    }
     catch (Exception e) {
       _logger.error("",e);
     }
  }

  public void removeInputSecureConnection(String peerId){
    for (int x = this.inputSecureConnections.size() - 1; x >= 0; x++) {
      if ( ( (EndpointSecurityManager)this.inputSecureConnections.get(x)).
          getPeerId().equals(peerId))
        this.inputSecureConnections.remove(this.inputSecureConnections.get(x));
    }
  }

  public void removeOutputSecureConnection(String peerId){
    for (int x = this.outputSecureConnections.size() - 1; x >= 0; x++) {
      if ( ( (EndpointSecurityManager)this.outputSecureConnections.get(x)).
          getPeerId().equals(peerId))
        this.outputSecureConnections.remove(this.outputSecureConnections.get(x));
    }
  }

  public void pullFile(String fileName, String fileHash, long offset, long blocks, String peerId, int blockSize, String localFileName, boolean resume)throws Exception{
    ants.p2p.security.EndpointSecurityManager esm = this.
          getOutputSecureConnectionManager(peerId);
      if (esm == null)
        throw new Exception("No secure connection avaiable with endpoint " +
                            peerId);

      FilePullMessage frm = new FilePullMessage(fileName,Base16.fromHexString(fileHash),new Long(offset), new Long(blocks), new Integer(blockSize), localFileName, new Boolean(resume));
      synchronized(this.pendingFilePullRequests){
        frm.encrypt(esm.getCipherEnc());
        MessageWrapper wm = this.sendMessage(frm, peerId, true, false);
        while(this.myMessages.contains(wm)){
          Thread.sleep(1000);
        }
        if (this.failedMessages.contains(wm)) {
          frm.decrypt(esm.getCipherDec());
          this.failedMessages.remove(wm);

          _logger.debug(this.getId() +
                               ": Error in sending PullFileMessage " +
                               frm.getFileName() +
                               " - Cannot send pull request to id " +
                               frm.getDest());

          this.propertyChangeSupport.firePropertyChange("filePullError",null,frm);
          return;
        }
        FilePullMessage frmCopia = new FilePullMessage(frm,fileName,Base16.fromHexString(fileHash),new Long(offset), new Long(blocks), new Integer(blockSize), localFileName, new Boolean(resume));

        boolean removed = false;
        for (int x = this.pendingFilePullRequests.size() - 1; x >= 0; x--) {
          FilePullMessage fpr = (FilePullMessage) (this.pendingFilePullRequests.get(x));
          if (fpr.getDest().equals(frmCopia.getDest()) &&
              compareHash(fpr.getHash(),frmCopia.getHash()) &&
              fpr.getLocalFileName().equals(frmCopia.getLocalFileName()) &&
              fpr.getBlocks().equals(frmCopia.getBlocks())&&
              fpr.getBlockSize().equals(frmCopia.getBlockSize())&&
              fpr.getOffset().equals(frmCopia.getOffset())) {
            this.pendingFilePullRequests.remove(x);
            removed = true;
          }
        }
        for (int x = this.inServiceFilePushRequests.size() - 1; x >= 0; x--) {
          FilePullMessage fpr = (FilePullMessage) (this.inServiceFilePushRequests.get(x));
          if (fpr.getDest().equals(frmCopia.getDest()) &&
              compareHash(fpr.getHash(),frmCopia.getHash()) &&
              fpr.getLocalFileName().equals(frmCopia.getLocalFileName()) &&
              fpr.getBlocks().equals(frmCopia.getBlocks())&&
              fpr.getBlockSize().equals(frmCopia.getBlockSize())&&
              fpr.getOffset().equals(frmCopia.getOffset())) {
            this.inServiceFilePushRequests.remove(x);
            removed = true;
          }
        }
        if(!removed && this.pendingFilePullRequests.size() >= WarriorAnt.maxOwnPullRequestsToTrace)
          this.pendingFilePullRequests.remove(0);
        this.pendingFilePullRequests.add(frmCopia);
      }
  }

  public void pullFileSize(String fileName, String fileHash, String peerId)throws Exception{
    ants.p2p.security.EndpointSecurityManager esm = this.
          getOutputSecureConnectionManager(peerId);
      if (esm == null)
        throw new Exception("No secure connection avaiable with endpoint " +
                            peerId);

      FileSizePullMessage fsrm = new FileSizePullMessage(fileName,Base16.fromHexString(fileHash));
      fsrm.encrypt(esm.getCipherEnc());
      MessageWrapper wm = this.sendMessage(fsrm, peerId, true, false);
      while(this.myMessages.contains(wm)){
        Thread.sleep(1000);
      }
      if (this.failedMessages.contains(wm)) {
        this.failedMessages.remove(wm);
        fsrm.decrypt(esm.getCipherDec());

        _logger.debug(this.getId() +
                             ": Error in sending PullFileSizeMessage " +
                             fsrm.getFileName() +
                             " - Cannot send pull request to id " +
                             fsrm.getDest());

        this.propertyChangeSupport.firePropertyChange("fileSizePullError",null,fsrm);
        return;
      }
  }

  public void getServersWithFreeSlots(long timeToLive){
    try{
      QueryInetAddressItem qia = new QueryInetAddressItem(null);
      AsymmetricProvider ap = new AsymmetricProvider(false);
      QueryMessage qm = new QueryMessage(qia, timeToLive, ap.getPublicHeader());

      QueryMessage qmLocal = new QueryMessage(qm);
      QueryManager queryManager = new QueryManager(qmLocal, this);
      queryManager = new QueryManager(qmLocal);
      this.propertyChangeSupport.firePropertyChange(
              "inetAddressQueryCompleted", null, queryManager.resultSet);

      if (this.getNeighboursNumber() > 0) {
        MessageWrapper wm = this.sendBroadcastMessage(qm);
      }
    }catch(Exception ex){_logger.error("",ex);}
  }

  public synchronized ArrayList getServersWithFreeSlots(InetAddress host, int port, InetAddress localhost){
    Socket local = null;
    try{
      local = new Socket(host, port);
      local.setSoTimeout(3000);
      ObjectInputStream ois = new ObjectInputStream(local.getInputStream());
      ArrayList peers = (ArrayList)ois.readObject();
      for(int x=peers.size()-1; x >= 0; x--){
        ServerInfo peer = (ServerInfo) peers.get(x);
        if(peer.getAddress().equals(localhost) &&
          peer.getPort().intValue()==this.getServerPort()){
         peers.remove(peer);
       }
      }
      ois.close();
      return peers;
    }catch(Exception e){
      _logger.error("Failed to get peers infos: "+e.getMessage());
      if(local != null)
        try{
          local.close();
        }catch(Exception ex){}
      return null;
    }
  }

  public MessageWrapper doQuery(QueryNode query, PublicHeader ph, long timeToLive) throws Exception{
      QueryMessage qm = new QueryMessage(query, timeToLive, ph);
      MessageWrapper wm = this.sendBroadcastMessage(qm);
      return wm;
  }

  //At least one semantic
  public boolean areInMyMessages(ArrayList list){
    for(int x=0; x<list.size(); x++){
      if (this.myMessages.contains(list.get(x)))
        return true;
    }
    return false;
  }

  //At least one semantic
  public boolean areInFailedMessages(ArrayList list){
    for(int x=0; x<list.size(); x++){
      if (this.failedMessages.contains(list.get(x)))
        return true;
    }
    return false;
  }

  public ArrayList removeDeliveredMessagesFromList(ArrayList list){
    for(int x=list.size()-1; x>=0 ; x--){
      if (!this.myMessages.contains(list.get(x)) &&
          !this.failedMessages.contains(list.get(x)))
        list.remove(x);
    }
    return list;
  }

  public static void checkChunksPath(){
    File cpf = new File(WarriorAnt.chunksPath);
    if(cpf.exists() && !cpf.isDirectory()){
      cpf.delete();
      cpf.mkdir();
    }else if(!cpf.exists()){
      cpf.mkdir();
    }
  }

  void processMessage(Message m, Router r){

    _logger.debug("Processing message: ID " + m.getAck_Id() + " Type: " + m.getType() +" From: " +
                         m.getSource() + " To: " + m.getDest());

    try{
      if (m.getType() == 1 && (m instanceof QueryMessage)) {
        processQueryMessage( (QueryMessage) m, r);
      }
      else if (m.getType() == 1) {
      }
      else if (m.getType() != 1) {
        if (m instanceof FilePushMessage) {
          this.processFilePushMessage( (FilePushMessage) m);
        }
        if (m instanceof FileSizePushMessage) {
          this.processFileSizePushMessage( (FileSizePushMessage) m);
        }
        if (m instanceof FilePullMessage) {
          this.processFilePullMessage( (FilePullMessage) m);
        }
        if (m instanceof FileSizePullMessage) {
          this.processFileSizePullMessage( (FileSizePullMessage) m);
        }
        if (m instanceof FilePartMessage) {
          this.processFilePartMessage( (FilePartMessage) m);
        }
        if (m instanceof ControlMessage) {
          this.processControlMessage( (ControlMessage) m);
        }
        if (m instanceof SecurityRequestMessage) {
          this.processSecurityRequestMessage( (SecurityRequestMessage) m);
        }
        if (m instanceof SecurityResponseMessage) {
          this.processSecurityResponseMessage( (SecurityResponseMessage) m);
        }
        if (m instanceof QueryMessage) {
          this.processQueryMessage( (QueryMessage) m, r);
        }
      }

      _logger.debug("Processed message: ID " + m.getAck_Id() + " Type: " + m.getType() +" From: " +
                           m.getSource() + " To: " + m.getDest());

    }
    catch (Exception e) {
      _logger.error(this.getId() +
                         ": Error In Processing message: id = " +
                         m.getAck_Id() + " Source: " + m.getSource() +
                         " Dest: " + m.getDest(),e);
    }
  }

  private void processFileSizePullMessage(FileSizePullMessage fspm) throws
      Exception {
    try {
      ants.p2p.security.EndpointSecurityManager esm = this.
          getInputSecureConnectionManager(fspm.getSource());
      if (esm == null)
        throw new Exception("No secure connection avaiable with endpoint " +
                            fspm.getSource());
      fspm.decrypt(esm.getCipherDec());
      byte[] hash = fspm.getHash();
      File f = new File(fspm.getFileName());

      if (!f.exists()) {
        FileSizePullErrorControlMessage fterrcm = new
            FileSizePullErrorControlMessage(new Integer(0), hash,
                                            "File don't exist");
        fterrcm.encrypt(esm.getCipherEnc());
        MessageWrapper wm = this.sendMessage(fterrcm, fspm.getSource(), false, false);

        _logger.debug(this.getId() + ": File " + fspm.getFileName() +
                             " don't exist. Cannot serve pull request from id " +
                             fspm.getSource());

        while (this.myMessages.contains(wm)) {
          Thread.sleep(1000);
        }
        if (this.failedMessages.contains(wm)) {
          this.failedMessages.remove(wm);
          fterrcm.decrypt(esm.getCipherDec());

          _logger.debug(this.getId() +
                               ": Error in sending ControlMessage " +
                               fterrcm.getMessage());

          return;
        }
        return;
      }
      FileSizePushMessage fsrm = new FileSizePushMessage(f.getName(), hash,
          new Long(f.length()));
      fsrm.encrypt(esm.getCipherEnc());
      MessageWrapper wm = this.sendMessage(fsrm, fspm.getSource(), false, false);
      while (this.myMessages.contains(wm)) {
        Thread.sleep(1000);
      }
      if (this.failedMessages.contains(wm)) {
        this.failedMessages.remove(wm);
        fsrm.decrypt(esm.getCipherDec());

        _logger.debug(this.getId() +
                             ": Error in sending FileSizePushMessage " +
                             fsrm.getFileName() +
                             " - Cannot serve pull request from id " +
                             fspm.getSource());

        return;
      }
    }
    catch (Exception e) {
      _logger.error("",e);
      synchronized (inServiceFilePullRequests) {
        if (inServiceFilePullRequests.contains(fspm)) {
          inServiceFilePullRequests.remove(fspm);
        }
      }

      throw new Exception(this.getId() +
                          ": Error In Processing FileSizePullMessage: id = " +
                          fspm.getAck_Id() + " Source: " + fspm.getSource() +
                          " Dest: " + fspm.getDest());
    }
  }

  private void processQueryMessage(QueryMessage qm, Router r) throws
      Exception {
    try {
      if (qm.getTTL() > WarriorAnt.maxQueryTimeToLive){
        qm.ttl = WarriorAnt.maxQueryTimeToLive;
      }
      if (qm.getType()==2 && !qm.getProcessed() && !qm.getDelivered()) {
        double mustProcess = Math.random() * 100;
        if(mustProcess < WarriorAnt.processProbability && !qm.getSource().equals(this.getId())){
          if (!this.inTransitMessages.contains(qm) &&
              !qm.getSource().equals(this.getId())) {
            _logger.info("Query viene processata");
            QueryResultSenderThread qrst = new QueryResultSenderThread(this, qm, r);
            qrst.start();

            _logger.debug(this.getId() + ": Processed query by " +
                                 qm.getSource() + ".");

          }
        }else
          _logger.info("Query non viene processata");
      }
      else if (qm.getType()==1 && qm.getProcessed()) {
        QueryManager queryManager = new QueryManager(qm);
        if(qm.getQuery() instanceof QueryInetAddressItem){
          _logger.info("Ricevuta parte di query inetAddress");
          this.propertyChangeSupport.firePropertyChange(
              "inetAddressQueryCompleted", null, queryManager.resultSet);
        }else{
          this.propertyChangeSupport.firePropertyChange("queryCompleted", qm,
              queryManager.resultSet);
          _logger.info("Ricevuta parte di query file");
        }
      }else
        throw new Exception(this.getId() +
                            ": Error In Processing query. Source: " +
                            qm.getSource());
    }
    catch (Exception e) {
      _logger.error("",e);
      throw new Exception(this.getId() +
                          ": Error In Processing query. Source: " + qm.getSource());
    }
  }

  private void processFilePullMessage(FilePullMessage fpm) throws Exception{
    synchronized(this.inServiceFilePullRequests){
      if(this.inServiceFilePullRequests.size()>=maxPullRequestsToServe){
        _logger.info("Already serving max number of pull requests.");
        return;
      }
    }
    _logger.info("Running FilePullMessageProcessor. Free slots: "+(WarriorAnt.maxPullRequestsToServe - this.inServiceFilePullRequests.size()));
    FilePullMessageProcessor fpmp = new FilePullMessageProcessor(this,fpm);
    fpmp.start();
  }

  private void processFileSizePushMessage(FileSizePushMessage fspm) throws Exception{
    try {
      ants.p2p.security.EndpointSecurityManager esm = this.
          getOutputSecureConnectionManager(fspm.getSource());
      if (esm == null)
        throw new Exception("No secure connection avaiable with endpoint " +
                            fspm.getSource());

      fspm.decrypt(esm.getCipherDec());
      _logger.info("File: name="+fspm.getFileName()+" hash="+Base16.toHexString(fspm.getHash())+" size="+fspm.getSize());
      this.propertyChangeSupport.firePropertyChange("fileSizePushReceived",null,fspm);
    }catch (Exception e) {
      this.propertyChangeSupport.firePropertyChange("fileSizePushError",null,fspm);
      _logger.error("",e);
      throw new Exception(this.getId() +
                          ": Error In Processing FileSizePushMessage: id = " +
                          fspm.getAck_Id() + " Source: " + fspm.getSource() +
                          " Dest: " + fspm.getDest());
    }
  }

  private void processFilePushMessage(FilePushMessage fpm) throws Exception{
    try{
      ants.p2p.security.EndpointSecurityManager esm = this.
          getOutputSecureConnectionManager(fpm.getSource());
      if (esm == null)
        throw new Exception("No secure connection avaiable with endpoint " +
                            fpm.getSource());

      fpm.decrypt(esm.getCipherDec());

      boolean proceed = false;
      synchronized (this.pendingFilePullRequests) {
        for (int x = this.pendingFilePullRequests.size() - 1; x >= 0; x--) {
          FilePullMessage fpr = (FilePullMessage) (this.pendingFilePullRequests.
              get(x));
          if (fpr.getDest().equals(fpm.getSource()) &&
              compareHash(fpr.getHash(),fpm.getHash())){
            synchronized (this.inServiceFilePushRequests) {
              _logger.info(this.getId() + ": Found pending pull request... received serving push!!!");
              if(this.inServiceFilePushRequests.size() >= WarriorAnt.maxOwnPullRequestsToTrace)
                this.inServiceFilePushRequests.remove(0);
              this.inServiceFilePushRequests.add(this.pendingFilePullRequests.get(x));
              this.pendingFilePullRequests.remove(x);
              proceed=true;
            }
          }
        }
      }
      if(!proceed){
        _logger.info(this.getId() + ": Not Found pending pull request... ingnoring message.");
        return;
      }

      this.propertyChangeSupport.firePropertyChange("filePushInit",null,fpm);
      _logger.info(this.getId() + ": Creating file: name="+fpm.getFileName()+" hash="+Base16.toHexString(fpm.getHash()));
      byte[] hash = fpm.getHash();
      String name = new String(fpm.getFileName());
      PartialFile pf = new PartialFile(name, fpm.getSource(), hash, fpm.getResume().booleanValue(), fpm.getOffset().longValue(), fpm, this.propertyChangeSupport);
      if(partialFiles.size() > WarriorAnt.maxPartialFiles){
        partialFiles.remove(0);
      }
      partialFiles.add(pf);
    }
    catch (Exception e) {
      _logger.error("",e);
      this.propertyChangeSupport.firePropertyChange("filePushError",null,fpm);
      throw new Exception(this.getId() +
                          ": Error In Processing FilePushMessage: id = " +
                          fpm.getAck_Id() + " Source: " + fpm.getSource() +
                          " Dest: " + fpm.getDest());
    }
  }

  private void processFilePartMessage(FilePartMessage fpm) throws Exception {
    try {
      ants.p2p.security.EndpointSecurityManager esm = this.
          getOutputSecureConnectionManager(fpm.getSource());
      if (esm == null){
        this.propertyChangeSupport.firePropertyChange("securityConnectionError",null,fpm);
        throw new Exception("No secure connection avaiable with endpoint " +
                            fpm.getSource());
      }

      fpm.decrypt(esm.getCipherDec());

      byte[] hash = fpm.getHash();
      PartialFile part = this.getPartialFile(hash, fpm.getFilePushMessage());
      if (part != null) {
        part.appendBytes(fpm.getContent(), fpm.getPartId().longValue());
        this.propertyChangeSupport.firePropertyChange("filePartUpdate",null,fpm);
      }else{
        throw new Exception(this.getId() +
                          ": Partial file is null. Error In Processing FilePartMessage: id = " +
                          fpm.getAck_Id() + " Source: " + fpm.getSource() +
                          " Dest: " + fpm.getDest());
      }
    }
    catch (Exception e) {
      _logger.error("",e);
      if (!e.getMessage().equals("No secure connection avaiable with endpoint " +
                                 fpm.getSource())) {
        this.propertyChangeSupport.firePropertyChange("filePartError", null,
            fpm);
      }
      throw new Exception(this.getId() +
                          ": Error In Processing FilePartMessage: id = " +
                          fpm.getAck_Id() + " Source: " + fpm.getSource() +
                          " Dest: " + fpm.getDest());
    }
  }

  private void processControlMessage(ControlMessage cm) throws Exception {
    try {
      ants.p2p.security.EndpointSecurityManager esm;

      if (cm instanceof FileTransferEndControlMessage){
        esm = this.
            getOutputSecureConnectionManager(cm.getSource());
        if (esm == null)
          throw new Exception("No secure connection avaiable with endpoint " +
                              cm.getSource());
        cm.decrypt(esm.getCipherDec());
        FileTransferEndControlMessage ftecm = (FileTransferEndControlMessage)
            cm;
        this.processFileTransferEndControlMessage(ftecm);
      }
      else if (cm instanceof FileTransferErrorControlMessage) {
        esm = this.
            getOutputSecureConnectionManager(cm.getSource());
        if (esm == null)
          throw new Exception("No secure connection avaiable with endpoint " +
                              cm.getSource());
        cm.decrypt(esm.getCipherDec());
        FileTransferErrorControlMessage fterrcm = (FileTransferErrorControlMessage)
            cm;
        this.processFileTransferErrorControlMessage(fterrcm);
      }
      else if (cm instanceof FileSizePullErrorControlMessage) {
        esm = this.
            getOutputSecureConnectionManager(cm.getSource());
        if (esm == null)
          throw new Exception("No secure connection avaiable with endpoint " +
                              cm.getSource());
        cm.decrypt(esm.getCipherDec());
        FileSizePullErrorControlMessage fsperrcm = (FileSizePullErrorControlMessage)
            cm;
        this.processFileSizePullErrorControlMessage(fsperrcm);
      }
      else
        throw new Exception("Unexpected Control Message");
    }
    catch (Exception e) {
      _logger.error("",e);
      throw new Exception(this.getId() +
                          ": Error In Processing ControlMessage: id = " +
                          cm.getAck_Id() + " Source: " + cm.getSource() +
                          " Dest: " + cm.getDest());
    }
  }

  private void processFileSizePullErrorControlMessage(FileSizePullErrorControlMessage fsperrcm) throws Exception {
    if (fsperrcm.getControlId().intValue() == 0) {
      _logger.info("\n" + this.getId() +
                         ": File size pull aborted from id " +
                         fsperrcm.getSource() +
                         ". File hash = " + Base16.toHexString(fsperrcm.getContent()) +
                         ".\nReason: " + fsperrcm.getMessage());
      this.propertyChangeSupport.firePropertyChange("fileSizePullError",null,fsperrcm);
    }
  }

  private void processFileTransferErrorControlMessage(FileTransferErrorControlMessage fterrcm) throws Exception {
    if (fterrcm.getControlId().intValue() == 0) {
      synchronized (this.pendingFilePullRequests) {
        for (int x = this.pendingFilePullRequests.size() - 1; x >= 0; x--) {
          FilePullMessage fpr = (FilePullMessage) (this.pendingFilePullRequests.
              get(x));
          if (fpr.getDest().equals(fterrcm.getSource()) &&
              compareHash(fpr.getHash(), fterrcm.getContent())) {
            this.pendingFilePullRequests.remove(x);
            _logger.info("\n" + this.getId() +
                               ": File transfer aborted from id " +
                               fterrcm.getSource() +
                               ". File hash = " + Base16.toHexString(fterrcm.getContent()) +
                               ".\nReason: " + fterrcm.getMessage());
            this.propertyChangeSupport.firePropertyChange("filePartDownloadError",null,fterrcm);
          }
        }
      }
    }
    else if (fterrcm.getControlId().intValue() == 1) {
      synchronized (this.inServiceFilePushRequests) {
        for (int x = this.inServiceFilePushRequests.size() - 1; x >= 0; x--) {
          FilePullMessage fpr = (FilePullMessage) (this.inServiceFilePushRequests.
              get(x));
          if (fpr.getDest().equals(fterrcm.getSource()) &&
              compareHash(fpr.getHash(), fterrcm.getContent())) {
            this.inServiceFilePushRequests.remove(x);
            _logger.info("\n" + this.getId() +
                               ": File transfer aborted from id " +
                               fterrcm.getSource() +
                               ". File hash = " + Base16.toHexString(fterrcm.getContent()) +
                               ".\nReason: " + fterrcm.getMessage());
            this.propertyChangeSupport.firePropertyChange("filePartDownloadError",null,fterrcm);
          }
        }
      }
    }
  }

  private void processFileTransferEndControlMessage(FileTransferEndControlMessage ftecm) throws Exception {
    if (ftecm.getControlId().intValue() == 0) {
      synchronized (this.inServiceFilePushRequests) {
        for (int x = this.inServiceFilePushRequests.size() - 1; x >= 0; x--) {
          FilePullMessage fpr = (FilePullMessage) (this.
              inServiceFilePushRequests.get(x));
          if (fpr.getDest().equals(ftecm.getSource()) &&
              compareHash(fpr.getHash(), ftecm.getContent())) {
            this.inServiceFilePushRequests.remove(x);
            _logger.info("\n" + this.getId() +
                               ": File transfer succeded from id " +
                               ftecm.getSource() +
                               "\nFile hash = " + Base16.toHexString(ftecm.getContent()) +
                               "\nFile local name = "+ftecm.getFilePushMessage().getFileName());
          }
        }
      }

      byte[] hash = ftecm.getContent();
      PartialFile part = this.getPartialFile(hash, ftecm.getFilePushMessage());
      if (part != null) {
        part.finalizeFile();
        this.partialFiles.remove(part);
      }else{
        _logger.warn("Partial null file hash:" + Base16.toHexString(hash) +
                            " " + ftecm.getFilePushMessage().getAck_Id());
      }
      this.propertyChangeSupport.firePropertyChange("filePartDownloadCompleted",null,ftecm.getFilePushMessage());
    }else if (ftecm.getControlId().intValue() == 1){
      synchronized (this.inServiceFilePushRequests) {
        for (int x = this.inServiceFilePushRequests.size() - 1; x >= 0; x--) {
          FilePullMessage fpr = (FilePullMessage) (this.
              inServiceFilePushRequests.get(x));
          if (fpr.getDest().equals(ftecm.getSource()) &&
              compareHash(fpr.getHash(), ftecm.getContent())) {
            this.inServiceFilePushRequests.remove(x);
            _logger.info("\n" + this.getId() +
                               ": File transfer aborted from id " +
                               ftecm.getSource() +
                               ". File hash = " + Base16.toHexString(ftecm.getContent()) +
                               ".\nReason: " + ftecm.getMessage());
            byte[] hash = ftecm.getContent();
            PartialFile part = this.getPartialFile(hash,
                ftecm.getFilePushMessage());
            if (part != null) {
              this.partialFiles.remove(part);
            }
            this.propertyChangeSupport.firePropertyChange("filePartDownloadInterrupted",null,ftecm.getFilePushMessage());
          }
        }
      }
    }
  }

  private void processSecurityRequestMessage(SecurityRequestMessage srm) throws Exception{
    _logger.info("Input secure connections: "+this.inputSecureConnections.size());
    for (int x = 0; x < this.inputSecureConnections.size(); x++) {
      EndpointSecurityManager esmTemp = (EndpointSecurityManager)this.
          inputSecureConnections.get(x);
      _logger.info(esmTemp.getPeerId() + " " +
                         (new Date(esmTemp.getLastTimeUsed())).toGMTString());
    }
    _logger.info("End of input secure connection list (last time used order)");
    if(this.inputSecureConnections.size()>=maxSecureConnections){
      EndpointSecurityManager esm = (EndpointSecurityManager)this.inputSecureConnections.get(this.inputSecureConnections.size()-1);
      if(System.currentTimeMillis() - esm.getLastTimeUsed() > Ant.messageTimeout * Ant.maxRetransmissions){
        this.inputSecureConnections.remove(this.inputSecureConnections.size() -
                                           1);
      }
      else{
        return;
      }
    }
    ants.p2p.security.EndpointSecurityManager esm = new ants.p2p.security.EndpointSecurityManager(this, srm);
  }

  private void processSecurityResponseMessage(SecurityResponseMessage srm) throws Exception{
    ants.p2p.security.EndpointSecurityManager esm;
    boolean found = false;
    for (int x = this.outputSecureConnections.size() - 1; x >= 0; x--) {
      esm = (EndpointSecurityManager) this.outputSecureConnections.get(x);
      if (esm.getPeerId().equals(srm.getSource()) && esm.getSecurityMessage().equals(srm.getRequestMessage())) {
        found = true;
      }
      if (found)
        return;
    }

    esm = this.getPendingSecureConnectionRequest(srm.getSource(), srm);
    if(esm!=null){
        esm.processSecurityResponseMessage(srm);
        _logger.info(this.getId() + ": Secure connection created");
        this.propertyChangeSupport.firePropertyChange("secureConnectionCreated",null,srm);
        this.checkScheduledDownloads(esm);
    }
  }

  PartialFile getPartialFile(byte[] hash, FilePushMessage fpm){
    for(int x = 0; x < this.partialFiles.size(); x++){
      if(compareHash(((PartialFile)this.partialFiles.get(x)).hash,hash) &&
         ((PartialFile)this.partialFiles.get(x)).getFilePushMessage().equals(fpm)){
        return (PartialFile)this.partialFiles.get(x);
      }
    }
    return null;
  }

  public EndpointSecurityManager getOutputSecureConnectionManager(String peerId){
    for(int x=0; x < this.outputSecureConnections.size(); x++){
      if(((EndpointSecurityManager)this.outputSecureConnections.get(x)).getPeerId().equals(peerId))
        return (EndpointSecurityManager)this.outputSecureConnections.get(x);
    }
    return null;
  }

  public EndpointSecurityManager getInputSecureConnectionManager(String peerId) {
    for (int x = 0; x < this.inputSecureConnections.size(); x++) {
      if ( ( (EndpointSecurityManager)this.inputSecureConnections.get(x)).
          getPeerId().equals(peerId)){
       EndpointSecurityManager esm = (EndpointSecurityManager)this.inputSecureConnections.get(x);
       return (EndpointSecurityManager)this.inputSecureConnections.get(x);
     }
    }
    return null;
  }

  public EndpointSecurityManager getPendingSecureConnectionRequest(String peerId, SecurityResponseMessage srm){
    for (int x = 0; x < this.pendingSecureRequest.size(); x++) {
      if ( ( (EndpointSecurityManager)this.pendingSecureRequest.get(x)).
          getPeerId().equals(peerId) &&
          ( (EndpointSecurityManager)this.pendingSecureRequest.get(x)).
          getSecurityMessage().equals(srm.getRequestMessage()))
        return (EndpointSecurityManager)this.pendingSecureRequest.get(x);
    }
    return null;
  }

  public static boolean compareHash(byte[] h1, byte[] h2){
    if(h1.length != h2.length)
      return false;
    for(int x=0; x < h1.length; x++){
      if(h1[x]!=h2[x])
        return false;
    }
    return true;
  }
}

