//******************************************************************
//******************************************************************
//**********          ANts Peer To Peer Sources        *************
//
// ANts P2P realizes a third generation P2P net. It protects your
// privacy while you are connected and makes you not trackable, hiding
// your identity (ip) and crypting everything you are sending/receiving
// from others.

// Copyright (C) 2004  Roberto Rossi

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

package ants.p2p;

import java.net.*;
import java.io.*;
import java.util.zip.*;
import javax.crypto.*;

import ants.p2p.messages.*;

import org.apache.log4j.*;

import com.jcraft.jzlib.*;

/**
 * Proxy class that models a gateway to a neighbour. It manage marshalling
 * and unmarshalling, encryption and dectryption of data.
 */
public class NeighbourAnt extends Thread{
  Ant local;
  String ip;
  int port;
  int remoteServerPort;
  Cipher enc;
  Cipher dec;
  Socket s;
  boolean failure = false;
  boolean terminate = false;
  boolean isRequirer;

  public static double totalCompressedSizeOut;
  public static double totalUncompressionSizeOut;
  public static double totalCompressedSizeIn;
  public static double totalUncompressionSizeIn;

  static Logger _logger = Logger.getLogger(NeighbourAnt.class.getName());

  public NeighbourAnt(Ant local, String ip, int port, int remoteServerPort, Cipher enc,
                       Cipher dec, Socket s, boolean isRequirer) throws IOException {
    this.s = s;
    this.ip = ip;
    this.port = port;
    this.remoteServerPort = remoteServerPort;
    this.enc = enc;
    this.dec = dec;
    this.local = local;
    this.isRequirer=isRequirer;

    this.start();
  }

  public boolean isRequirer(){
    return this.isRequirer;
  }

  public void terminate(){
    try{
      this.terminate = true;
      if(s != null)
        this.s.close();
      //this.stop();
    }catch(Exception ex){_logger.error("",ex);}
  }

  public String getId(){
    if(ip.startsWith("/"))
      ip = ip.substring(1);
    return ip+":"+port;
  }

  public String getRemoteId() {
    if(ip.startsWith("/"))
      ip = ip.substring(1);
    return ip+":"+this.remoteServerPort;
  }


  public boolean equals(Object o){
    if(o instanceof NeighbourAnt){
      String idThis;
      String idO;
      if(this.isRequirer())
        idThis = this.getId();
      else
        idThis = this.getRemoteId();
      if(((NeighbourAnt)o).isRequirer())
        idO = ((NeighbourAnt)o).getId();
      else
        idO = ((NeighbourAnt)o).getRemoteId();
      return idThis.equals(idO);
    }
    else
      return o==this;
  }

  public SenderThread route(Message m){
    SenderThread st = new SenderThread(m, this);
    st.start();
    return st;
  }

  public synchronized void send(Message m) throws Exception {
    try {
      SealedObject sealedMessage;
      synchronized (this.enc) {
        _logger.debug(this.getId() + ": Routing message: ID "+m.getAck_Id()+  " Type: " + m.getType() +" From: " +" To: "+m.getDest());
        sealedMessage = new SealedObject(m, this.enc);
      }
      synchronized (this.s) {
        OutputStream sos = this.s.getOutputStream();

        ByteArrayOutputStream byteArrayOut = new ByteArrayOutputStream();
        ZOutputStream zOut = new ZOutputStream(byteArrayOut, JZlib.Z_BEST_COMPRESSION);
        ObjectOutputStream zOOs = new ObjectOutputStream(zOut);
        zOOs.writeObject(sealedMessage);

        NeighbourAnt.totalCompressedSizeOut += zOut.getTotalOut();
        NeighbourAnt.totalUncompressionSizeOut += zOut.getTotalIn();

        zOut.close();

        CompressedByteArray cba = new CompressedByteArray(byteArrayOut.toByteArray());
        ObjectOutputStream oos = new ObjectOutputStream(sos);
        oos.writeObject(cba);

        local.propertyChangeSupport.firePropertyChange("updatePacketCompressionStats", null, null);
      }
    }
    catch (Exception e) {
      e.printStackTrace();
      if (!failure) {
        failure = true;
        local.propertyChangeSupport.firePropertyChange("removedNeighbour",null,this);
        _logger.info(this.getId() +
                           ": Closed connection with neighbour: " +
                           this.getId());
        if(s!=null)
          s.close();
        local.removeNeighbour(this);
      }
    }
  }

  public void run() {
    try {
      while (!terminate) {
        Message m = null;
        _logger.debug(this.getId() + ": Waiting for message to route...");
        ObjectInputStream ois;
        Object obj;

        ois = new ObjectInputStream(this.s.getInputStream());
        obj = ois.readObject();

        if (obj instanceof CompressedByteArray) {
          CompressedByteArray cba = (CompressedByteArray)obj;
          ByteArrayInputStream in = new ByteArrayInputStream(cba.getArray());
          ZInputStream zIn = new ZInputStream(in);
          ObjectInputStream objIn = new ObjectInputStream(zIn);
          obj = objIn.readObject();

          NeighbourAnt.totalCompressedSizeIn += zIn.getTotalIn();
          NeighbourAnt.totalUncompressionSizeIn += zIn.getTotalOut();
        }else{
          _logger.debug(obj.toString());
          continue;
        }

        local.propertyChangeSupport.firePropertyChange("updatePacketCompressionStats", null, null);

        if (obj instanceof SealedObject) {
          obj = ( (SealedObject) obj).getObject(dec);
          if (obj instanceof Message) {
            m = (Message) obj;
            _logger.debug(this.getId() +
                                 ": Message received ID "+m.getAck_Id()+" From: "+m.getSource()+" To: "+m.getDest());
            local.activateNewRouterProcess(m, this.getId());
          }
          else
          _logger.debug(obj.toString());
        }
      }
      _logger.info(local.getId() +
                           ": 1Closed connection with neighbour: " +
                           this.getId());
      local.propertyChangeSupport.firePropertyChange("removedNeighbour", null, this);
    }
    catch (Exception e) {
      e.printStackTrace();
      if (!failure) {
        try{
          sleep(1000);
        }catch (Exception ex) {}
        failure = true;
        _logger.info(local.getId() +
                           ": 2Closed connection with neighbour: " +
                           this.getId());
        _logger.debug("NeighbourAnt connection failed",e);
        try{
          if (s != null)
            s.close();
        }catch(IOException ex){}
        local.removeNeighbour(this);
        local.propertyChangeSupport.firePropertyChange("removedNeighbour", null, this);
      }
    }
  }

  public String toString(){
    return this.getId();
  }

  public static int getTotalCompressionGainOut(){
    return (int)Math.ceil(((NeighbourAnt.totalUncompressionSizeOut / NeighbourAnt.totalCompressedSizeOut)-1)*100);
  }

  public static int getTotalCompressionGainIn(){
    return (int)Math.ceil(((NeighbourAnt.totalUncompressionSizeIn / NeighbourAnt.totalCompressedSizeIn)-1)*100);
  }
}

class CompressedByteArray implements Serializable{
  byte[] array;

  public CompressedByteArray(byte[] array){
    this.array = array;
  }

  public byte[] getArray(){
    return this.array;
  }
}