//******************************************************************
//******************************************************************
//**********          ANts Peer To Peer Sources        *************
//
// ANts P2P realizes a third generation P2P net. It protects your
// privacy while you are connected and makes you not trackable, hiding
// your identity (ip) and crypting everything you are sending/receiving
// from others.

// Copyright (C) 2004  Roberto Rossi

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

package ants.p2p;

import java.util.*;
import java.beans.*;
import java.io.*;

import ants.p2p.messages.*;
import ants.p2p.utils.*;
import ants.p2p.query.*;

import org.apache.log4j.*;

/**
 * This thread manages a download from multiple sources (net nodes).
 */
public class MultipleSourcesDownloadManager extends Thread{

  static Logger _logger = Logger.getLogger(MultipleSourcesDownloadManager.class.getName());

  MultipleSourcesDownloadManager instance = null;

  AutoresumeEngine ae;

  String fileName;
  String fileHash;
  long bytesDownloaded;
  long initialBytesDownloaded;
  long fileSize;
  int blockSize;
  java.util.List peerList = Collections.synchronizedList(new ArrayList());
  java.util.List inServicepeerList = Collections.synchronizedList(new ArrayList());
  WarriorAnt n;
  boolean interrupt = false;
  boolean[] downloadedBlockGroups;
  boolean[] inServiceBlockGroups;
  int[] pullErrorsPerBlock;
  int partId;
  DownloadWatchDog watchDog = new DownloadWatchDog(this);
  long assignedCoprime;

  long startTime;
  long lastPartCompletedTime;

  public static int blocksPerSource = 10;
  public static int MaxSources = 10;

  public PropertyChangeSupport propertyChangeSupport;

  public MultipleSourcesDownloadManager(WarriorAnt n, String fileName, String fileHash, long bytesDownloaded, int blockSize, long fileSize) throws Exception{
    instance = this;
    this.fileName=fileName;
    this.fileHash=fileHash;
    this.bytesDownloaded=bytesDownloaded;
    this.initialBytesDownloaded=bytesDownloaded;
    this.fileSize=fileSize;
    this.blockSize=blockSize;
    this.n=n;
    this.setPriority(Thread.MIN_PRIORITY);
    startTime = System.currentTimeMillis();
    lastPartCompletedTime = System.currentTimeMillis();
    downloadedBlockGroups=new boolean[(int)Math.ceil(fileSize/((blockSize*blocksPerSource)*1.0))];
    inServiceBlockGroups=new boolean[downloadedBlockGroups.length];
    pullErrorsPerBlock=new int[downloadedBlockGroups.length];

    partId=0;

    this.assignedCoprime = (long)Math.floor(Math.random()*downloadedBlockGroups.length);
    while(!((new java.math.BigInteger(this.assignedCoprime+"")).gcd(new java.math.BigInteger(downloadedBlockGroups.length+"")).equals(java.math.BigInteger.ONE))){
      this.assignedCoprime++;
      if(this.assignedCoprime==downloadedBlockGroups.length){
        this.assignedCoprime = (long)Math.floor(Math.random()*downloadedBlockGroups.length);
      }
    }

    Arrays.fill(downloadedBlockGroups,false);
    Arrays.fill(inServiceBlockGroups,false);
    Arrays.fill(pullErrorsPerBlock,0);
    this.propertyChangeSupport = new PropertyChangeSupport(this);

    this.propertyChangeSupport.addPropertyChangeListener(
        new PropertyChangeListener() {
      public void propertyChange(PropertyChangeEvent e) {
        if (e.getPropertyName().equals("peerAvaiable")) {
          resume();
        }else if (e.getPropertyName().equals("multipleDownloadCompleted")){
          BackgroundEngine.getInstance().removePartialFile(getFileHash());
          deactivateAutoresumeEngine();
        }
      }
    }
    );

    n.getPropertyChangeSupport().addPropertyChangeListener(
        new PropertyChangeListener() {
      public void propertyChange(PropertyChangeEvent e) {
        if (e.getPropertyName().equals("filePartDownloadCompleted")) {
          if(((FilePushMessage)e.getNewValue()).getFileName().lastIndexOf(getFileHash())>=0){
            String peerId = ( (FilePushMessage) e.getNewValue()).getSource();
            String suffix = ( (FilePushMessage) e.getNewValue()).getFileName();
            int index = suffix.lastIndexOf(".");
            suffix = suffix.substring(index + 1, suffix.length());
            int id = Integer.parseInt(suffix);
            synchronized (downloadedBlockGroups) {
              synchronized (inServiceBlockGroups) {
                synchronized (inServicepeerList) {
                  synchronized (peerList) {
                    watchDog.setReceivedPart();
                    downloadedBlockGroups[id] = true;
                    inServiceBlockGroups[id] = false;
                    QueryFileTuple peer = findInServicePeer(peerId);
                    if(peer != null){
                      inServicepeerList.remove(peer);
                      peerList.add(peer);
                    }
                    resume();
                  }
                }
              }
            }
            lastPartCompletedTime = System.currentTimeMillis();
          }
        }
        if (e.getPropertyName().equals("filePartDownloadError")) {
          if ( ( (FileTransferErrorControlMessage) e.getNewValue()).
              getFilePullMessage().getLocalFileName().lastIndexOf(getFileHash())>=0){
            String peerId = ( (FileTransferErrorControlMessage) e.getNewValue()).
                getSource();
            String suffix = ( (FileTransferErrorControlMessage) e.getNewValue()).
                getFilePullMessage().getLocalFileName();
            int index = suffix.lastIndexOf(".");
            suffix = suffix.substring(index + 1, suffix.length());
            int id = Integer.parseInt(suffix);
            synchronized (inServiceBlockGroups) {
              synchronized (inServicepeerList) {
                inServiceBlockGroups[id] = false;
                removePeer(peerId);
                propertyChangeSupport.firePropertyChange("removedSourcePeer", instance, peerId);
                resume();
              }
            }
          }
        }
        if (e.getPropertyName().equals("filePartDownloadInterrupted")) {
          if ( ( (FilePushMessage) e.getNewValue()).getFileName().lastIndexOf(getFileHash())>=0){
            String peerId = ( (FilePushMessage) e.getNewValue()).getSource();
            String suffix = ( (FilePushMessage) e.getNewValue()).getFileName();
            int index = suffix.lastIndexOf(".");
            suffix = suffix.substring(index + 1, suffix.length());
            int id = Integer.parseInt(suffix);
            synchronized (inServiceBlockGroups) {
              synchronized (inServicepeerList) {
                inServiceBlockGroups[id] = false;
                removePeer(peerId);
                propertyChangeSupport.firePropertyChange("removedSourcePeer", instance, peerId);
                resume();
              }
            }
          }
        }
        if (e.getPropertyName().equals("filePullError")) {
          if ( ( (FilePullMessage) e.getNewValue()).getLocalFileName().lastIndexOf(getFileHash())>=0){
            String peerId = ( (FilePullMessage) e.getNewValue()).
                getSource();
            String suffix = ( (FilePullMessage) e.getNewValue()).
                getLocalFileName();
            int index = suffix.lastIndexOf(".");
            suffix = suffix.substring(index + 1, suffix.length());
            int id = Integer.parseInt(suffix);
            synchronized (inServiceBlockGroups) {
              synchronized (inServicepeerList) {
                inServiceBlockGroups[id] = false;
                pullErrorsPerBlock[id] += 1;
                QueryFileTuple peer = findInServicePeer(peerId);
                if (peer != null) {
                  inServicepeerList.remove(peer);
                  peerList.add(peer);
                }
                if(pullErrorsPerBlock[id] >= Ant.maxRetransmissions){
                  removePeer(peerId);
                  propertyChangeSupport.firePropertyChange("removedSourcePeer",
                      instance, peerId);
                }
                resume();
              }
            }
          }
        }
        if (e.getPropertyName().equals("filePushError")) {
          if ( ( (FilePushMessage) e.getNewValue()).getFileName().lastIndexOf(getFileHash())>=0){
            String peerId = ( (FilePushMessage) e.getNewValue()).getSource();
            String suffix = ( (FilePushMessage) e.getNewValue()).getFileName();
            int index = suffix.lastIndexOf(".");
            suffix = suffix.substring(index + 1, suffix.length());
            int id = Integer.parseInt(suffix);
            synchronized (inServiceBlockGroups) {
              synchronized (inServicepeerList) {
                inServiceBlockGroups[id] = false;
                removePeer(peerId);
                propertyChangeSupport.firePropertyChange("removedSourcePeer", instance, peerId);
                resume();
              }
            }
          }
        }
        if (e.getPropertyName().equals("filePartError")) {
          if ( ( (FilePartMessage) e.getNewValue()).getFilePushMessage().getFileName().lastIndexOf(getFileHash())>=0){
            String peerId = ( (FilePartMessage) e.getNewValue()).getSource();
            String suffix = ( (FilePartMessage) e.getNewValue()).
                getFilePushMessage().getFileName();
            int index = suffix.lastIndexOf(".");
            suffix = suffix.substring(index + 1, suffix.length());
            int id = Integer.parseInt(suffix);
            synchronized (inServiceBlockGroups) {
              synchronized (inServicepeerList) {
                inServiceBlockGroups[id] = false;
                removePeer(peerId);
                propertyChangeSupport.firePropertyChange("removedSourcePeer", instance, peerId);
                resume();
              }
            }
          }
        }
        if (e.getPropertyName().equals("securityConnectionError")) {
          _logger.error("Critical Security Error: stopping downloading...");
        }
      }
    });
    WarriorAnt.checkChunksPath();
    File store = new File(WarriorAnt.chunksPath + fileHash + ".mul");
    if(store.exists()){
      ObjectInputStream ois = new ObjectInputStream(new FileInputStream(
          store));
      downloadedBlockGroups = (boolean[])ois.readObject();
      this.fileName = (String)ois.readObject();
      this.fileHash = (String)ois.readObject();
      this.bytesDownloaded = ((Long)ois.readObject()).longValue();
      this.initialBytesDownloaded=bytesDownloaded;
      this.fileSize = ((Long)ois.readObject()).longValue();
      this.blockSize = ((Integer)ois.readObject()).intValue();
      this.assignedCoprime = ((Long)ois.readObject()).longValue();
      ois.close();

      this.checkDownloadedBlockGroups();
    }else{
      this.checkDownloadedBlockGroups();

      ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(
          store));
      oos.writeObject(downloadedBlockGroups);
      oos.writeObject(fileName);
      oos.writeObject(fileHash);
      oos.writeObject(new Long(bytesDownloaded));
      oos.writeObject(new Long(fileSize));
      oos.writeObject(new Integer(blockSize));
      oos.writeObject(new Long(assignedCoprime));
      oos.flush();
      oos.close();
    }

    BackgroundEngine.getInstance().addPartialFile(this);
  }

  public int getPeersNumber(){
    return this.peerList.size()+this.inServicepeerList.size();
  }

  /**
   * Adds a peer to download from
   * @param peerId
   * @return
   */
  public boolean addPeer(QueryFileTuple peerId){
    //Solo se c' una connessione sicura
    synchronized(this.inServicepeerList){
      synchronized(this.peerList){
        if(peerId instanceof QueryCompletedFileTuple){
          QueryCompletedFileTuple qftPeerId = (QueryCompletedFileTuple) peerId;
          if (n.getOutputSecureConnectionManager(qftPeerId.getOwnerID()) != null) {
            if (!this.peerList.contains(qftPeerId) &&
                !this.inServicepeerList.contains(qftPeerId)) {
              this.peerList.add(qftPeerId);
              this.propertyChangeSupport.firePropertyChange("peerAvaiable", null, null);
              return true;
            }
          }
        }else if(peerId instanceof QueryPartialFileTuple){
          QueryPartialFileTuple qpftPeerId = (QueryPartialFileTuple) peerId;
          if (n.getOutputSecureConnectionManager(qpftPeerId.getOwnerID()) != null) {
            if (!this.peerList.contains(qpftPeerId) &&
                !this.inServicepeerList.contains(qpftPeerId)) {
              if(this.getBlockSize() == qpftPeerId.getBlockSize().intValue() &&
                 blocksPerSource == qpftPeerId.getBlocksPerSource().intValue()){
                this.peerList.add(qpftPeerId);
                this.propertyChangeSupport.firePropertyChange("peerAvaiable", null, null);
                return true;
              }
            }
          }
        }
      }
    }
    return false;
  }

  public Object removePeer(String peerID) {
    //Solo se c' una connessione sicura
    synchronized (this.inServicepeerList) {
      synchronized (this.peerList) {
        QueryFileTuple peer = this.findPeer(peerID);
        if (peer != null) {
          this.peerList.remove(peer);
          this.propertyChangeSupport.firePropertyChange("peerRemoved", null, null);
          return peer;
        }
        QueryFileTuple ispeer = this.findInServicePeer(peerID);
        if (ispeer != null) {
          this.inServicepeerList.remove(ispeer);
          this.propertyChangeSupport.firePropertyChange("peerRemoved", null, null);
          return ispeer;
        }
      }
    }
    return null;
  }

  public void setInterrupt(boolean deactivateAutoresume){
    if (isAutoresumeActive() && deactivateAutoresume) {
      deactivateAutoresumeEngine();
    }
    this.interrupt=true;
    this.resume();
  }

  public void resetInterrupt(){
    this.interrupt=false;
    this.resume();
  }

  public void run(){
    try{
      WarriorAnt.checkChunksPath();
      this.watchDog.start();
      _logger.info("Current sources number: " + this.peerList.size());
      partId = 0;
      while(!this.chechCompleted()){
        if(this.peerList.size()==0 || this.inServicepeerList.size() == MaxSources){// && this.inServicepeerList.size()==0){
          this.propertyChangeSupport.firePropertyChange("multipleDownloadSuspended", null, this);
          suspend();
          //sleep(5000);
        }
        /*
        if(this.peerList.size()==0 || this.inServicepeerList.size() == 0){
          interrupt = true;
          this.propertyChangeSupport.firePropertyChange("multipleDownloadStalled", null, this);
        }
        */
        if(interrupt){
          while(this.inServicepeerList.size() > 0){
            Thread.sleep(1000);
          }
          File store = new File(WarriorAnt.chunksPath + fileHash + ".mul");
          ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(
              store));
          oos.writeObject(downloadedBlockGroups);
          oos.writeObject(fileName);
          oos.writeObject(fileHash);
          int num = 0;
          for (int x = 0; x < this.downloadedBlockGroups.length; x++){
            if (this.downloadedBlockGroups[x] == true) {
              num++;
            }
          }
          this.bytesDownloaded = num * (blockSize * blocksPerSource);
          oos.writeObject(new Long(bytesDownloaded));
          oos.writeObject(new Long(fileSize));
          oos.writeObject(new Integer(blockSize));
          oos.writeObject(new Long(assignedCoprime));
          oos.close();
          this.watchDog.stop();
          this.stop();
        }
        synchronized (downloadedBlockGroups) {
          synchronized (inServiceBlockGroups) {
            synchronized (inServicepeerList) {
              synchronized (peerList) {
                for(int t = this.peerList.size() - 1; t >= 0 && this.inServicepeerList.size() < MaxSources && !this.chechCompleted(); t--){
                  //this.peerList.trimToSize();
                  QueryFileTuple peerTuple = (QueryFileTuple)this.peerList.get(t);
                  if(peerTuple instanceof QueryCompletedFileTuple){
                    QueryCompletedFileTuple peerFileTuple = (QueryCompletedFileTuple)peerTuple;
                    String peerId = peerFileTuple.getOwnerID();
                    boolean requiredPart = false;
                    /*for (partId = 0;
                         partId != (downloadedBlockGroups.length - this.assignedPrime - 1) && !requiredPart;
                         partId = (int)((partId + this.assignedPrime) % downloadedBlockGroups.length)) {*/
                    partId = 0;
                    do{
                      if (downloadedBlockGroups[partId] == false &&
                          inServiceBlockGroups[partId] == false) {
                        this.peerList.remove(t);
                        //this.peerList.trimToSize();
                        long curOffset = partId * (blockSize * blocksPerSource);
                        File resume = new File(WarriorAnt.chunksPath + fileHash +
                                               "." + partId);
                        if (resume.exists()) {
                          resume.delete();
                        }
                        _logger.info("Pulling file " + fileName +
                                     " from id " +
                                     peerId + " - " + fileHash + "." + partId);
                        _logger.info(fileName + " " + blocksPerSource + " " +
                                     blockSize);
                        requiredPart = true;
                        n.pullFile(fileName, fileHash, curOffset,
                                   blocksPerSource,
                                   peerId,
                                   blockSize,
                                   WarriorAnt.chunksPath + fileHash + "." +
                                   partId, false);
                        this.inServicepeerList.add(peerTuple);
                        inServiceBlockGroups[partId] = true;
                      }
                      partId = (int)((partId + this.assignedCoprime) % downloadedBlockGroups.length);
                    }while(partId != 0 && !requiredPart);
                  }else if(peerTuple instanceof QueryPartialFileTuple){
                    QueryPartialFileTuple peerPartialFileTuple = (QueryPartialFileTuple)peerTuple;
                    String peerId = peerPartialFileTuple.getOwnerID();
                    boolean requiredPart = false;
                    if(!this.checkPartsToDownload(downloadedBlockGroups, inServiceBlockGroups, peerPartialFileTuple.getDownloadedBlockGroups())){
                      this.removePeer(peerId);
                      propertyChangeSupport.firePropertyChange("removedSourcePeer", instance, peerId);
                      continue;
                    }
                    partId = 0;
                    do{
                      if (downloadedBlockGroups[partId] == false &&
                          inServiceBlockGroups[partId] == false &&
                          peerPartialFileTuple.getDownloadedBlockGroups()[partId]==true) {
                        this.peerList.remove(t);
                        //this.peerList.trimToSize();
                        long curOffset = partId * (blockSize * blocksPerSource);
                        File resume = new File(WarriorAnt.chunksPath + fileHash +
                                               "." + partId);
                        if (resume.exists()) {
                          resume.delete();
                        }
                        _logger.info("Pulling file " + fileName +
                                     " from id " +
                                     peerId + " - " + fileHash + "." + partId);
                        _logger.info(fileName + " " + blocksPerSource + " " +
                                     blockSize);
                        requiredPart = true;
                        n.pullFile(fileName, fileHash, curOffset,
                                   blocksPerSource,
                                   peerId,
                                   blockSize,
                                   WarriorAnt.chunksPath + fileHash + "." +
                                   partId, false);
                        this.inServicepeerList.add(peerTuple);
                        inServiceBlockGroups[partId] = true;
                      }
                      partId = (int)((partId + this.assignedCoprime) % downloadedBlockGroups.length);
                    }while(partId != 0 && !requiredPart);
                  }
                }
              }
            }
          }
        }
        checkDownloadedBlockGroups();
        File store = new File(WarriorAnt.chunksPath + fileHash + ".mul");
        ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(
            store));
        oos.writeObject(downloadedBlockGroups);
        oos.writeObject(fileName);
        oos.writeObject(fileHash);
        int num = 0;
        for (int x = 0; x < this.downloadedBlockGroups.length; x++) {
          if (this.downloadedBlockGroups[x] == true) {
            num++;
          }
        }
        this.bytesDownloaded = num * (blocksPerSource * blockSize);
        oos.writeObject(new Long(this.bytesDownloaded));
        oos.writeObject(new Long(fileSize));
        oos.writeObject(new Integer(blockSize));
        oos.writeObject(new Long(assignedCoprime));
        oos.close();
      }
      if(this.chechCompleted()){
        this.watchDog.stop();
        _logger.info("Download completed: " + this.fileHash);
        File complete = new File(WarriorAnt.downloadPath + (new File(this.fileName)).getName());
        FileOutputStream fos = new FileOutputStream(complete);
        for(int x = 0; x < downloadedBlockGroups.length; x++){
          File partial = new File(WarriorAnt.chunksPath + this.fileHash + "." + x);
          FileInputStream fis = new FileInputStream(partial);
          byte[] content = new byte[blockSize];
          while (fis.available() > 0) {
            int read = fis.read(content);
            if (read > 0) {
              if (read < content.length) {
                byte[] reduced = new byte[read];
                for (int g = 0; g < read; g++) {
                  reduced[g] = content[g];
                }
                content = reduced;
              }
              fos.write(content);
            }
          }
          fis.close();
          partial.delete();
        }
        fos.flush();
        fos.close();
        _logger.info("Written to disk: "+complete.getName());
        File data = new File(WarriorAnt.chunksPath + this.fileHash + ".mul");
        data.delete();
        this.propertyChangeSupport.firePropertyChange("multipleDownloadCompleted", null, this);
      }
    }catch(Exception e){
      _logger.error("",e);
    }
  }

  public int getBlockSize() {
    return this.blockSize;
  }

  public String getFileHash(){
    return this.fileHash;
  }

  public String getFileName(){
    return this.fileName;
  }

  public long getFileSize(){
    return this.fileSize;
  }

  public long getByteDownloaded(){
    return this.bytesDownloaded;
  }

  public boolean[] getDownloadedBlockGroups(){
    return this.downloadedBlockGroups;
  }

  public String getPercentage(){
    int den = this.downloadedBlockGroups.length;
    int num = 0;
    for(int x = 0; x < this.downloadedBlockGroups.length; x++)
      if(this.downloadedBlockGroups[x]==true)num++;
    double perc = (num*1.0/den)*100;
    return (int)perc+"%  ";
  }

  public String getSpeed(){
    double elapsed = (this.lastPartCompletedTime - this.startTime)/1000.0;
    int downloadedGroups = 0;
    for(int x = 0; x < this.downloadedBlockGroups.length; x++)
      if(this.downloadedBlockGroups[x]==true)
        downloadedGroups++;
    double speed;
    if(elapsed > 0){
      speed = ( ( (downloadedGroups * (blocksPerSource * blockSize)) -
                 this.initialBytesDownloaded) / elapsed) / Math.pow(2, 10);
    }
    else
      speed = 0;
    return (""+speed).substring(0,(""+speed).indexOf(".")+2)+"Kb/s ";
  }

  public String toString(){
    int max = 45;
    String result;
    if(!this.chechCompleted()){
      result = this.getPercentage();
    }else{
      result = "Completed ";
    }
    result += this.getSpeed();
    if (max > this.getFileName().length())
      max = this.getFileName().length();
    result +=
        this.getFileName().substring(0, max);
    while (result.length() < 60)
      result += ".";
    result += " File Size: " + this.sizeToString(this.getFileSize());
    while (result.length() < 85)
      result += ".";
    result += "File hash: " + this.getFileHash();
    return result;
  }

  public String sizeToString(double dSize){
    double currentSize = dSize;
    int x = 0;
    while (dSize > 1 && x < 4) {
      dSize = dSize / Math.pow(2, 20);
      x++;
    }
    String suffix = " bytes";
    String size = currentSize+suffix;
    if (x == 2) {
      suffix = " Mb";
      size = (new Double(currentSize * 1.0 / Math.pow(2, 20))).
          toString();
      int index = size.indexOf(".");
      if (index >= 0)
        size = size.substring(0, index + 2);
      size += suffix;
    }
    if (x == 3) {
      suffix = " Gb";
      size = (new Double(currentSize * 1.0 / Math.pow(2, 30))).
          toString();
      int index = size.indexOf(".");
      if (index >= 0)
        size = size.substring(0, index + 2);
      size += suffix;
    }
    return size;
  }

  public boolean checkPartsToDownload(boolean[] downloadedBlockGroups, boolean[] inServiceBlockGroups, boolean[] remoteParts) throws Exception{
    if(downloadedBlockGroups.length != remoteParts.length)
      throw new Exception("Array sizes do not match");
    for(int x = 0; x < downloadedBlockGroups.length; x++){
      if(downloadedBlockGroups[x] == false &&
         inServiceBlockGroups[x] == false &&
         remoteParts[x] == true)
        return true;
    }
    return false;
  }

  public void checkDownloadedBlockGroups(){
    for(int x = 0; x < downloadedBlockGroups.length; x++){
      File currentFile = new File(WarriorAnt.chunksPath + this.getFileHash() + "." + x);
      if(x != downloadedBlockGroups.length - 1){
        if (currentFile.exists() &&
            currentFile.length() == (blocksPerSource * blockSize)) {
          downloadedBlockGroups[x] = true;
        }
        else {
          downloadedBlockGroups[x] = false;
        }
      }else{
        if (currentFile.exists() &&
            (currentFile.length() == this.fileSize - ((downloadedBlockGroups.length - 1)*(blocksPerSource * blockSize)))) {
          downloadedBlockGroups[x] = true;
        }
        else {
          downloadedBlockGroups[x] = false;
        }
      }
    }
  }

  public QueryFileTuple findPeer(String peerId){
    for(int x = 0; x < this.peerList.size(); x++){
      if(((QueryFileTuple)this.peerList.get(x)).getOwnerID().equals(peerId))
        return (QueryFileTuple)this.peerList.get(x);
    }
    return null;
  }

  public QueryFileTuple findInServicePeer(String peerId){
    for(int x = 0; x < this.inServicepeerList.size(); x++){
      if(((QueryFileTuple)this.inServicepeerList.get(x)).getOwnerID().equals(peerId))
        return (QueryFileTuple)this.inServicepeerList.get(x);
    }
    return null;
  }

  /**
   * Adds a handler to manage autoresume of this download
   * @param ae
   */
  public void addAutoresumeEngine(AutoresumeEngine ae){
    if(this.ae != null && this.ae.isAlive())
      this.ae.deactivate();
    this.ae = ae;
    if (this.ae != null && !this.ae.isAlive())
      this.ae.start();
  }

  public AutoresumeEngine getAutoresumeEngine(){
    return this.ae;
  }

  public void deactivateAutoresumeEngine(){
    if(this.ae != null && this.ae.isAlive()){
      this.ae.deactivate();
      this.ae = null;
    }
  }

  public boolean isAutoresumeActive(){
    if(this.ae != null && this.ae.isAlive())
      return true;
    else
      return false;
  }

  public synchronized boolean chechCompleted(){
    boolean completed = true;
    for (int x = 0; x < downloadedBlockGroups.length;
         x++) {
      if (downloadedBlockGroups[x] == false) {
        completed = false;
      }
    }
    return completed;
  }
}


class DownloadWatchDog extends Thread{
  public static long downloadTimeout = 900000;
  long lastPartReceivedTime;
  MultipleSourcesDownloadManager msdm = null;

  static Logger _logger = Logger.getLogger(DownloadWatchDog.class.getName());

  public DownloadWatchDog(MultipleSourcesDownloadManager msdm){
    super();
    this.msdm = msdm;
    lastPartReceivedTime = System.currentTimeMillis();
    this.setPriority(Thread.MIN_PRIORITY);
  }

  public void setReceivedPart(){
    lastPartReceivedTime = System.currentTimeMillis();
  }

  public void run(){
    while (true) {
      try {
        sleep(5000);
      }
      catch (Exception e) {
        _logger.error("",e);
      }
      long elapsed = System.currentTimeMillis() - this.lastPartReceivedTime;
      if(elapsed > downloadTimeout){
        msdm.propertyChangeSupport.firePropertyChange("multipleDownloadTimedOut", null, msdm);
        return;
      }
    }
  }
}