//******************************************************************
//******************************************************************
//**********          ANts Peer To Peer Sources        *************
//
// ANts P2P realizes a third generation P2P net. It protects your
// privacy while you are connected and makes you not trackable, hiding
// your identity (ip) and crypting everything you are sending/receiving
// from others.

// Copyright (C) 2004  Roberto Rossi

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

package ants.p2p;

import java.io.*;
import java.util.*;
import java.beans.*;
import javax.crypto.*;
import ants.p2p.messages.*;
import ants.p2p.security.*;
import ants.p2p.query.*;
import ants.p2p.query.security.*;
import ants.p2p.utils.*;

import org.apache.log4j.*;

/**
 * Thread to serve a pull message request
 */
public class FilePullMessageProcessor extends Thread{
  PullWatchDog watchDog = new PullWatchDog(this);

  static Logger _logger = Logger.getLogger(FilePullMessageProcessor.class.getName());

  WarriorAnt n;
  FilePullMessage fpm;
  int percentage = 0;
  public FilePullMessageProcessor(WarriorAnt n, FilePullMessage fpm){
    this.n = n;
    this.fpm = fpm;
    this.setPriority(Thread.MIN_PRIORITY);
  }

  public FilePullMessage getFilePullMessage(){
    return this.fpm;
  }

  public boolean checkInServiceFilePullRequests(boolean addThis){
    synchronized (n.inServiceFilePullRequests) {
      if (n.inServiceFilePullRequests.contains(fpm)) {
        _logger.info(n.getId() +
                     ": Found in service file pull! Rejecting message");
        return true;
      }else{
        for(int x = 0; x < n.inServiceFilePullRequests.size(); x++){
          FilePullMessage currentFpM = (FilePullMessage)n.inServiceFilePullRequests.get(x);
          if(currentFpM != null &&
            currentFpM.getBlocks().equals(fpm.getBlocks()) &&
            currentFpM.getBlockSize().equals(fpm.getBlockSize()) &&
            currentFpM.getFileName().equals(fpm.getFileName()) &&
            WarriorAnt.compareHash(currentFpM.getHash(),fpm.getHash()) &&
            currentFpM.getLocalFileName().equals(fpm.getLocalFileName()) &&
            currentFpM.getOffset().equals(fpm.getOffset()) &&
            currentFpM.getResume().equals(fpm.getResume()) &&
            currentFpM.getSource().equals(fpm.getSource())){
           _logger.info(n.getId() +
                     ": Found in service file pull! Rejecting message");
           return true;
          }
        }
      }
      _logger.info(n.getId() +
                   ": Not Found in service file pull! Processing message");

      if(addThis)
        n.inServiceFilePullRequests.add(fpm);

      return false;
    }
  }

  public void run(){
    FileInputStream fis = null;
    watchDog.start();
    try {
      ants.p2p.security.EndpointSecurityManager esm = n.
          getInputSecureConnectionManager(fpm.getSource());
      if (esm == null){
        n.propertyChangeSupport.firePropertyChange("noSecureConnection",null,fpm);
        throw new Exception("No secure connection avaiable with endpoint " +
                            fpm.getSource());
      }

      fpm.decrypt(esm.getCipherDec());

      if(this.checkInServiceFilePullRequests(false)){
        watchDog.stop();
        return;
      }

      byte[] hash = fpm.getHash();
      Long offset = fpm.getOffset();
      File f = null;

      //controlla ultima modifica e caratteristiche del file
      BackgroundEngine be = BackgroundEngine.getInstance();
      boolean shared = false;
      boolean partial = false;
      Enumeration infos = be.sharedFiles.elements();
      Enumeration files = be.sharedFiles.keys();
      while(infos.hasMoreElements() && files.hasMoreElements()){
        String fileName = (String)files.nextElement();
        String fileHash = ((FileInfos)infos.nextElement()).getHash();
        String fpmFileName = fpm.getFileName();
        String fpmHash = Base16.toHexString(hash);
        if(fileName.indexOf(fpmFileName) >= 0 && fileHash.equals(fpmHash)){
          shared = true;
          f = new File(fileName);
          break;
        }
      }
      if(!shared || f == null || !f.exists()){
        String fpmHash = Base16.toHexString(hash);
        Enumeration partialFiles = BackgroundEngine.getInstance().getPartialFiles();
        while(partialFiles.hasMoreElements()){
          if(partialFiles.nextElement().equals(fpmHash)){
            shared = true;
            long partId = fpm.getOffset().longValue() / (fpm.getBlockSize().intValue() * fpm.getBlocks().longValue());
            f = new File(WarriorAnt.chunksPath + fpmHash + "." + partId);
            partial = true;
            break;
          }
        }
      }
      if(!shared || f == null || !f.exists()){
        FileTransferErrorControlMessage fterrcm = new FileTransferErrorControlMessage(new Integer(0),hash, "File don't exist", fpm);
        fterrcm.encrypt(esm.getCipherEnc());
        MessageWrapper wm = n.sendMessage(fterrcm,fpm.getSource(),false,false);
        _logger.debug(n.getId() + ": File " + fpm.getFileName() +
                             " don't exist. Cannot serve pull request from id " +
                             fpm.getSource());
        while (n.myMessages.contains(wm)) {
          Thread.sleep(1000);
        }
        if (n.failedMessages.contains(wm)) {
          n.failedMessages.remove(wm);
          fterrcm.decrypt(esm.getCipherDec());
          _logger.debug(n.getId() +
                               ": Error in sending ControlMessage " +
                               fterrcm.getMessage());
        }
        watchDog.stop();
        return;
      }

      n.propertyChangeSupport.firePropertyChange("filePullInit",null,this);
      fis = new FileInputStream(f);
      FilePushMessage frm = new FilePushMessage(fpm.getLocalFileName(),hash,offset,fpm.getBlockSize(), fpm.getResume(), new Long(f.length()));
      frm.encrypt(esm.getCipherEnc());
      MessageWrapper wm = n.sendMessage(frm,fpm.getSource(),false, false);
      while(n.myMessages.contains(wm)){
        Thread.sleep(1000);
      }
      if (n.failedMessages.contains(wm)) {
        n.failedMessages.remove(wm);
        frm.decrypt(esm.getCipherDec());
        _logger.debug(n.getId() +
                      ": Error in sending FilePushMessage " +
                      frm.getFileName() +
                      " - Cannot serve pull request from id " +
                      fpm.getSource());
        watchDog.stop();
        return;
      }
      watchDog.setReceivedPart();

      frm.decrypt(esm.getCipherDec());
      if(this.checkInServiceFilePullRequests(true)){
        n.propertyChangeSupport.firePropertyChange("filePullError",null,this.getFilePullMessage());
        watchDog.stop();
        return;
      }

      byte[] block = new byte[fpm.getBlockSize().intValue()];
      long blocks = fpm.getBlocks().longValue();
      if(!partial){
        fis.skip(offset.longValue());
      }
      long partId = 0;ArrayList messageGroup = new ArrayList();
      while(fis.available()>0 && partId < blocks){
        for(int g = 0; g < (PartialFile.computeGroupFactor(fpm.getBlockSize().intValue())-messageGroup.size()) && partId < blocks; g++){
          int read = fis.read(block);
          if(read > 0){
            if (read < block.length) {
              byte[] reduced = new byte[read];
              for (int x = 0; x < read; x++) {
                reduced[x] = block[x];
              }
              block = reduced;
            }
            FilePartMessage filepartmess = new FilePartMessage(block, hash, new Long(partId), frm);
            filepartmess.encrypt(esm.getCipherEnc());
            MessageWrapper wm1 = n.sendMessage(filepartmess, fpm.getSource(),false,false);
            partId++;
            this.percentage = (int)(((partId * 1.0)/blocks)*100.0);
            n.propertyChangeSupport.firePropertyChange("filePullUpdate",null,this);
            messageGroup.add(wm1);
          }
        }
        while(messageGroup.size() == PartialFile.computeGroupFactor(fpm.getBlockSize().intValue()) &&
              !n.areInFailedMessages(messageGroup)){
          messageGroup = n.removeDeliveredMessagesFromList(messageGroup);
          Thread.sleep(1000);
        }
        if (n.areInFailedMessages(messageGroup)) {
          n.failedMessages.removeAll(messageGroup);
          n.propertyChangeSupport.firePropertyChange("filePullError",null,this.getFilePullMessage());
          _logger.info(n.getId() + ": Error in file transfer " + fpm.getFileName() +
                             ". Cannot serve pull request from id " +
                             fpm.getSource());
          fis.close();

          n.inServiceFilePullRequests.remove(fpm);

          watchDog.stop();
          return;
        }
        watchDog.setReceivedPart();
      }
      while (messageGroup.size() > 0 &&
             !n.areInFailedMessages(messageGroup)) {
        messageGroup = n.removeDeliveredMessagesFromList(messageGroup);
        Thread.sleep(1000);
      }
      if (n.areInFailedMessages(messageGroup)) {
        n.failedMessages.removeAll(messageGroup);
        n.propertyChangeSupport.firePropertyChange("filePullError", null,
            this.getFilePullMessage());
        _logger.info(n.getId() + ": Error in file transfer " + fpm.getFileName() +
                     ". Cannot serve pull request from id " +
                     fpm.getSource());
        fis.close();

        n.inServiceFilePullRequests.remove(fpm);

        watchDog.stop();
        return;
      }
      watchDog.setReceivedPart();

      fis.close();
      FileTransferEndControlMessage cm = new FileTransferEndControlMessage(new Integer(0), hash, "", frm);
      cm.encrypt(esm.getCipherEnc());
      MessageWrapper wm2 = n.sendMessage(cm, fpm.getSource(),false,false);
      while(n.myMessages.contains(wm2)){
        Thread.sleep(1000);
      }
      if(n.failedMessages.contains(wm2)){
        n.failedMessages.remove(wm2);
        n.propertyChangeSupport.firePropertyChange("filePullError",null,this.getFilePullMessage());
        _logger.info(n.getId() + ": Error in file transfer " + fpm.getFileName() +
                             ". Cannot serve pull request from id " +
                             fpm.getSource());
        watchDog.stop();
        return;
      }
      watchDog.setReceivedPart();

      n.inServiceFilePullRequests.remove(fpm);

      n.propertyChangeSupport.firePropertyChange("filePullEnd",null,this.getFilePullMessage());
      _logger.info(n.getId() + ": File sent " + fpm.getFileName() + " - " + fpm.getLocalFileName() +
                             ". Pull request from id " +
                             fpm.getSource() + " served.");
      watchDog.stop();
    }
    catch (Exception e) {
      try {
        fis.close();
      }
      catch (Exception ex) {}

      n.propertyChangeSupport.firePropertyChange("filePullError", null,
                                                 this.getFilePullMessage());
      n.inServiceFilePullRequests.remove(fpm);
      watchDog.stop();

      _logger.error(n.getId() +
                    ": Error In Processing FilePullMessage: id = " +
                    fpm.getAck_Id() + " Source: " + fpm.getSource() +
                    " Dest: " + fpm.getDest(), e);
    }
  }

  public void terminate(){
    this.stop();
    this.watchDog.stop();
    try {
      ants.p2p.security.EndpointSecurityManager esm = n.
          getInputSecureConnectionManager(fpm.getSource());
      if (esm == null) {
        n.propertyChangeSupport.firePropertyChange("noSecureConnection", null,
            fpm);
        _logger.error("No secure connection avaiable with endpoint " +
                      fpm.getSource());
      }

      FileTransferErrorControlMessage cm = new
          FileTransferErrorControlMessage(new
                                          Integer(1), fpm.getHash(), "",
                                          fpm);
      cm.encrypt(esm.getCipherEnc());
      MessageWrapper wm2 = n.sendMessage(cm, fpm.getSource(), false, false);
      while (n.myMessages.contains(wm2)) {
        Thread.sleep(1000);
      }
      if (n.failedMessages.contains(wm2)) {
        n.failedMessages.remove(wm2);
        _logger.info(n.getId() + ": Error file transfer interruption " +
                     fpm.getFileName() +
                     ". Cannot notificate interruption of msg id " +
                     fpm.getSource());
        return;
      }
      _logger.warn(n.getId() +
                   ": File pull interrupted: id = " +
                   fpm.getAck_Id() + " Source: " + fpm.getSource() +
                   " Dest: " + fpm.getDest());
    }
    catch (Exception ex) {
      _logger.error(
          "Exception while trying to notify file transfer interruption", ex);
    }
  }

  public String toString(){
    return percentage+"%   Hash: "+Base16.toHexString(fpm.getHash())+"   Requirer: "+ fpm.getSource() +"    Name: "+fpm.getFileName();
  }
}

class PullWatchDog extends Thread{
  public static long downloadTimeout = 900000;
  long lastPartReceivedTime;
  FilePullMessageProcessor fpmp = null;

  static Logger _logger = Logger.getLogger(PullWatchDog.class.getName());

  public PullWatchDog(FilePullMessageProcessor fpmp){
    super();
    this.fpmp = fpmp;
    lastPartReceivedTime = System.currentTimeMillis();
    this.setPriority(Thread.MIN_PRIORITY);
  }

  public void setReceivedPart(){
    lastPartReceivedTime = System.currentTimeMillis();
  }

  public void run(){
    while (true) {
      try {
        sleep(5000);
      }
      catch (Exception e) {
        _logger.error("",e);
      }
      long elapsed = System.currentTimeMillis() - this.lastPartReceivedTime;
      if(elapsed > downloadTimeout){
        fpmp.stop();
        _logger.info("File pull message processor timed out");
        fpmp.n.propertyChangeSupport.firePropertyChange("filePullError",null,fpmp.getFilePullMessage());
        fpmp.n.inServiceFilePullRequests.remove(fpmp.fpm);
        return;
      }
    }
  }
}

