//******************************************************************
 //******************************************************************
  //**********          ANts Peer To Peer Sources        *************
   //
   // ANts P2P realizes a third generation P2P net. It protects your
   // privacy while you are connected and makes you not trackable, hiding
   // your identity (ip) and crypting everything you are sending/receiving
   // from others.

   // Copyright (C) 2004  Roberto Rossi

   // This program is free software; you can redistribute it and/or
   // modify it under the terms of the GNU General Public License
   // as published by the Free Software Foundation; either version 2
   // of the License, or (at your option) any later version.

   // This program is distributed in the hope that it will be useful,
   // but WITHOUT ANY WARRANTY; without even the implied warranty of
   // MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   // GNU General Public License for more details.

   // You should have received a copy of the GNU General Public License
   // along with this program; if not, write to the Free Software
   // Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


package ants.p2p;

import java.util.*;
import java.net.*;
import java.beans.*;
import java.security.*;

import ants.p2p.security.sockets.*;
import ants.p2p.messages.*;
import ants.p2p.utils.*;
import ants.p2p.query.*;
import ants.p2p.exceptions.*;

import org.apache.log4j.*;

/**
 * This class models a network node with every service related to Link/Net
 * level.
 */
public class Ant
    extends Thread {
  private static final String antVersion = "alpha9.8.6";

  private static Logger _logger = Logger.getLogger(Ant.class.getName());

  int serverPort = 4000;
  public static int myMessageSizeMax = Router.ownDeliveredMessagesQueue;
  public static int inTransitDelivered = Router.ownDeliveredMessagesQueue;
  public static int inTransitMessageSizeMax = Router.inTransitQueue;
  public static int maxNeighbours = 1;
  java.util.List neighbours = Collections.synchronizedList(new ArrayList());
  Hashtable neighbourEfficiency = new Hashtable();
  public java.util.List myMessages = Collections.synchronizedList(new ArrayList());
  public java.util.List failedMessages = Collections.synchronizedList(new ArrayList());
  public java.util.List deliveredMessages = Collections.synchronizedList(new ArrayList());
  java.util.List inTransitMessages = Collections.synchronizedList(new ArrayList());
  protected SecureServerSocket sss;

  public static long messageTimeout = 180000;
  public static int maxRetransmissions = 5;
  public static int maxRetransmissionsForceDirection = 3;
  public static int maxFailedMessageToTrace = 100;

  public static int beingRoutedMessages = 0;
  public static int maxMessagesToRouteToghether = 200;

  boolean terminate = false;

  public PropertyChangeSupport propertyChangeSupport = new
      PropertyChangeSupport(this);

  String id = null;


  /**
   * Creates and start an ANt net node
   * @param id
   * @param maxNeighbours
   * @param serverPort
   */
  public Ant(String id, int maxNeighbours, int serverPort) {
    this.id = id;
    this.serverPort = serverPort;
    if (maxNeighbours >= 1) {
      this.setMaxNeighbours(maxNeighbours);
    }
    this.setPriority(Thread.MIN_PRIORITY);
    this.start();
  }

  public static String getVersion() {
    return antVersion;
  }

  public String getId() {
    return this.id;
  }

  public int getServerPort() {
    return this.serverPort;
  }

  public java.util.List getFailedMessages() {
    return this.failedMessages;
  }

  public void setMaxNeighbours(int mn) {
    maxNeighbours = mn;
  }

  public int getMaxNeighbours() {
    return maxNeighbours;
  }

  /**
   * Disconnects a neighbour
   * @param n
   */
  public synchronized void removeNeighbour(NeighbourAnt n) {
    synchronized (neighbours) {
      synchronized (neighbourEfficiency) {
        try {
          this.neighbours.remove(n);
          this.neighbourEfficiency.remove(n.getId() + "");
          n.terminate();
        }
        catch (Exception e) {
          _logger.error(this.getId() + "",e);
        }
      }
    }
  }

  public NeighbourAnt getNeighbour(String id) {
    for (int x = 0; x < this.neighbours.size(); x++) {
      if ( ( (NeighbourAnt)this.neighbours.get(x)).getId().equals(id) ||
          ( (NeighbourAnt)this.neighbours.get(x)).getRemoteId().equals(id)) {
        return (NeighbourAnt)this.neighbours.get(x);
      }
    }
    return null;
  }

  public boolean equals(Object o) {
    if (o instanceof Ant)
      return ( (Ant) o).getId().equals(this.getId());
    else
      return o == this;
  }

  public int getNeighboursNumber() {
    return this.neighbours.size();
  }

  /**
   * Adds a neighbour node to this ANt node
   * @param n
   * @throws java.lang.Exception
   */
  public synchronized void addNeighbour(NeighbourAnt n) throws Exception {
    synchronized (neighbours) {
      synchronized (neighbourEfficiency) {
        if (this.neighbours.size() < this.maxNeighbours &&
            !this.neighbours.contains(n)) {
          this.neighbours.add(n);
          this.neighbourEfficiency.put(n.getId() + "", new Integer(0));
        }
        else {
          if (this.neighbours.size() >= this.maxNeighbours)
            throw new Exception(this.getId() +
                                ": Max neighbourg number reached");
          else if (this.neighbours.contains(n))
            throw new Exception(this.getId() + ": Neighbourg already connected");
          throw new Exception(this.getId() +
                              ": Strange thing in neighbour add process.");
        }
      }
    }
    _logger.debug(this.getId() + ": Added Neighbour by request IP(local): "+n.getId()+" IP(remote): "+n.getRemoteId());
    this.propertyChangeSupport.firePropertyChange("newNeighbour", null, n);
  }

  /**
   * Adds a neighbour by its remote address
   * @param remoteAddress
   * @param port
   * @param isRequirer
   * @param localhost
   * @throws java.lang.Exception
   */
  public synchronized void addNeighbour(String remoteAddress, int port,
                                        boolean isRequirer,
                                        InetAddress localhost) throws Exception {
    InetAddress remote = InetAddress.getByName(remoteAddress);
    if ( (InetAddress.getByName("127.0.0.1")).getHostAddress().equals(remote.
        getHostAddress()) &&
        port == this.getServerPort())
      return;
    if (localhost.getHostAddress().equals(remote.getHostAddress()) &&
        port == this.getServerPort())
      return;
    for (int x = 0; x < this.neighbours.size(); x++) {
      NeighbourAnt na = ( (NeighbourAnt)this.neighbours.get(x));
      if (na.getId().equals(remoteAddress + ":" + port) ||
          na.getRemoteId().equals(remoteAddress + ":" + port))
        return;
    }
    SecureClientSocket scs = new SecureClientSocket(remoteAddress, port,
        this.getServerPort());
    if (scs.isConnected()) {
      NeighbourAnt n = new NeighbourAnt(this, remoteAddress, port,
                                        scs.getLocalServerPort(),
                                        scs.getCipherEnc(), scs.getCipherDec(),
                                        scs, isRequirer);
      synchronized (neighbours) {
        synchronized (neighbourEfficiency) {
          if (this.neighbours.size() < this.maxNeighbours &&
              !this.neighbours.contains(n)) {
            this.neighbours.add(n);
            this.neighbourEfficiency.put(n.getId() + "", new Integer(0));
          }
          else {
            n.terminate();
            if (this.neighbours.size() >= this.maxNeighbours)
              throw new Exception(this.getId() +
                                  ": Max neighbourg number reached");
            else if (this.neighbours.contains(n))
              throw new Exception(this.getId() +
                                  ": Neighbourg already connected");
            throw new Exception(this.getId() +
                                ": Strange thing in neighbour add process.");
          }
        }
      }
      _logger.debug(this.getId() + ": Added Neighbourg from address IP(local): "+n.getId()+" IP(remote): "+n.getRemoteId());
      this.propertyChangeSupport.firePropertyChange("newNeighbour", null, n);
    }
    else {
      _logger.info(this.getId() + ": Connection rejected...");
    }
  }

  public java.util.List getNeighbours() {
    return this.neighbours;
  }

  public MessageWrapper getMessage(String id) {
    for (int x = 0; x < this.myMessages.size(); x++) {
      if ( ( (MessageWrapper)this.myMessages.get(x)).getMessage().getAck_Id().
          equals(id)) {
        return (MessageWrapper)this.myMessages.get(x);
      }
    }
    return null;
  }

  /**
   * Traces a delivered message and keeps track
   * of the route used to its destnation
   * @param wm
   */
  public void traceDeliveredMessage(MessageWrapper wm) {
    wm.setStoreFormat();
    //synchronized (deliveredMessages) {
      boolean removed = false;
      if (deliveredMessages.size() >= Router.ownDeliveredMessagesQueue &&
          deliveredMessages.size() > 0) {
        for (int x = deliveredMessages.size() - 1; x >= 0 && !removed; x--) {
          if ( ( (MessageWrapper) deliveredMessages.get(x)).getMessage().
              getDest().equals(wm.getMessage().getDest()) &&
              ( (MessageWrapper) deliveredMessages.get(x)).getMessage().
              getSource().equals(wm.getMessage().getSource())) {
            deliveredMessages.remove(x);
            removed = true;
          }
        }
        for (int x = deliveredMessages.size() - 1; x >= 0 && !removed; x--) {
          if ( ( (MessageWrapper) deliveredMessages.get(x)).getMessage().
              getDest().equals(wm.getMessage().getDest())) {
            deliveredMessages.remove(x);
            removed = true;
          }
        }
        for (int x = deliveredMessages.size() - 1; x >= 0 && !removed; x--) {
          if ( ( (MessageWrapper) deliveredMessages.get(x)).getMessage().
              getSource().equals(wm.getMessage().getDest())) {
            deliveredMessages.remove(x);
            removed = true;
          }
        }
        if (!removed) {
          deliveredMessages.remove(deliveredMessages.size() - 1);
          //deliveredMessages.trimToSize();
        }
        deliveredMessages.add(0, wm);
      }
      else {
        deliveredMessages.add(0, wm);
      }
    //}
  }

  /**
   * Traces an in-transit message
   * @param m
   * @param requirer
   * @param routedTo
   */
  public void traceMessage(Message m, String requirer, String routedTo) {
    MessageWrapper wm = new MessageWrapper(m, requirer);
    wm.setStoreFormat();
    //synchronized (wm) {
      wm.setRoutedTo(routedTo);
      wm.alreadyVisited.add(requirer);
    //}
    //synchronized (inTransitMessages) {
      int index = this.inTransitMessages.indexOf(m);
      if(index >=0){
        this.inTransitMessages.remove(index);
        //this.inTransitMessages.trimToSize();
      }
      if (this.inTransitMessages.size() < this.inTransitMessageSizeMax &&
          !this.inTransitMessages.contains(m)) {
        this.inTransitMessages.add(wm);
      }
      else if (this.inTransitMessages.size() >= this.inTransitMessageSizeMax &&
          !this.inTransitMessages.contains(m)) {
        this.inTransitMessages.add(wm);
        this.inTransitMessages.remove(0);
        //this.inTransitMessages.trimToSize();
      }
    //}
  }

  /**
   * Traces your own message
   * @param wm
   * @param routedTo
   */
  public void traceMyMessage(MessageWrapper wm, String routedTo) {
    //synchronized (wm) {
      wm.setRoutedTo(routedTo);
    //}
    //synchronized (this.myMessages) {
      int index = this.myMessages.indexOf(wm);
      if(index < 0){
        this.myMessages.add(wm);
      }else{
        this.myMessages.remove(wm);
        this.myMessages.add(wm);
        //this.myMessages.trimToSize();
      }
    //}
  }

  public static void decBeingRoutedMessages() {
    Ant.beingRoutedMessages--;
  }

  public static synchronized boolean verifyAndIncBeingRoutedMessages() {
    if (Ant.beingRoutedMessages < Ant.maxMessagesToRouteToghether) {
      Ant.beingRoutedMessages++;
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Activates a process to handle a message routing.
   * Priority is assigned watching at message class.
   * @param m
   * @param requirer
   * @throws java.lang.Exception
   */
  public void activateNewRouterProcess(Message m, String requirer) throws Exception {
    if (Ant.verifyAndIncBeingRoutedMessages()) {
      Router r = new Router(this, m, requirer);
      /*Prova Priorit*/
      if (m instanceof Message) {
        r.setPriority(Thread.MAX_PRIORITY);
      }
      else if (m instanceof ControlMessage ||
               m instanceof FileTransferEndControlMessage ||
               m instanceof FileTransferErrorControlMessage ||
               m instanceof SecurityRequestMessage ||
               m instanceof SecurityResponseMessage ||
               m instanceof FileSizePullErrorControlMessage ||
               m instanceof QueryMessage) {
        r.setPriority(Thread.MAX_PRIORITY);
      }
      else if (m instanceof FileSizePullMessage ||
               m instanceof FileSizePushMessage ||
               m instanceof FilePullMessage ||
               m instanceof FilePushMessage) {
        r.setPriority(Thread.NORM_PRIORITY);
      }
      else if (m instanceof FilePartMessage) {
        r.setPriority(Thread.MIN_PRIORITY);
      }
      if (Router.PARALLEL)
        r.start();
      else
        r.run();
    }
  }

  /**
   * Activates the right method to handle a message routing
   * @param wm
   * @param m
   * @param requirer
   * @param r
   * @param external
   * @param retransmission
   * @return
   * @throws java.lang.Exception
   */
  public SenderThread[] route(MessageWrapper wm, Message m, String requirer, Router r, boolean external, boolean retransmission)throws
      Exception {
    if(external){
      return this.routeExternalMessage(m, requirer, r);
    }else if(!external && retransmission){
      return this.retransmitMessage(wm);
    }else{
      return this.routeMyMessage(wm);
    }
  }

  /**
   * Main routing procedure for external message
   * @param m
   * @param requirer
   * @param r
   * @return
   * @throws java.lang.Exception
   */
  private SenderThread[] routeExternalMessage(Message m, String requirer, Router r) throws
      Exception {
    try {
      boolean jump = false;
      if (Math.random() < Router.jumpProbability)
        jump = true;
      if (m.getType() == 2)
        jump = true;

      NeighbourAnt requirerAnt = this.getNeighbour(requirer);
      if (Router.delay > 0)
        Thread.sleep( (long) Math.floor(Router.delay * Math.random()));

      if (m.ttl > Router.timeToLive) {
        m.ttl = Router.timeToLive;
      }
      m.decTTL();
      double mustDie = Math.random() * 100;
      if (m.getType() != 2 && m.getTTL() == 0 &&
          mustDie < r.dieProbability) {
        throw new Exception(this.getId() +
                            ": Destination node " + m.getDest() +
                            " unreachable for message from node " + m.getSource() +
                            " with id " +
                            m.ack_id + " type = " + m.getType() +
                            " Reason: Connection Timeout");
      }
      if (this.neighbours.size() == 0) {
        throw new Exception(this.getId() +
                            ": Destination node " + m.getDest() +
                            " unreachable for message from node " + m.getSource() +
                            " with id " +
                            m.ack_id + " type = " + m.getType() +
                            "\nReason: No Route Found " + "Node " + this.getId() +
                            " has no neighbours");
      }
      if (this.myMessages.contains(m)) {
        if (this.getNeighbour(requirer) != null &&
            m.getType() != 2) {
          int efficiency = ( (Integer)this.neighbourEfficiency.get(
              requirer + "")).
              intValue();
          if (efficiency > -Router.efficiencyLimit) {
            efficiency -=
                ( (Router.timeToLive -
                   ( ( (MessageWrapper)this.myMessages.get(this.myMessages.
                indexOf(
                m))).getMessage().getTTL() -
                    m.getTTL())) * 1.0 / Router.timeToLive) *
                Router.efficiencyModifier;
            if (efficiency < -Router.efficiencyLimit)
              efficiency = -Router.efficiencyLimit;
          }
          //synchronized (this.neighbourEfficiency) {
            this.neighbourEfficiency.put(requirer + "",
                                         new Integer(efficiency));
          //}
        }
        if(m.getType() != 2){
          int index = this.myMessages.indexOf(m);
          MessageWrapper wm = (MessageWrapper)this.myMessages.get(index);

          if (wm.getRetrasmissions() < m.getRetrasmissions() &&
              m.getRetrasmissions() <= Ant.maxRetransmissionsForceDirection) {
            wm.notToBeConsidered = Collections.synchronizedList(new ArrayList());
            wm.alreadyVisited = Collections.synchronizedList(new ArrayList());
          }
          else {
            if (!wm.notToBeConsidered.contains(wm.getRoutedTo())) {
              wm.notToBeConsidered.add(wm.getRoutedTo());
              wm.alreadyVisited.remove(wm.getRoutedTo());
            }
            if (!wm.alreadyVisited.contains(requirer)) {
              wm.alreadyVisited.add(requirer);
            }
          }

          for (int x = 0; x < this.deliveredMessages.size() && !jump; x++) {
            MessageWrapper old = (MessageWrapper)this.deliveredMessages.
                get(
                x);
            if (old.getMessage().getDest().equals(m.getDest()) &&
                !wm.notToBeConsidered.contains(old.getRoutedTo()) &&
                old.getMessage().getDelivered() == true) {
              NeighbourAnt n = this.getNeighbour(old.getRoutedTo());
              if (n == null)
                continue;
              this.traceMyMessage(wm, old.getRoutedTo());

              _logger.debug(this.getId() +
                            ": A Forwarding message with id = " +
                            m.ack_id + " type = " +
                            m.getType() +
                            " to id = " +
                            n.getId());

              this.deliveredMessages.remove(old);
              //this.deliveredMessages.trimToSize();
              this.deliveredMessages.add(0, old);

              SenderThread[] st = new SenderThread[1];
              st[0] =  n.route(m);
              return st;
            }
            else if (old.getMessage().getSource().equals(m.getDest()) &&
                     !wm.notToBeConsidered.contains(old.getRequirer()) &&
                     old.getMessage().getDelivered() == true) {
              NeighbourAnt n = this.getNeighbour(old.getRequirer());
              if (n == null)
                continue;
              this.traceMyMessage(wm, old.getRequirer());

              _logger.debug(this.getId() +
                            ": D Forwarding message with id = " +
                            m.ack_id + " type = " +
                            m.getType() +
                            " to id = " +
                            n.getId());

              this.deliveredMessages.remove(old);
              //this.deliveredMessages.trimToSize();
              this.deliveredMessages.add(0, old);

              SenderThread[] st = new SenderThread[1];
              st[0] = n.route(m);
              return st;
            }
          }

          NeighbourAnt n = this.findBestNeighbour(wm.notToBeConsidered,
                                                  wm.alreadyVisited,
                                                  requirer,
                                                  m.getResolveLoop(), jump);

          if (n != null) {
            this.traceMyMessage(wm, n.getId());

            _logger.debug(this.getId() +
                          ": Message with id = " +
                          m.ack_id + " type = " + m.getType() +
                          " returned. Not considering node " +
                          wm.getRoutedTo() + " routing to node " +
                          n.getId());

            SenderThread[] st = new SenderThread[1];
            st[0] = n.route(m);
            return st;
          }
          else {
            throw new Exception(this.getId() +
                                ": Destination node " + m.getDest() +
                                " unreachable for message from node " +
                                m.getSource() + " with id " +
                                m.ack_id +
                                " type = " + m.getType() +
                                "\nReason: No Route Found " + "Node " +
                                this.getId() + " excluded all its " +
                                wm.notToBeConsidered.size() + " neghbours");
          }
        }
        else {  // m.getType() == 2
          _logger.debug(this.getId() +
              ": Broadcast Message returned no more node avaiable to send. Killing...");
          return null;
        }
      }
      else if (!m.getDest().equals(this.getId())) {
        if (!this.inTransitMessages.contains(m)) {
          if (m.getType() == 1) {
            for (int k = 0; k < this.inTransitMessages.size(); k++) {
              MessageWrapper inTransit = (MessageWrapper)this.
                  inTransitMessages.get(k);
              if ( (inTransit.getMessage().getType() == 0 ||
                    inTransit.getMessage().getType() == 2) &&
                  inTransit.getMessage().getAck_Id().equals(m.ack_id)) {
                NeighbourAnt n = this.getNeighbour( ( (MessageWrapper)this.
                    inTransitMessages.get(k)).getRequirer());
                if (n != null) {
                  int efficiency = ( (Integer)this.neighbourEfficiency.get(
                      n.
                      getId() +
                      "")).intValue();
                  if (efficiency < Router.efficiencyLimit) {
                    efficiency +=
                        (m.getTTL() * 1.0 / Router.timeToLive) *
                        Router.efficiencyModifier;
                    if (efficiency > Router.efficiencyLimit)
                      efficiency = Router.efficiencyLimit;
                  }
                  this.neighbourEfficiency.put(n.getId() + "",
                                               new Integer(efficiency));

                  ( (MessageWrapper)this.inTransitMessages.get(k)).
                      getMessage().invalidate();
                  this.traceDeliveredMessage( (MessageWrapper)this.
                                             inTransitMessages.get(k));

                  _logger.debug(this.getId() +
                                ": C Forwarding message with id = " +
                                m.ack_id + " type = " +
                                m.getType() +
                                " to id = " +
                                n.getId());

                  SenderThread[] st = new SenderThread[1];
                  st[0] = n.route(m);
                  return st;
                }
              }
            }
          }
          if (m.getType() == 2) {
            this.processMessage(m, r);
            ArrayList threadsList = new ArrayList();
            for(int x = 0; x < this.getNeighbours().size(); x++){
              double mustRoute = Math.random() * 100;
              int ttlRouteFactor = (int)Math.floor((m.getTTL()*1.0 / WarriorAnt.maxQueryTimeToLive)*85);
              if(m.getTTL() != 0 ||
                 mustRoute < r.routeProbability + ttlRouteFactor){
                NeighbourAnt n = (NeighbourAnt)this.getNeighbours().get(x);
                if (n != null && !n.getId().equals(requirer)) {
                  threadsList.add(n.route(m));
                  _logger.debug(this.getId() +
                                ": Flooding message with id = " +
                                m.ack_id + " type = " +
                                m.getType() +
                                " to id = " +
                                n.getId());
                }
              }
            }
            this.traceMessage(m, requirer, "");
            SenderThread[] st = new SenderThread[threadsList.size()];
            for(int x = 0; x < threadsList.size(); x++)
              st[x] = (SenderThread)threadsList.get(x);
            return st;
          }
          else{
            for (int x = 0; x < this.deliveredMessages.size() && !jump; x++) {
              MessageWrapper old = (MessageWrapper)this.deliveredMessages.get(
                  x);
              if (old.getMessage().getDest().equals(m.getDest()) &&
                  old.getMessage().getDelivered() == true) {
                NeighbourAnt n = this.getNeighbour(old.getRoutedTo());
                if (n == null)
                  continue;
                this.traceMessage(m, requirer, old.getRoutedTo());

                _logger.debug(this.getId() +
                              ": A Forwarding message with id = " +
                              m.ack_id + " type = " +
                              m.getType() +
                              " to id = " +
                              n.getId());

                this.deliveredMessages.remove(old);
                //this.deliveredMessages.trimToSize();
                this.deliveredMessages.add(0, old);

                SenderThread[] st = new SenderThread[1];
                st[0] = n.route(m);
                return st;
              }
              else if (old.getMessage().getSource().equals(m.getDest()) &&
                       old.getMessage().getDelivered() == true) {
                NeighbourAnt n = this.getNeighbour(old.getRequirer());
                if (n == null)
                  continue;
                this.traceMessage(m, requirer, old.getRequirer());

                _logger.debug(this.getId() +
                              ": D Forwarding message with id = " +
                              m.ack_id + " type = " +
                              m.getType() +
                              " to id = " +
                              n.getId());

                this.deliveredMessages.remove(old);
                //this.deliveredMessages.trimToSize();
                this.deliveredMessages.add(0, old);

                SenderThread[] st = new SenderThread[1];
                st[0] = n.route(m);
                return st;
              }
            }

            NeighbourAnt n = this.findBestNeighbour(Collections.
                synchronizedList(new ArrayList()),
                Collections.synchronizedList(new ArrayList()),
                requirer,
                m.getResolveLoop(),
                jump);

            if (n != null) {

              _logger.debug(this.getId() +
                            ": B Forwarding message with id = " +
                            m.ack_id + " type = " + m.getType() +
                            " to id = " +
                            n.getId());

              this.traceMessage(m, requirer, n.getId());
              SenderThread[] st = new SenderThread[1];
              st[0] = n.route(m);
              return st;
            }
            else {
              throw new NullNeighbourException(this.getId() +
                                               ": Destination node " +
                                               m.getDest() +
                  " unreachable for message from node " +
                  m.getSource() + " with id " +
                  m.ack_id +
                  " type = " + m.getType() +
                  "\nReason: No Route Found with no excluded neghbours");
            }
          }
        }
        else {
          int index = this.inTransitMessages.indexOf(m);
          MessageWrapper wm = (MessageWrapper)this.inTransitMessages.get(
              index);
            if (this.getNeighbour(requirer) != null &&
                m.getType() != 2) {
              int efficiency = ( (Integer)this.neighbourEfficiency.get(
                  requirer + "")).intValue();
              if (efficiency > -Router.efficiencyLimit) {
                efficiency -=
                    ( (Router.timeToLive -
                       ( ( (MessageWrapper)this.inTransitMessages.
                          get(this.inTransitMessages.indexOf(m))).getMessage().
                        getTTL() -
                        m.getTTL())) *
                     1.0 / Router.timeToLive) * Router.efficiencyModifier;
                if (efficiency < -Router.efficiencyLimit)
                  efficiency = -Router.efficiencyLimit;
              }
              this.neighbourEfficiency.put(requirer + "",
                                           new Integer(efficiency));
            }
            if(m.getType() != 2){
              if (wm.getRetrasmissions() < m.getRetrasmissions() &&
                  m.getRetrasmissions() <= Ant.maxRetransmissionsForceDirection) {
                wm.notToBeConsidered = Collections.synchronizedList(new ArrayList());
                wm.alreadyVisited = Collections.synchronizedList(new ArrayList());
              }
              else {
                if (!wm.notToBeConsidered.contains(wm.getRoutedTo())) {
                  wm.notToBeConsidered.add(wm.getRoutedTo());
                  wm.alreadyVisited.remove(wm.getRoutedTo());
                }
                if (!wm.alreadyVisited.contains(requirer)) {
                  wm.alreadyVisited.add(requirer);
                }
              }

              _logger.debug(this.getId() +
                            ": Message with id = " +
                            m.ack_id + " type = " + m.getType() +
                            " returned. Not considering node " +
                            wm.getRoutedTo());

              for (int x = 0; x < this.deliveredMessages.size() && !jump; x++) {
                MessageWrapper old = (MessageWrapper)this.deliveredMessages.
                    get(
                    x);
                if (old.getMessage().getDest().equals(m.getDest()) &&
                    !wm.notToBeConsidered.contains(old.getRoutedTo()) &&
                    old.getMessage().getDelivered() == true) {
                  NeighbourAnt n = this.getNeighbour(old.getRoutedTo());
                  if (n == null)
                    continue;
                  //synchronized (wm) {
                  wm.setRoutedTo(old.getRoutedTo());
                  //}

                  _logger.debug(this.getId() +
                                ": A Forwarding message with id = " +
                                m.ack_id + " type = " +
                                m.getType() +
                                " to id = " +
                                n.getId());

                  this.deliveredMessages.remove(old);
                  //this.deliveredMessages.trimToSize();
                  this.deliveredMessages.add(0, old);

                  SenderThread[] st = new SenderThread[1];
                  st[0] = n.route(m);
                  return st;
                }
                else if (old.getMessage().getSource().equals(m.getDest()) &&
                         !wm.notToBeConsidered.contains(old.getRequirer()) &&
                         old.getMessage().getDelivered() == true) {
                  NeighbourAnt n = this.getNeighbour(old.getRequirer());
                  if (n == null)
                    continue;
                  //synchronized (wm) {
                  wm.setRoutedTo(old.getRoutedTo());
                  //}

                  _logger.debug(this.getId() +
                                ": D Forwarding message with id = " +
                                m.ack_id + " type = " +
                                m.getType() +
                                " to id = " +
                                n.getId());

                  this.deliveredMessages.remove(old);
                  //this.deliveredMessages.trimToSize();
                  this.deliveredMessages.add(0, old);

                  SenderThread[] st = new SenderThread[1];
                  st[0] = n.route(m);
                  return st;
                }
              }

              NeighbourAnt n = this.findBestNeighbour(wm.notToBeConsidered,
                  wm.alreadyVisited,
                  wm.getRequirer(), m.getResolveLoop(), jump);

              if (n != null) {
                wm.setRoutedTo(n.getId());
                //synchronized (this.inTransitMessages) {
                this.inTransitMessages.remove(wm);
                //this.inTransitMessages.trimToSize();
                this.inTransitMessages.add(wm);
                //}
                SenderThread[] st = new SenderThread[1];
                st[0] = n.route(m);
                return st;
              }
              else {
                throw new NullNeighbourException(this.getId() +
                                                 ": Destination node " +
                                                 m.getDest() +
                    " unreachable for message from node " +
                    m.getSource() + " with id " +
                    m.ack_id +
                    " type = " + m.getType() +
                    "\nReason: No Route Found Node " +
                    this.getId() +
                    " excluded all its " +
                    wm.notToBeConsidered.size() +
                    " neghbours");
              }
            }else{ // m.getType() == 2
              _logger.debug(this.getId() +
                                ": D Flooded message with id = " +
                                m.ack_id + " type = " +
                                m.getType() +
                                " returned. Killing!");
              return null;
            }
        }
      }
      else if (m.getDest().equals(this.getId())) {
        if (this.getNeighbour(requirer) != null &&
            m.getType() != 2) {
          int efficiency = ( (Integer)this.neighbourEfficiency.get(
              requirer + "")).
              intValue();
          if (efficiency < Router.efficiencyLimit) {
            efficiency += (m.getTTL() * 1.0 / Router.timeToLive) *
                Router.efficiencyModifier;
            if (efficiency > Router.efficiencyLimit)
              efficiency = Router.efficiencyLimit;
          }
          this.neighbourEfficiency.put(requirer + "",
                                       new Integer(efficiency));
        }
        if (m.getType() == 0 && !m.getSource().equals("")) {

          _logger.debug(this.getId() +
                        ": Received Message sending Ack for message with id = " +
                        m.ack_id + " type = " + m.getType());

          this.traceDeliveredMessage(new MessageWrapper(m, requirer));
          Message ackMessage = new Message();
          ackMessage.fillMessageProperties(this.getId(),
                                           m.getSource(), 1,
                                           m.ack_id
                                           );
          MessageWrapper wm = new MessageWrapper(ackMessage, requirer);
          this.processMessage(m, r);
          if (this.getNeighbour(requirer) != null) {
            SenderThread[] st = new SenderThread[1];
            st[0] = this.getNeighbour(requirer).route(new Message(wm));
            return st;
          }
          else {
            NeighbourAnt n = this.findBestNeighbour(Collections.synchronizedList(new ArrayList()),
                Collections.synchronizedList(new ArrayList()),
                requirer, m.getResolveLoop(), jump);
            SenderThread[] st = new SenderThread[1];
            st[0] = n.route(new Message(wm));
            return st;
          }
        }
        else if ( (m.getType() == 0 || m.getType() == 1) &&
                 m.getSource().equals("")) {

          _logger.debug(this.getId() +
                        ": Received Message with masked source id = " +
                        m.ack_id + " type = " + m.getType());

          this.processMessage(m, r);

          return null;
        }
        else if (m.getType() == 1) {
          if (this.getMessage(m.ack_id) != null) {
            if (m instanceof Message) {
              MessageWrapper movingMessage = null;
              //synchronized (this.myMessages) {
                movingMessage = (MessageWrapper)this.getMessage(m.ack_id);
                if (movingMessage != null) {
                  movingMessage.getMessage().invalidate();
                  this.myMessages.remove(movingMessage);
                  this.traceDeliveredMessage(movingMessage);
                  Ant.decBeingRoutedMessages();
                }
                else {
                  Ant.decBeingRoutedMessages();
                }
              //}
            }

            _logger.debug(this.getId() +
                          ": Message succesfully routed id = " +
                          m.ack_id +
                          " type = " + m.getType());

            this.processMessage(m, r);

            return null;
          }
          else {
            _logger.debug(this.getId() +
                          ": (Retransmission) Message succesfully routed id = " +
                          m.ack_id +
                          " type = " + m.getType());

            this.processMessage(m, r);

            return null;
          }
        }
        else {
          throw new Exception(this.getId() +
              ": There is something strange(2) for message with id = " +
              m.ack_id + " type = " + m.getType());
        }
      }
      else {
        throw new Exception(this.getId() +
            ": There is something strange(3) for message with id = " +
            m.ack_id +
            " type = " + m.getType());
      }
    }
    catch (Exception e) {
      if (e instanceof NullNeighbourException && m.getType() == 2) {

        _logger.debug(this.getId() +
            ": Received Broadcast Message sending Ack for message with id = " +
            m.ack_id + " type = " + m.getType());

        this.traceDeliveredMessage(new MessageWrapper(m,
            requirer));
        Message ackMessage = new Message();
        ackMessage.fillMessageProperties(this.getId(),
                                         m.getSource(), 1,
                                         m.ack_id
                                         );
        MessageWrapper wm = new MessageWrapper(ackMessage, requirer);
        SenderThread[] st = null;
        if (getNeighbour(requirer) != null) {
          st[0] = this.getNeighbour(requirer).route(new Message(wm));
        }
        else {
          NeighbourAnt n = this.findBestNeighbour(Collections.synchronizedList(new ArrayList()),
                                                  Collections.synchronizedList(new ArrayList()),
                                                  requirer, m.getResolveLoop(), false);
          st[0] = n.route(new Message(wm));
        }
        m.invalidate();
        this.processMessage(m, r);
        return st;
      }
      else {
        _logger.error(this.getId() + " routing error: ", e);
        return null;
      }
    }
  }

  /**
   * Main routing procedure for own messages
   * @param wm
   * @return
   * @throws java.lang.Exception
   */
  private SenderThread[] routeMyMessage(MessageWrapper wm) throws Exception {
    boolean jump = false;
    if (Math.random() < Router.jumpProbability)
      jump = true;
    if (wm.getMessage().getType() == 2)
      jump = true;

    if(wm.getMessage().getType() != 2){
      NeighbourAnt n = this.checkForRoute(wm, jump);
      if (n != null) {
        SenderThread[] st = new SenderThread[1];
        st[0] = n.route(wm.getMessage());
        return st;
      }
      else {
        n = this.findBestNeighbour(Collections.synchronizedList(new ArrayList()),
                                   Collections.synchronizedList(new ArrayList()),
                                   this.getId(),
                                   wm.getMessage().getResolveLoop(),
                                   jump);
        if (n != null) {
          this.traceMyMessage(wm, n.getId());
          _logger.debug(this.getId() + ": Routing message with id = " +
                        wm.getMessage().getAck_Id() + " type = " +
                        wm.getMessage().getType() + " to id = " +
                        wm.getRoutedTo());

          SenderThread[] st = new SenderThread[1];
          st[0] = n.route(wm.getMessage());
          return st;
        }
        else {
          throw new Exception(this.getId() +
              ": Destination unreachable for message with id = " +
              wm.getMessage().getAck_Id() + " type = " +
              wm.getMessage().getType());
        }
      }
    }else{
      ArrayList threadsList = new ArrayList();
      for (int x = 0; x < this.getNeighbours().size(); x++) {
        NeighbourAnt n = (NeighbourAnt)this.getNeighbours().get(x);
        if (n != null) {
          threadsList.add(n.route(wm.getMessage()));
          _logger.debug(this.getId() +
                        ": Flooding message with id = " +
                        wm.getMessage().ack_id + " type = " +
                        wm.getMessage().getType() +
                        " to id = " +
                        n.getId());
        }
      }
      this.traceMyMessage(wm, "");
      SenderThread[] st = new SenderThread[threadsList.size()];
      for (int x = 0; x < threadsList.size(); x++)
        st[x] = (SenderThread) threadsList.get(x);
      return st;
    }
  }

  /**
   * Sends a unicast message
   * @param message
   * @param dest
   * @param disableAutoRetransmit
   * @param maskSource
   * @return
   * @throws java.lang.InterruptedException
   */
  public MessageWrapper sendMessage(Message message, String dest,
                                    boolean disableAutoRetransmit,
                                    boolean maskSource) throws
      InterruptedException {
    String messageID = this.getId() + "@" + System.currentTimeMillis();
    if (!maskSource)
      message.fillMessageProperties(this.getId(), dest, 0, messageID);
    else
      message.fillMessageProperties("", dest, 0, messageID);

    _logger.debug("There are " + this.failedMessages.size() +
                         " messages failed!");
    _logger.debug("There are " + this.myMessages.size() +
                         " messages pending!");
    _logger.debug(this.getId() + ": Sending Message from id = " +
                         this.getId() + " to id = " + dest + " message id = " +
                         messageID);

    Thread.sleep(50);

    MessageWrapper wm = new MessageWrapper(message, this.getId());
    if (disableAutoRetransmit) {
      wm.disableAutoRetransmit();
    }
    try {
      SenderThread[] st = this.route(wm,null,null,null,false,false);
      if(st != null){
        for(int x = 0; x < st.length; x++){
          st[x].join();
        }
      }
      return wm;
    }
    catch (Exception e) {
      _logger.error(this.getId() + "",e);
      return null;
    }
  }

  /**
   * Sends a broadcast message
   * @param message
   * @return
   */
  public MessageWrapper sendBroadcastMessage(Message message) {
    String messageID = this.getId() + "@" + System.currentTimeMillis();
    SecureRandom sr = new SecureRandom();
    byte[] randomDest = new byte[10];
    sr.nextBytes(randomDest);
    String dest = Base16.toHexString(randomDest);
    message.fillMessageProperties(this.getId(), dest, 2, messageID);

    try {
      Thread.sleep(50);
    }
    catch (Exception e) {
      _logger.error(this.getId() + "",e);
    }

    MessageWrapper wm = new MessageWrapper(message, this.getId());
    wm.disableAutoRetransmit();
    try {
      SenderThread st[] = this.route(wm,null,null,null,false,false);
      if(st != null){
        for(int x = 0; x < st.length; x++){
          st[x].join();
        }
      }
      return wm;
    }
    catch (Exception e) {
      _logger.error(this.getId() + "",e);
      return null;
    }
  }

  /**
   * Handles retrasmissions of messages
   * @param wm
   * @return
   */
  private SenderThread[] retransmitMessage(MessageWrapper wm) {
    try {
      Thread.sleep(1000);
    }
    catch (Exception e) {
      _logger.error(this.getId() + "",e);
    }
    try {
      if (wm.getRetrasmissions() > Ant.maxRetransmissions)
        throw new Exception("Retransmissions exceeeded");
      _logger.debug(this.getId() + ": Retransmitting Message from id = " +
                           this.getId() + " to id = " + wm.getMessage().getDest() +
                           " message id = " + wm.getMessage().getAck_Id());
      wm.getMessage().resetResolveLoop();
      wm.getMessage().resetTTL();
      wm.getMessage().resetDelivered();


      boolean jump = false;
      if (Math.random() < Router.jumpProbability)
        jump = true;
      if (wm.getMessage().getType() == 2)
        jump = true;

      if (wm.getRetrasmissions() <= Ant.maxRetransmissionsForceDirection) {
        wm.notToBeConsidered = Collections.synchronizedList(new ArrayList());
        wm.alreadyVisited = Collections.synchronizedList(new ArrayList());
      }
      else if (!wm.notToBeConsidered.contains(wm.getRoutedTo())) {
        wm.notToBeConsidered.add(wm.getRoutedTo());
        wm.alreadyVisited.remove(wm.getRoutedTo());
      }

      NeighbourAnt n = this.checkForRoute(wm, jump);
      if (n != null) {
        SenderThread[] st = new SenderThread[1];
        st[0] = n.route(wm.getMessage());
        return st;
      }else{
        n = this.findBestNeighbour(wm.notToBeConsidered, wm.alreadyVisited,
                                   this.getId(),
                                   wm.getMessage().getResolveLoop(),
                                   jump);
        if (n != null) {
          this.traceMyMessage(wm, n.getId());
          _logger.debug(this.getId() +
                        ": Routing (retransmission) message with id = " +
                        wm.getMessage().getAck_Id() + " type = " +
                        wm.getMessage().getType() + " to id = " +
                        wm.getRoutedTo());
          SenderThread[] st = new SenderThread[1];
          st[0] =  n.route(wm.getMessage());
          return st;
        }
        else {
          int x = 0;
          throw new Exception(this.getId() +
              ": Destination (retransmission) unreachable for message with id = " +
              wm.getMessage().getAck_Id() + " type = " +
              wm.getMessage().getType());
        }
      }
    }
    catch (Exception e) {
      _logger.error(this.getId() + "",e);
      return null;
    }
  }

  /**
   * Check for a known route to message destination
   * @param wm
   * @param jump
   * @return
   */
  public NeighbourAnt checkForRoute(MessageWrapper wm, boolean jump){
    for (int x = 0; x < this.deliveredMessages.size() && !jump; x++) {
      MessageWrapper old = (MessageWrapper)this.deliveredMessages.get(x);
      if (old.getMessage().getDest().equals(wm.getMessage().getDest()) &&
          old.getMessage().getDelivered() == true) {
        NeighbourAnt n = this.getNeighbour(old.getRoutedTo());
        if (n == null)
          continue;
        this.traceMyMessage(wm, old.getRoutedTo());
        _logger.debug(this.getId() + ": Routing message with id = " +
                             wm.getMessage().getAck_Id() + " type = " +
                             wm.getMessage().getType() + " to id = " +
                             wm.getRoutedTo());
        //synchronized (this.deliveredMessages) {
          this.deliveredMessages.remove(old);
          //this.deliveredMessages.trimToSize();
          this.deliveredMessages.add(old);
        //}
        return n;
      }
      else if (old.getMessage().getSource().equals(wm.getMessage().getDest()) &&
               old.getMessage().getDelivered() == true) {
        NeighbourAnt n = this.getNeighbour(old.getRequirer());
        if (n == null)
          continue;
        this.traceMyMessage(wm, old.getRequirer());
        _logger.debug(this.getId() + ": Routing message with id = " +
                             wm.getMessage().getAck_Id() + " type = " +
                             wm.getMessage().getType() + " to id = " +
                             wm.getRoutedTo());
        //synchronized (this.deliveredMessages) {
          this.deliveredMessages.remove(old);
          //this.deliveredMessages.trimToSize();
          this.deliveredMessages.add(old);
        //}
        return n;
      }
    }
    return null;
  }

  /**
   * Looks for the best neighbour to route the message to.
   * This method is called when no existing route has been
   * found to a message destination.
   * @param alreadyRoutedTo
   * @param alreadyComingFrom
   * @param source
   * @param resolveLoop
   * @param jump
   * @return
   */
  public NeighbourAnt findBestNeighbour(java.util.List alreadyRoutedTo, java.util.List alreadyComingFrom, String source,
                                  boolean resolveLoop, boolean jump) {
    String bestNeighbour = null;
    if (jump) {
      Enumeration neighbours = neighbourEfficiency.keys();
      java.util.List avaiableNeighbours = Collections.synchronizedList(new ArrayList());
      java.util.List notVisitedNeighbours = Collections.synchronizedList(new ArrayList());
      while (neighbours.hasMoreElements()) {
        String currentNeighbour = (String) neighbours.nextElement();
        if(this.getNeighbour(currentNeighbour) == null){
          this.neighbourEfficiency.remove(currentNeighbour);
        }
        if (!alreadyRoutedTo.contains(currentNeighbour) &&
            !alreadyComingFrom.contains(currentNeighbour) &&
            !currentNeighbour.equals(source))
          notVisitedNeighbours.add(currentNeighbour);
        else if(!alreadyRoutedTo.contains(currentNeighbour) &&
                !currentNeighbour.equals(source))
          avaiableNeighbours.add(currentNeighbour);
      }
      if ((avaiableNeighbours.size() == 0 && notVisitedNeighbours.size() == 0) &&
               (alreadyRoutedTo.contains(source) || source.equals(this.getId()))) {
        bestNeighbour = null;
      }else if ((avaiableNeighbours.size() == 0 && notVisitedNeighbours.size() == 0 ) &&
          !(source.equals(this.getId()) || alreadyRoutedTo.contains(source))) {
        bestNeighbour = source;
      }
      else if(notVisitedNeighbours.size() > 0){
        int index = (int) Math.ceil(Math.random() * notVisitedNeighbours.size());
        for (int x = 0; x < index; x++) {
          bestNeighbour = (String) notVisitedNeighbours.get(x);
        }
      }else if(avaiableNeighbours.size() > 0){
        int index = (int) Math.ceil(Math.random() * avaiableNeighbours.size());
        for (int x = 0; x < index; x++) {
          bestNeighbour = (String) avaiableNeighbours.get(x);
        }
      }
    }
    else {
      int efficiency = Integer.MIN_VALUE;
      if (resolveLoop)
        efficiency = Integer.MAX_VALUE;
      //synchronized (neighbourEfficiency) {
        Enumeration neighbours = neighbourEfficiency.keys();
        Enumeration efficiencies = neighbourEfficiency.elements();
        while (efficiencies.hasMoreElements()) {
          String currentNeighbour = (String) neighbours.nextElement();
          if (this.getNeighbour(currentNeighbour) == null) {
            this.neighbourEfficiency.remove(currentNeighbour);
          }
          int currentEfficiency = ( (Integer) efficiencies.nextElement()).
              intValue();
          if(!alreadyComingFrom.contains(currentNeighbour) && !resolveLoop)
            currentEfficiency += Router.efficiencyLimit;
          else if(!alreadyComingFrom.contains(currentNeighbour) && resolveLoop)
            currentEfficiency -= Router.efficiencyLimit;
          if ( ( ( (efficiency < currentEfficiency) && !resolveLoop) ||
                ( (efficiency > currentEfficiency) && resolveLoop)) &&
              !alreadyRoutedTo.contains(currentNeighbour) &&
              !currentNeighbour.equals(source)) {
            efficiency = currentEfficiency;
            bestNeighbour = currentNeighbour;
          }
          else if (efficiency == currentEfficiency &&
                   !alreadyRoutedTo.contains(currentNeighbour) &&
                   !currentNeighbour.equals(source)) {
            double mustSelect = Math.random();
            if (mustSelect < 0.5) {
              bestNeighbour = currentNeighbour;
            }
          }
        }
        if (bestNeighbour == null &&
            !source.equals(this.getId()) &&
            !alreadyRoutedTo.contains(source))
          bestNeighbour = source;
      //}
    }
    return this.getNeighbour(bestNeighbour);
  }

  /**
   * Ant main loop. Manage stale neighbours & message retransmissions.
   */
  public void run() {
    try {
      sss = new SecureServerSocket(this, this.serverPort);
      if (sss == null)
        return;
      while (!terminate) {
        try {
          System.gc();
          for (int z = 0; z < 3; z++) {
            int sleepTime = (int) Math.floor( (Ant.messageTimeout * 1.0) / 3);
            this.sleep(sleepTime);
            try {
              for (int x = 0; x < this.neighbours.size(); x++) {
                if (! ( (NeighbourAnt)this.neighbours.get(x)).isAlive()) {
                  this.removeNeighbour( (NeighbourAnt)this.neighbours.get(x));
                  _logger.info("Removing stuck neighbour...");
                }
              }
              for (int x = this.neighbours.size() - 1; x >= 0; x--) {
                for (int y = x - 1; y >= 0; y--) {
                  if (this.neighbours.get(x).equals(this.neighbours.get(y))) {
                    this.removeNeighbour( (NeighbourAnt)this.neighbours.get(x));
                    _logger.info("Removing duplicated neighbour...");
                    break;
                  }
                }
              }
            }
            catch (Exception e) {
              _logger.error(this.getId() +
                            ": Error in removing stuck neighbour", e);
            }
          }
          while (this.failedMessages.size() > Ant.maxFailedMessageToTrace) {
            this.failedMessages.remove(0);
            //this.failedMessages.trimToSize();
          }
          //synchronized (this.myMessages) {
          if (this.myMessages.size() != 0) {
            _logger.debug(this.getId() + ": Processing " +
                          this.myMessages.size() +
                          " messages for eventual retransmission...");
            _logger.debug("There are " + this.failedMessages.size() +
                          " messages failed!");
            int[] toBeRemoved = new int[this.myMessages.size()];
            for (int x = 0;
                 x < this.myMessages.size() && x < toBeRemoved.length; x++) {
              MessageWrapper wm = (MessageWrapper)this.myMessages.get(x);
              if (wm.getLifetime() > Ant.messageTimeout) {
                if (wm.getRetrasmissions() < Ant.maxRetransmissions) {
                  wm.retrasmitted(this);
                  SenderThread[] st = this.route(wm, null, null, null, false, true);
                  if (st != null) {
                    for (int i = 0; i < st.length; i++) {
                      st[i].join();
                    }
                  }
                  _logger.debug("Retransmited Message: " +
                                wm.getMessage().getAck_Id() +
                                wm.getMessage().getType());
                }
                else {
                  if (wm.getMessage().getType() == 2) {
                    _logger.info(this.getId() +
                                 ": Broadcast message timed out " +
                                 wm.getMessage().getDest());
                  }
                  else if (wm.getMessage().getType() == 0 &&
                           wm.getMessage().getSource().equals("")) {
                    _logger.info(this.getId() +
                                 ": Masked message timed out " +
                                 wm.getMessage().getDest());
                  }
                  else {
                    _logger.info(this.getId() +
                                 ": Destination unreachable " +
                                 wm.getMessage().getDest());
                    //synchronized (this.failedMessages) {
                    this.failedMessages.add(wm);
                    //}
                  }
                  toBeRemoved[x] = 1;
                }
              }
            }
            for (int x = this.myMessages.size() - 1; x >= 0; x--) {
              if (toBeRemoved[x] == 1) {
                MessageWrapper wm = (MessageWrapper)this.myMessages.remove(x);
                _logger.info("Removed Message from myMessages: " +
                             wm.getMessage().getAck_Id() +
                             wm.getMessage().getType());
              }
            }
            //this.myMessages.trimToSize();
          }
          //}
        }
        catch (Exception cycleException) {
          _logger.info("ANt(ID_" + this.getId() + ") main cycle error! Cause: " +
                       cycleException.getMessage());
          _logger.debug(this.getId(), cycleException);
        }
      }
    }
    catch (Exception e) {
      this.terminate = true;
      _logger.info(this.getId() + " - Failure in setting up Ant server port: " +
                   e.getMessage());
      _logger.debug(this.getId(), e);
    }
  }

  void processMessage(Message m, Router r) throws Exception {
    _logger.debug(this.getId() +
                         ": Received Message = " +
                         m.getAck_Id() +
                         " type = " + m.getType() +
                         " from a peer running version: " + m.getVersion());
  }

  public PropertyChangeSupport getPropertyChangeSupport() {
    return this.propertyChangeSupport;
  }

  /**
   * Disconnects this node
   */
  public void disconnect() {
    try {
      for (int x = 0; x < this.neighbours.size(); x++) {
        NeighbourAnt na = ( (NeighbourAnt)this.neighbours.get(x));
        na.terminate();
      }
      if (sss != null) {
        sss.close();
        sss.serverThread.stop();
      }
      this.terminate = true;
      this.stop();
    }
    catch (Exception e) {
      _logger.error(this.getId() + "",e);
    }
  }

  public boolean isDisconnected() {
    return this.terminate;
  }
}