package JSR.ServiceManagers;
import JSR.*;
import java.io.IOException;
import java.rmi.RemoteException;
import java.util.Vector;
import JSR.Exceptions.JSRServiceNotAvaliable;

/**
 * ServiceManager Con gestione a Code delle richieste.
 * <br/>Viene gestita una coda per ogni livello di priorit (vedere {@link JSR.Constants.Priorities Costants.Priorities}),
 * per evitare starvation se l'attesa su una coda si protrae per {@link QueuedServiceManager#WAIT_THRESHOLD WAIT_THRESHOLD} la richiesta viene
 * passata alla coda di priorit immediatamente superiore. 
 * @author Mattia Righini Mat 170738
 */
public class QueuedServiceManager extends AServiceManager {

    public static int WAIT_THRESHOLD=5;
	protected int waitThreshold;
	protected Vector[] queues;
	public QueuedServiceManager(String serviceName,Interface serviceInterface,IReplicationManager parentManager,int waitThreshold)
	{
		super(serviceName,serviceInterface,parentManager);
		this.waitThreshold=waitThreshold;
		queues=new Vector[Constants.Priorities.MIN+1];
		for(int i=0;i<=Constants.Priorities.MIN;i++)
			queues[i]=new Vector();
	}
	synchronized protected void wait(Account a)throws RemoteException
	{
		Activate();
		Configuration.DebugOutput("Servizio: "+getServiceName()+" wait("+a+")");
		Configuration.DebugOutput("\tlock="+serviceCurrentStatus.isLock());
		for(int i=0;i<=Constants.Priorities.MIN;i++)
			Configuration.DebugOutput("Coda a livello "+i+" Richieste "+queues[i].size());
		if((serviceCurrentStatus.isLock() && (!serviceCurrentStatus.getLockAccount().equals(a)))
				|| (!serviceCurrentStatus.isLock())){
			int basePriority=a.getBasePriority();
			boolean queued=serviceCurrentStatus.isLock();
			if(!serviceCurrentStatus.isLock()){
				for(int i=0;i<=basePriority;i++)
					if(!queues[i].isEmpty())
						queued=true;
			}
			if(queued)
			{
				Configuration.DebugOutput("\tAccount "+a+" accodato");
				queues[basePriority].add(a);
				int waitcount=0;
				int priority=basePriority;
				boolean rewait;
				do{
					try{
						wait();
					}catch(InterruptedException e)
					{
						System.out.println(e);
						System.exit(1);
					}
					rewait=false;
					for(int i=0;i<priority;i++)
					{
						if(!queues[i].isEmpty())
						{
							rewait=true;
							break;
						}
					}
					rewait=rewait||( (!queues[priority].isEmpty()) && 
									(!((Account)queues[priority].firstElement()).equals(a)) );
					if(rewait)
					{
						waitcount++;
						if((waitcount>=waitThreshold) && (priority>0))
						{
							queues[priority].remove(a);
							priority--;
							waitcount=0;
							queues[priority].add(a);
							Configuration.DebugOutput("\tAccount "+a+" nuova priorit="+priority);
						}
					}
					else
					{
						queues[priority].remove(0);
					}
				}while(rewait);
			}
		}		
	}
	
	public Object SendRequest(Request r, Account a) throws RemoteException
	{
		wait(a);
		if(activeProvider==null)Activate();
		while(activeProvider!=null)
		{
			try
			{
				Object ret=activeProvider.sendRequest(r,a);
				synchronized (this) {
					if(!serviceCurrentStatus.isLock())notifyAll();
				} 
				return ret;
			}catch(IOException e)
			{
				Activate();
			}
		}
		throw new JSRServiceNotAvaliable(getServiceName());
	}
	public boolean lock(Account a) throws RemoteException{
		wait(a);
		if(activeProvider==null)Activate();
		while(activeProvider!=null)
		{
			try
			{				
				boolean ret=activeProvider.lock(a);
				if(!ret)
					synchronized (this) {
						notifyAll();
					} 
				return ret;
			}catch(IOException e)
			{
				Activate();
			}
		}
		throw new JSRServiceNotAvaliable(getServiceName());		
	}

	public boolean unLock(Account a)throws RemoteException {
		wait(a);
		if(activeProvider==null)Activate();
		while(activeProvider!=null)
		{
			try
			{
				boolean ret=activeProvider.unLock(a);
				if(ret)
					synchronized (this) {
						notifyAll();
					} 
				return ret;
			}
			catch(IOException e)
			{
				Activate();
			}
		}
		throw new JSRServiceNotAvaliable(getServiceName());
	}
	public boolean updateStatus(Status s)
	{
		if((serviceCurrentStatus==null)||
				(serviceCurrentStatus.getStatusCounter()<=s.getStatusCounter()))
		{
			serviceCurrentStatus=s;
			synchronized (this) {
				notifyAll();
			}
			return true;
		}
		return false;
	}
	public String toString()
	{
	    String ret=super.toString();
	    for(int i=0;i<=Constants.Priorities.MIN;i++)
			ret+="\nCoda a livello "+i+" Richieste "+queues[i].size();
	    return ret;
	}
}
