package JSR.ServiceManagers;
import JSR.*;
import JSR.Exceptions.JSRServiceNotAvaliable;

import java.io.IOException;
import java.rmi.RemoteException;
import java.util.*;
/**
 * Classe astratta che implementa alcune funzioni base per la gestione di un servizio
 * replicato. 
 * @author Mattia Righini Mat 170738
 */
public abstract class AServiceManager{
	/**
	 * IReplicationManager relativo all'oggetto corrente. 
	 */
	protected IReplicationManager parentManager;
	/**
	 * Provider Attivo per il servizio.
	 */
	protected IReplicableService activeProvider;
	/**
	 * Stato corrente del servizio.
	 */
	protected Status serviceCurrentStatus;
	/**
	 * Interfaccia del servizio.
	 */
	protected Interface serviceInterface;
	/**
	 * Nome del servizio.
	 */
	protected String serviceName;
	/**
	 * Vettore che mantiene i provider inattivi pronti a sostituire quello attivo in caso
	 * di malfunzionamento.
	 */
	protected ProviderVector inactiveProviders;
	/**
	 * Nel caso in cui non sia stato possibile effettuare una unLock a causa dell'assenza
	 * di provider questa variabile viene settata a true, la unlock verr eseguita 
	 * appena un nuovo provider andr a registrarsi.
	 */
	protected boolean deferedUnlock;
	/**
	 * Costruttore base.
	 * @param serviceName Nome del servizio.
	 * @param serviceInterface Interfaccia del servizio.
	 * @param parentManager Manager di appartenenza.
	 */
	protected AServiceManager(String serviceName,Interface serviceInterface,IReplicationManager parentManager)
	{
		this.serviceName=serviceName;
		this.serviceInterface=serviceInterface;
		this.parentManager=parentManager;
		activeProvider=null;
		serviceCurrentStatus=null;
		inactiveProviders=new ProviderVector();
		deferedUnlock=false;
	}
	//Funzioni Astratte da Implementare per realizzare un ServiceManager Concreto.
	/**
	 * Metodo per inviare una richiesta al provider correntemente attivo. Nel caso in cui
	 * non ci sia un provider attivo l'implementazione deve provvedere a invocare il metodo
	 * {@link AServiceManager#Activate() Activate()} al fine di riattivare il servizio e 
	 * eventualmente gestire l'impossibilit dell'erogazione del servizio lanciando un'opportuna
	 * eccezzione.
	 * @param r Richiesta da inviare al provider.
	 * @param a Account richiedente.
	 * @return La risposta ottenuta dal provider.
	 * @throws RemoteException
	 */
	public abstract Object SendRequest(Request r,Account a)throws RemoteException;
	/**
	 * Metodo per richidere il lock del servizio.Nel caso in cui
	 * non ci sia un provider attivo l'implementazione deve provvedere a invocare il metodo
	 * {@link AServiceManager#Activate() Activate()} al fine di riattivare il servizio e 
	 * eventualmente gestire l'impossibilit dell'erogazione del servizio lanciando un'opportuna
	 * eccezzione.
	 * @param a Account richiedente
	 * @return il risultato dell'operazione, true se il lock  stato ottenuto, false altrimenti.
	 * @throws RemoteException
	 */
	public abstract boolean lock(Account a)throws RemoteException;
	/**
	 * Metodo per rilasciare il lock del servizio.Nel caso in cui
	 * non ci sia un provider attivo l'implementazione deve provvedere a invocare il metodo
	 * {@link AServiceManager#Activate() Activate()} al fine di riattivare il servizio e 
	 * eventualmente gestire l'impossibilit dell'erogazione del servizio lanciando un'opportuna
	 * eccezzione. 
	 * @param a Account che detiene il lock.
	 * @return il risultato dell'operazione, true se il lock  stato rilasciato, false se l'account non deteneva il lock sul servizio.
	 * @throws RemoteException
	 */
	public abstract boolean unLock(Account a)throws RemoteException;
	//Fine funzioni Astratte
	/**
	 * Metodo per aggiungere un provider inattivo alla lista dei provider inattivi disponibili.
	 * @param provider Provider da aggiungere.
	 * @return true se l'inserimento ha avuto successo.
	 */
	public boolean addInactiveProvider(IReplicableService provider)
	{
		return inactiveProviders.addProvider(provider);
	}
	/**
	 * Metodo per aggiungere un array di provider inattivi.
	 * @param providers Array di provider da aggiungere.
	 */
	public void addInactiveProviders(IReplicableService[] providers)
	{
		for(int i=0;i<providers.length;i++)
			addInactiveProvider(providers[i]);
	}
	/**
	 * Metodo per rimuovere un provider indipendentemente dal fatto che sia attivo o meno.
	 * @param provider Provider da rimuovere. Nel caso venga rimosso il provider attivo
	 * la riattivazione non  immediata ma avverr alla prossima richiesta per il servizio.
	 */
	public void removeProvider(IReplicableService provider)
	{
		if(provider.equals(activeProvider))
		{
			activeProvider=null;
		}
		else
		{
			inactiveProviders.remove(provider);
		}
	}
	/**
	 * Metodo per verificare se un servizio  potenzialmente disponibile.
	 * @return True se esiste un provider attivo o esiste almeno un provider inattivo, false altrimenti.
	 */
	public boolean isServiceAvaliable(){return (activeProvider!=null) ||(!inactiveProviders.isEmpty());}
	/**
	 * Metodo per effettuare una sostituzione dello stato corrente.
	 * Non vengono effettuate verifiche sul grado di aggiornamento dello stato passato 
	 * in ingresso rispetto a quello corrente. Metodo da utilizzare in caso si voglia impporre
	 * un determinato stato al servizio.
	 * @param s Stato da impostare.
	 */
	public void replaceStatus(Status s)
	{
	    serviceCurrentStatus=s;
	}
	/**
	 * Metodo per aggiornare eventualmente lo stato corrente con uno stato passato in ingresso.
	 * Lo stato viene aggiornato solo se lo stato passato in ingresso  pi recente di quello 
	 * presente.
	 * @param s Stato da impostare.
	 * @return true se lo Stato  stato aggionrato false altrimenti
	 */
	public boolean updateStatus(Status s)
	{
		if((serviceCurrentStatus==null)||(s.getStatusCounter()>=serviceCurrentStatus.getStatusCounter())){
			serviceCurrentStatus=s;
			return true;
		}
		return false;
	}
	/**
	 * Sincronizza lo stato tra il manager e il provider passato in ingresso.
	 * Se remote=false l'aggiornamento viene eventualmente fatto solo sul manager.
	 * Se remote=true l'aggiornamento viene eventualmente fatto anche sul provider
	 * @param provider Provider da sincronizzare
	 * @param currentProviderStatus Stato presente sul provider passato in ingresso, se non viene specificato (null) lo stato verr richiesto al provider.
	 * @param remote indica se l'aggiornamento deve essere bidirezionale o meno
	 * @throws RemoteException
	 */
	public void synchronizeStatus(IReplicableService provider,Status currentProviderStatus,boolean remote)throws RemoteException
	{
	    Status s=null;
		if(currentProviderStatus==null)s=provider.getStatus();
		else s=currentProviderStatus;
		if((serviceCurrentStatus==null)||(s.getStatusCounter()>=serviceCurrentStatus.getStatusCounter()))
			serviceCurrentStatus=s;
		else
			if(remote)
				provider.setStatus(serviceCurrentStatus,parentManager);		
	}
	/**
	 * Metodo per attivare o riattivare un servizio, viene prima verificata la presenza
	 * e la funzionalit del provider attivo, in caso sia assente o non funzionante si procede
	 * all'attivazione del provider inattivo a Id pi basso fino a che un l'attivazione 
	 * non ha buon fine o non vi sono pi provider a disposizione.
	 */
	synchronized public void Activate()
	{
		Configuration.DebugOutput("AServiceManager.Activate()");				
		if(activeProvider!=null)
		{
			try{
				if(activeProvider.alive()){
					synchronizeStatus(activeProvider,null,true);
					return;
				}
			}catch(IOException e)
			{
				activeProvider=null;
			}
		}
		while((activeProvider==null)&&(!inactiveProviders.isEmpty()))
		{
			try
			{
				IReplicableService current=(IReplicableService)inactiveProviders.firstElement();
				inactiveProviders.remove(current);
				activeProvider=current;
				synchronizeStatus(activeProvider,null,true);
				current.activate(parentManager);
				break;
			}catch(IOException e)
			{
				continue;
			}
		}
	}
	/**
	 * Metodo per ottenere il provider correntemente attivo.
	 * @return Provider correntemente attivo.
	 */
	public IReplicableService getActiveProvider(){
		return activeProvider;
	}
	/**
	 * Metodo per ottenere il numero di provider inattivi presenti per il servizio.
	 * @return Numero di Provider inattivi.
	 */
	public int getNumberofInactiveProvider(){return inactiveProviders.size();}
	/**
	 * Metodo per ottenere l'array dei provider inattivi attualmente presenti per il 
	 * servizio.
	 * @return Array dei provider inattivi.
	 */
	public IReplicableService [] getInactiveProviders()
	{
		IReplicableService []ret=new IReplicableService[inactiveProviders.size()];
		for(int i=0;i<inactiveProviders.size();i++)
			ret[i]=(IReplicableService)inactiveProviders.get(i);
		return ret;
	}
	/**
	 * Verifica la necessit di effettuare una unLock differita nel tempo.<br/>
	 * Contestualmente viene resettata questa indicazione poich l'unlock  automatico
	 * se questo metodo restituisce true.
	 * @return True se c' la necessit di effettuare una unLock differita, false altrimenti.
	 */
	public boolean needToPerformDeferedUnlock()
	{
		if(deferedUnlock){
			deferedUnlock=false;
			return true;
		}
		return false;
	}
	/**
	 * Imposta l'active provider per il servizio, eventuali provider inattivi con id inferiore a quello
	 * del nuovo provider attivo vengono rimossi (si assume che non siano pi funzionanti, altrimenti
	 * sarebbero stati attivati loro dal momento che l'attivazione procede in modo ordinato con il valore
	 * crescente dell'id). Viene effettuata una sincronizzazione dello stato tra il provider e il manager.
	 * Lo stato presente sul provider (fornito attraverso il parametro <code>currentProviderStatus</code> o ottenuto 
	 * dal provider) va a sostituire se pi recente quello del manager, in caso contrario
	 * non si hanno aggiornamenti sul provider.
	 * @param provider Provider che diventa l'activeProvider per il servizio
	 * @param currentProviderStatus Status fornito dal provider se la chiamata al metodo si ha in seguito 
	 * ad una notifyActivation oppure null se l'attivazione  stata decisa a livello locale
	 */
	synchronized public void setActiveProvider(IReplicableService provider,Status currentProviderStatus)
	{
		Configuration.DebugOutput("AServiceManager.setActiveProvider("+provider+")");
		if(provider.equals(activeProvider))return;
		inactiveProviders.removeAllBefore(provider);
		activeProvider=provider;
		try{
			synchronizeStatus(provider,currentProviderStatus,false);
		}catch(RemoteException e){}
	}
	/**
	 * Metodo per deregistrare l' IReplicationManager relativo al manager.
	 */
	synchronized public void unRegisterManager() throws RemoteException
	{
		if(activeProvider==null)Activate();
		while(activeProvider!=null)
		{
			try
			{
				activeProvider.unRegisterReplicationManager(parentManager);
				return;
			}catch(IOException e)
			{
				Activate();
			}
		}
		throw new JSRServiceNotAvaliable(getServiceName());
	}
	/**
	 * Metodo per registrare l'IReplicationManager parent presso l'active provider.
	 * <br/>Metodo utilizzato dal {@link ServiceBroker#ServiceBroker(String, IReplicationManager) costruttore di ServiceBroker}
	 *  per registrarsi ai servizi presenti sul Manager master.
	 * @throws RemoteException
	 */
	synchronized public void registerManager() throws RemoteException
	{
		if(activeProvider==null)Activate();
		while(activeProvider!=null)
		{
			try
			{
				activeProvider.registerReplicationManager(parentManager);
				return;
			}catch(IOException e)
			{
				Activate();
			}
		}
		throw new JSRServiceNotAvaliable(getServiceName());	    
	}
	/**
	 * Metodo che restituisce l'interfaccia del servizio.
	 * @return Interfaccia del servizio.
	 */
	public Interface getServiceInterface(){return serviceInterface;}
	/**
	 * Metodo che restituisce il nome del servizio gestito.
	 * @return Nome del servizio-
	 */
	public String getServiceName(){return serviceName;}
	/**
	 * Metodo per impostare l'unLock differito.
	 */
	public void setDeferedUnlock(){deferedUnlock=true;}
	/**
	 * Metodo che restituisce lo Stato corrente del servizio.
	 * @return Stato corrente del servizio.
	 */
	public Status getServiceStatus(){return serviceCurrentStatus;}
	public String toString()
	{
	    String ret="Service :"+serviceName;
	    ret+="\n"+serviceInterface;
	    ret+="\nActive Provider="+activeProvider;
	    ret+="\n# Inactive Providers="+inactiveProviders.size();
	    ret+="\nCurrent Status\n"+serviceCurrentStatus;
	    return ret;
	}	
}
/**
 * Vettore per la gestione dei Provider inattivi, i provider sono ordinati per Id.
 * @author Mattia Righini Mat 170738
 */
class ProviderVector extends Vector
{
	public ProviderVector(){super();}
	public boolean addProvider(IReplicableService provider)
	{
		if(contains(provider))return false;
		try{
			int id=provider.getId();
			for(int i=0;i<size();i++)
			{
				try{
					int currentId=((IReplicableService)get(i)).getId();
					if(id>=currentId)
					{
						add(i+1,provider);
						return true;
					}
				}catch(IOException e)
				{
					remove(i);
				}
			}
			return add(provider);
		}catch(Exception e)
		{
			return false;
		}
	}
	public void removeAllBefore(IReplicableService provider)
	{
		int i=indexOf(provider);
        if(i==0)remove(0);
        else
        {    
            if(i>0)
            {
                removeRange(0,i);
                remove(0);
            }
        }
	}
}