package JSR;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import java.rmi.*;
import java.rmi.server.UnicastRemoteObject;

import JSR.ServiceManagers.*;
import JSR.Events.*;
import JSR.Exceptions.*;
import java.sql.*;
import java.util.Vector;

/**
 * Classe che implementa le interfaccie IReplicationManager e IServiceBroker.
 * @author Mattia Righini Mat 170738
 */
public class ServiceBroker extends UnicastRemoteObject implements
		IReplicationManager, IServiceBroker {
	/**
	 * Vettore che mantiene elementi di classe ActiveAccountDescriptor.
	 * Contestualmente al Login da parte di un Account viene creato un ActiveAccountDescriptor e
	 * viene aggiunto a questo vettore, al Logout viene rimosso.
	 */
    protected Vector activeAccounts;
	/**
	 * Lista dei servizi presenti sul Manager
	 */
	protected ServiceList serviceList;
	/**
	 * Vettore che mantiene i AServiceManager 
	 */
	protected Vector serviceManagers;
	/**
	 * Array di AccountInfo dal quale vengono prelevate le informazioni per caratterizzare gli account.
	 */
	protected AccountInfo[] accountsInfo;
	/**
	 * Array di ServiceInfo dal quale vengono prelevate le informazioni per caratterizzare i servizi.
	 */
	protected ServiceInfo[] servicesInfo;
	/**
	 * Vettore che mantiene oggetti di classe JSR.Events.AccountListener.
	 * Al verificarsi di un evento riguardante gli account tutti gli oggetti presenti nel vettore vengono
	 * notificati.
	 */
	protected Vector accountListeners;
	/**
	 * Vettore che mantiene oggetti di classe JSR.Events.ServiceListener.
	 * Al verificarsi di un evento riguardante i servizi tutti gli oggetti presenti nel vettore vengono
	 * notificati.
	 */	
	protected Vector serviceListeners;
	/**
	 * ServiceManager per il servizio base di generazione degli id.
	 */
	protected QueueLessServiceManager idGenerator;	
	/**
	 * Costruttore base.
	 * Effettua le istanzazioni dei vari oggetti e tenta una connessione al DB specificato per popolare
	 * gli array accountsInfo[] e servicesInfo[]. Al momento  supportata la connessione con sorgenti di tipo
	 * ODBC, per le quali la stringa di connessione  jdbc:odbc:DSName.
	 * @param dbConnectString Stringa di connessione per il DB odbc
	 * @throws RemoteException
	 */
	protected ServiceBroker(String dbConnectString)throws RemoteException
	{
		super();
		activeAccounts=new Vector();
		serviceManagers=new Vector();
		serviceList=new ServiceList();
		accountListeners=new Vector();
		serviceListeners=new Vector();
		try{
			//Connessione al DB
		    Vector temp=new Vector();
	        Class.forName ("sun.jdbc.odbc.JdbcOdbcDriver");
	        Connection con=DriverManager.getConnection(dbConnectString);
	        ResultSet accounts=con.createStatement().executeQuery("Select * from Accounts");
	        while(accounts.next())
	            temp.add(new AccountInfo(accounts.getString("accountid"),
	                    accounts.getInt("basepriority"),accounts.getInt("permissionmask")));
	        accounts.close();
	        accountsInfo=new AccountInfo[temp.size()];
	        for(int i=0;i<temp.size();i++)
	            accountsInfo[i]=(AccountInfo)temp.get(i);
	        temp.removeAllElements();
	        ResultSet services=con.createStatement().executeQuery("Select * from Services");
	        while(services.next())
	            temp.add(new ServiceInfo(services.getString("servicename"),services.getString("managertype")));
	        servicesInfo=new ServiceInfo[temp.size()];
	        for(int i=0;i<temp.size();i++)
	            servicesInfo[i]=(ServiceInfo)temp.get(i);
	        services.close();
	        con.close();
		}catch(Exception e)
		{
		    Configuration.DebugOutput(e.toString());
		    accountsInfo=new AccountInfo[0];
		    servicesInfo=new ServiceInfo[0];
		}
	}
	/**
	 * Costruttore che initializza un ServiceBroker e lo collega al servizio di generazione degli ID passato
	 * in ingresso. La creazione pu fallire se il provider richiesto presenta malfunzionamenti o non supporta
	 * le operazioni necessarie. Tenta una connessione al DB specificato per popolare
	 * gli array accountsInfo[] e servicesInfo[].Al momento  supportata la connessione con sorgenti di tipo
	 * ODBC, per le quali la stringa di connessione  jdbc:odbc:DSName.
	 * @param dbConnectString Stringa di connessione DB odbc
	 * @param idGenerator riferimento ad un provider per il servizio di generazione degli ID.
	 * @throws RemoteException
	 */
	public ServiceBroker(String dbConnectString,IReplicableService idGenerator)throws RemoteException
	{
		this(dbConnectString);
		this.idGenerator=new QueueLessServiceManager("GlobalIdGenerator",idGenerator.getInterface(),this);
		this.idGenerator.setActiveProvider(idGenerator,idGenerator.getStatus());
		Interface gidg=this.idGenerator.getServiceInterface();
		//Verifica della presenza del metodo int getNextId()
		Method []rm=gidg.getMethodsByName("getNextId");
		if((rm==null) ||(rm[0].getReturnType().equals(int.class.getClass())))
		    throw new JSRGenericException("IdGenerator not valid");
		this.idGenerator.registerManager();
	}
	/**
	 * Costruttore che inizializza il ServiceBroker clonando un IReplicatonManager il cui riferimento  passato
	 * in ingresso, Vengono importati tutti i servizi attualmente dispoinibili sul manager. Tenta una connessione al DB specificato per popolare
	 * gli array accountsInfo[] e servicesInfo[].Al momento  supportata la connessione con sorgenti di tipo
	 * ODBC, per le quali la stringa di connessione  jdbc:odbc:DSName.
	 * @param dbConnectString Stringa di connessione DB odbc
	 * @param master Riferimento ad un IReplicationManager in funzione
	 * @throws RemoteException
	 */
	public ServiceBroker(String dbConnectString,IReplicationManager master)throws RemoteException
	{
		this(dbConnectString);
		ServiceList masterServiceList=master.getServiceList();		
		//Collegamento a GlobalId Generator
		ServiceListElement gi=masterServiceList.getService("GlobalIdGenerator");
		if(gi==null)throw new JSRGenericException("Global Id Not found");
		this.idGenerator=new QueueLessServiceManager("GlobalIdGenerator",gi.getServiceInterface(),this);
		IReplicableService idg=master.getActiveProvider("GlobalIdGenerator");
		if(idg==null)throw new JSRGenericException("GlobalIdGenerator not avaliable");
		this.idGenerator.setActiveProvider(idg,idg.getStatus());
		this.idGenerator.addInactiveProviders(master.getInactiveProviders("GlobalIdGenerator"));
		//Invio la richiesta di registrazione al GlobalIdGenerator
		try{
			this.idGenerator.registerManager();
		}catch(RemoteException e)
		{
			throw new JSRGenericException("GlobalIdGenerator not avaliable");
		}
		//Clono il resto dei servizi
		for(int i=0;i<masterServiceList.getServiceCount();i++)
		{
			ServiceListElement current=masterServiceList.getService(i);
			if(!current.getServiceName().equals("GlobalIdGenerator"))
			{
			    AServiceManager sm;
			    ServiceInfo si=findServiceInfo(current.getServiceName());
			    if((si==null)||(!si.getManagerType().equals("QueuedServiceManager")))
			        sm=new QueueLessServiceManager(current.getServiceName(),current.getServiceInterface(),this);
			    else
			        sm=new QueuedServiceManager(current.getServiceName(),current.getServiceInterface(),this,QueuedServiceManager.WAIT_THRESHOLD);
				IReplicableService currentProvider=null;
				try{
					currentProvider=master.getActiveProvider(current.getServiceName());
					if(currentProvider!=null)
					{
							serviceManagers.add(sm);
							sm.setActiveProvider(currentProvider,currentProvider.getStatus());
							sm.addInactiveProviders(master.getInactiveProviders(current.getServiceName()));
							currentProvider.registerReplicationManager(this);				
							serviceList.addService(current.getServiceName(),current.getServiceInterface());					
					}
				}catch(Exception e){
					if(currentProvider!=null)
						serviceManagers.remove(sm);
					continue;
				}
			}
		}
	}
	/**
	 * Metodo per ottenere un id libero dal generatore di ID
	 * @return ID utilizzabile per un nuovo provider.
	 * @throws RemoteException
	 */
	protected int getNextId()throws RemoteException
	{
		
		Configuration.DebugOutput("ServiceBroker.getNextId()");
		Interface gidg=idGenerator.getServiceInterface();
		Method []gni=gidg.getMethodsByName("getNextId");
		if(gni==null)
			throw new JSRGenericException("IdGenerator not valid");
		Object[] arg=new Object[0];
		int ret=((Integer)idGenerator.SendRequest(new Request(gni[0],arg),null)).intValue();
		return ret;
	}	
	/**
	 * Metodo per ottenere il ServiceManager di un determinato servizio.
	 * @param serviceName nome del servizio di cui si vuole ottenere il Manager
	 * @return ServiceManager richiesto o null se il servizio non viene trovato.
	 */
	public AServiceManager findServiceManager(String serviceName)
	{
		if(serviceName.equals("GlobalIdGenerator"))return idGenerator;
		for(int i=0;i<serviceManagers.size();i++)
		{
			AServiceManager current=(AServiceManager)serviceManagers.get(i);
			if(current.getServiceName().equals(serviceName))return current;
		}
		return null;
	}
	/**
	 * Metodo per ottenere l'ActiveAccountDescriptor dell'Account con id passato in ingresso
	 * @param accountId Id dell'Account di cui si vuole il descrittore
	 * @return ActiveAccountDescriptor voluto o null se non viene trovato il descrittore.
	 */
	public ActiveAccountDescriptor findAccountDescriptor(String accountId)
	{
		for(int i=0;i<activeAccounts.size();i++)
		{
			ActiveAccountDescriptor current=(ActiveAccountDescriptor)activeAccounts.get(i);
			if(current.getAccount().getAccountId().equals(accountId))return current;
		}
		return null;
	}	
	/**
	 * Metodo per ottenere l'ActiveAccountDescriptor dell'Account passato in ingresso
	 * @param a Account di cui si vuole il descrittore
	 * @return ActiveAccountDescriptor voluto o null se non viene trovato il descrittore.
	 */
	public ActiveAccountDescriptor findAccountDescriptor(Account a)
	{
		for(int i=0;i<activeAccounts.size();i++)
		{
			ActiveAccountDescriptor current=(ActiveAccountDescriptor)activeAccounts.get(i);
			if(current.getAccount().equals(a))return current;
		}
		return null;
	}
	/**
	 * Metodo da eseguire prima dell'interruzione del Broker, viene effettuata la deregistrazione presso tutti i
	 * servizi.
	 */
	synchronized public void disableBroker()
	{
	    try{idGenerator.unRegisterManager();}
		catch(RemoteException e)
		{
		    Configuration.DebugOutput(e.toString());
		}
		for(int i=0;i<serviceManagers.size();i++)
		{
		    AServiceManager current=(AServiceManager)serviceManagers.get(i);
		    try{current.unRegisterManager();}
		    catch(RemoteException e)
		    {
		        Configuration.DebugOutput(e.toString());
		    }
		}
	}	
	protected AccountInfo findAccountInfo(String accountId)
	{
	    for(int i=0;i<accountsInfo.length;i++)
	        if(accountsInfo[i].getAccountId().equals(accountId))return accountsInfo[i];
	    return null;
	}
	protected ServiceInfo findServiceInfo(String serviceName)
	{
	    for(int i=0;i<servicesInfo.length;i++)
	        if(servicesInfo[i].getServiceName().equals(serviceName))return servicesInfo[i];
	    return null;
	}	
	//Implementazione dell'interfaccia IReplicationManager
	public RegisterResponse registerService(String serviceName, IReplicableService provider)
			throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.registerService("+
				serviceName+","+provider+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		int id=provider.getId();
		if(id<0)
			id=getNextId();
		AServiceManager sm=findServiceManager(serviceName);
		boolean activate=false;
		if(sm==null)
		{
		    ServiceInfo si=findServiceInfo(serviceName);
		    if((si==null)||(!si.getManagerType().equals("QueuedServiceManager")))
		        sm=new QueueLessServiceManager(serviceName,provider.getInterface(),this);
		    else
		        sm=new QueuedServiceManager(serviceName,provider.getInterface(),this,QueuedServiceManager.WAIT_THRESHOLD);
			serviceManagers.add(sm);
			serviceList.addService(sm.getServiceName(),sm.getServiceInterface());
			fireServiceEvent(new ServiceChangeEvent(serviceName,ServiceChangeEvent.SERVICE_ADDED));
			activate=true;
		}else
		{
		    //Controllo che l'interfaccia sia corretta.
		    if(!sm.getServiceInterface().equals(provider.getInterface()))
		    {
		        Configuration.DebugOutput("Provider per "+serviceName+" con interfaccia scorretta per il servizio");
		        return null;
		    }
		}
		if(sm.addInactiveProvider(provider)){
			Configuration.DebugOutput("Aggiunto nuovo Provider per "+serviceName);
			if((sm.getActiveProvider()==null)&&(sm.getNumberofInactiveProvider()==1))
				activate=true;
			fireServiceEvent(new ServiceChangeEvent(serviceName,ServiceChangeEvent.SERVICE_CHANGED));
		}
		else
		    Configuration.DebugOutput("Fallita aggiunta nuovo Provider per "+serviceName);
		return new RegisterResponse(id,activate,sm.getServiceStatus());
	}
	public void unRegisterService(String serviceName,
			IReplicableService provider) throws RemoteException{
		Configuration.DebugOutput("ServiceBroker.unRegisterService("+
				serviceName+","+provider+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if(sm==null)throw new JSRServiceNotFound(serviceName);
		sm.removeProvider(provider);
	}
	public void notifyActivation(String serviceName,
			IReplicableService provider,Status currentStatus) throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.notifyActivation("+
				serviceName+","+provider+","+currentStatus+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if(sm==null)throw new JSRServiceNotFound(serviceName);
		sm.setActiveProvider(provider,currentStatus);
		fireServiceEvent(new ServiceChangeEvent(serviceName,ServiceChangeEvent.SERVICE_CHANGED));
	}
	public void updateStatus(String serviceName, IReplicableService provider,Status newStatus) throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.updateStatus("+
				serviceName+","+provider+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if(sm==null)throw new JSRServiceNotFound(serviceName);
		if(!(sm.getActiveProvider().equals(provider)))throw new JSRProviderNotActive(provider);
		if(sm.updateStatus(newStatus))
			fireServiceEvent(new ServiceChangeEvent(serviceName,ServiceChangeEvent.SERVICE_CHANGED));
		else throw new JSRGenericException("Status is older than the Current Status held by the Manager");
	}
    public void resetStatus(String serviceName,IReplicableService provider,Status s) throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.resetStatus("+
				serviceName+","+provider+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if(sm==null)throw new JSRServiceNotFound(serviceName);
		if(!(sm.getActiveProvider().equals(provider)))throw new JSRProviderNotActive(provider);
		sm.replaceStatus(s);
		fireServiceEvent(new ServiceChangeEvent(serviceName,ServiceChangeEvent.SERVICE_CHANGED));
    }
	public Status getCurrentStatus(String serviceName) throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.getCurrentStatus("+
				serviceName+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if(sm==null)throw new JSRServiceNotFound(serviceName);
		if(sm.getActiveProvider()==null && sm.isServiceAvaliable())sm.Activate();
		return sm.getServiceStatus();
	}
	public boolean canRemoveLock(String serviceName, IReplicableService provider) throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.canRemoveLock("+
				serviceName+","+provider+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if(sm==null)throw new JSRServiceNotFound(serviceName);
		if(!(sm.getActiveProvider().equals(provider)))throw new JSRProviderNotActive(provider);
		return sm.needToPerformDeferedUnlock();
	}
	public IReplicableService[] getInactiveProviders(String serviceName) throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.getInactiveProvider("+
				serviceName+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if(sm==null)throw new JSRServiceNotFound(serviceName);
		return sm.getInactiveProviders();
	}
	public IReplicableService getActiveProvider(String serviceName) throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.getActiveProvider("+
				serviceName+")");
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if(sm==null)throw new JSRServiceNotFound(serviceName);
		if(sm.getActiveProvider()==null && sm.isServiceAvaliable())sm.Activate();
		return sm.getActiveProvider();
	}	
	public ServiceList getServiceList() throws RemoteException {
		ServiceList ret=new ServiceList(serviceList);
		ret.addService(idGenerator.getServiceName(),idGenerator.getServiceInterface());
		return ret;
	}
	//Fine Implementazione dell'interfaccia IReplicationManager
	//Implementazione dell'interfaccia IServiceBroker
	public Account logIn(String accountId) throws RemoteException {
		int seed=(int)(Math.random()*Integer.MAX_VALUE);
		Configuration.DebugOutput("ServiceBroker.logIn("+
				accountId+") seed="+seed+")");
		int prio=Constants.Priorities.NORMAL;
		int perm=Constants.Permission.ALL;
		AccountInfo ai=findAccountInfo(accountId);
		if(ai!=null)
		{
		    prio=ai.getBasePriority();
		    perm=ai.getPermissionMask();
		}
		Account newAccount=new Account(accountId,seed,prio,perm,this);
		ActiveAccountDescriptor ad=new ActiveAccountDescriptor(newAccount,this);
		activeAccounts.add(ad);
		fireAccountEvent(new AccountsStateChangeEvent(newAccount,AccountsStateChangeEvent.ACCOUNT_LOGIN));
		return newAccount;
	}
	public void logOut(Account account) throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.logOut("+
				account+")");
		ActiveAccountDescriptor ad=findAccountDescriptor(account);
		if(ad!=null)
		{
			String[] lockedService=ad.getLockedService();
			for(int i=0;i<lockedService.length;i++){
				try{
					unLockService(lockedService[i],account);
				}catch(RemoteException e)
				{
					Configuration.DebugOutput(e.toString());
				}
			}
			activeAccounts.remove(ad);
			fireAccountEvent(new AccountsStateChangeEvent(account,AccountsStateChangeEvent.ACCOUNT_LOGOUT));
		}else
			throw new JSRAccountNotValid(account);
		
	}
	public ServiceList getServiceList(Account account) throws RemoteException {
		ActiveAccountDescriptor ad=findAccountDescriptor(account);
		if(ad!=null)
			ad.resetTimer();
		else
			throw new JSRAccountNotValid(account);
		if(!ad.verifyPermission(Constants.Permission.GET_SERVICELIST))
			throw new JSROperationNotAllowed(Constants.Permission.GET_SERVICELIST);		
		return serviceList;		
	}	
	public Interface getServiceInterface(String serviceName,Account account)
			throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.getServiceInterface("+
				serviceName+","+account+")");
		ActiveAccountDescriptor ad=findAccountDescriptor(account);
		if(ad!=null)
			ad.resetTimer();
		else
			throw new JSRAccountNotValid(account);
		if(!ad.verifyPermission(Constants.Permission.QUERY_INTERFACE))
			throw new JSROperationNotAllowed(Constants.Permission.QUERY_INTERFACE);		
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if((sm==null)||(serviceName.equals("GlobalIdGenerator")))throw new JSRServiceNotFound(serviceName);
		return sm.getServiceInterface();
	}
	public Object sendRequest(String serviceName,Request request,Account account)
			throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.sendRequest("+serviceName+
				","+request+","+account+")");
		ActiveAccountDescriptor ad=findAccountDescriptor(account);
		if(ad!=null)
			ad.resetTimer();
		else
			throw new JSRAccountNotValid(account);
		if(!ad.verifyPermission(Constants.Permission.SEND_REQUEST))
			throw new JSROperationNotAllowed(Constants.Permission.SEND_REQUEST);
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if((sm==null)||(serviceName.equals("GlobalIdGenerator")))throw new JSRServiceNotFound(serviceName);
		if(sm.isServiceAvaliable()){
			return sm.SendRequest(request,account);
		}
		else
			throw new JSRServiceNotAvaliable(serviceName);
	}
	public boolean lockService(String serviceName, Account account)
			throws RemoteException {
		Configuration.DebugOutput("ServiceBroker.lockService("+serviceName+
				","+account+")");
		ActiveAccountDescriptor ad=findAccountDescriptor(account);
		if(ad!=null)ad.resetTimer();
		else throw new JSRAccountNotValid(account);
		if(!ad.verifyPermission(Constants.Permission.LOCK))
			throw new JSROperationNotAllowed(Constants.Permission.LOCK);
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if((sm==null)||(serviceName.equals("GlobalIdGenerator")))throw new JSRServiceNotFound(serviceName);
		if(sm.isServiceAvaliable()){
			if(sm.lock(account))
			{
				ad.addLock(serviceName);
				ad.resetTimer();
				fireAccountEvent(new AccountsStateChangeEvent(account,AccountsStateChangeEvent.ACCOUNT_UPDATE));
				fireServiceEvent(new ServiceChangeEvent(serviceName,ServiceChangeEvent.SERVICE_CHANGED));
				return true;
			}
			return false;
		}
		else
			throw new JSRServiceNotAvaliable(serviceName);
	}
	public boolean unLockService(String serviceName, Account account)
			throws RemoteException {
		ActiveAccountDescriptor ad=findAccountDescriptor(account);
		if(ad!=null)ad.resetTimer();
		else throw new JSRAccountNotValid(account);
		Configuration.DebugOutput("ServiceBroker.unLockService("+serviceName+
				","+account+")");
		if(!ad.verifyPermission(Constants.Permission.LOCK))
			throw new JSROperationNotAllowed(Constants.Permission.LOCK);		
		if(serviceName==null)throw new JSRGenericException("Service Name not Set");
		AServiceManager sm=findServiceManager(serviceName);
		if((sm==null)||(serviceName.equals("GlobalIdGenerator")))throw new JSRServiceNotFound(serviceName);
		if(sm.isServiceAvaliable()){
			try{
				if(sm.unLock(account))
				{
					ad.removeLock(serviceName);
					ad.resetTimer();
					fireAccountEvent(new AccountsStateChangeEvent(account,AccountsStateChangeEvent.ACCOUNT_UPDATE));
					fireServiceEvent(new ServiceChangeEvent(serviceName,ServiceChangeEvent.SERVICE_CHANGED));					
					return true;
				}
				return false;
			}catch(JSRServiceNotAvaliable e)
			{ 
				sm.setDeferedUnlock();
				Configuration.DebugOutput("Impossibile effettuare unLock "+serviceName);
				return false;
			}
		}
		else
			throw new JSRServiceNotAvaliable(serviceName);
	}
	//Fine Implementazione dell'interfaccia IServiceBroker	
	public static void main(String[] args){	
		if(args.length<5)
		{
			System.out.println("Uso ServiceBroker <Indirizzo Broker> <debug> <ODBCDSName> <start> <indirizzo GeneratorId|indirizzo ReplicationManager da clonare>");
			System.exit(1);
		}
		Configuration.DEBUG=args[1].toLowerCase().equals("true");
		if(Configuration.DEBUG)
		    Configuration.addDebugListener(new JSR.Events.ConsoleDebug());
		if (System.getSecurityManager() == null)
		{ 
			System.setSecurityManager(new java.rmi.RMISecurityManager());
		}
		boolean start=args[3].equalsIgnoreCase("true");
		ServiceBroker server=null;
		if(start){
			IReplicableService idgen=null;
			try
			{
				idgen=(IReplicableService)Naming.lookup(args[4]);
			}catch(NotBoundException nbe)
			{
				Configuration.DebugOutput("Servizio di IdGenerator non trovato, creazione");
				try{
				    Runtime.getRuntime().exec("java -Djava.security.policy=SecPolicy.policy JSR.GlobalIdGenerator true true "+args[4]);
				    for(int i=0;i<10;i++){
				    	Thread.sleep(100);
				    	try{
				    		idgen=(IReplicableService)Naming.lookup(args[4]);
				    		break;
				    	}catch(NotBoundException nbe2)
						{
				    		continue;
						}
				    }
				    if(idgen==null)throw new Exception("Creazione Fallita.");
				}catch(Exception e)
				{
				    System.out.println(e);
				    System.exit(1);
				}
			}
			catch(Exception e)
			{
				System.err.println(e);
				System.exit(0);
			}
			try{
				server=new ServiceBroker("jdbc:odbc:"+args[2],idgen);
			}catch(Exception e)
			{
				System.err.println(e);
				System.exit(0);			
			}
		}else
		{
			IReplicationManager master=null;
			try
			{
				master=(IReplicationManager)Naming.lookup(args[4]);
				server=new ServiceBroker("jdbc:odbc:"+args[2],master);
			}catch(Exception e)
			{
				System.out.println(e);
				System.exit(1);
			}
		}
		ServiceBrokerFrame bf=new ServiceBrokerFrame(server,args[0]);
		bf.show();
	}	
	/**
	 * Metodo per aggiungere un listener per gli eventi riguardanti gli Account
	 * @param a Listener da aggiungere
	 */
	public void addAccountListener(AccountListener a){accountListeners.add(a);}
	protected void fireAccountEvent(AccountsStateChangeEvent e)
	{
	    for(int i=0;i<accountListeners.size();i++)
	        ((AccountListener)accountListeners.get(i)).AccountEvent(e);
	}
	/**
	 * Metodo per aggiungere un listener per gli eventi riguardanti i Servizi
	 * @param s Listener da aggiungere
	 */
	public void addServiceListener(ServiceListener s){serviceListeners.add(s);}
	protected void fireServiceEvent(ServiceChangeEvent e)
	{
	    for(int i=0;i<serviceListeners.size();i++)
	        ((ServiceListener)serviceListeners.get(i)).ServiceEvent(e);
	}
	/**
	 * Metodo per ottenere l'elenco degli AccountId attivi
	 * @return Ogni elemento dell'array  l'AccountId di un Account attivo sul Broker
	 */
	public String[] getActiveAccounts()
	{
	    String[] ret=new String[activeAccounts.size()];
	    for(int i=0;i<activeAccounts.size();i++)
	        ret[i]=((ActiveAccountDescriptor)activeAccounts.get(i)).getAccount().getAccountId();
	    return ret;
	}
}
/**
 * Classe che mantiene le politiche per un determinato Account identificato dall'id.
 * @author Mattia Rignini Mat 170738
 */
class AccountInfo
{
    protected String accountId;
    protected int basePriority;
    protected int permissionMask;
    public AccountInfo(String accountId,int basePriority,int permissionMask)
    {
        this.accountId=accountId;
        this.basePriority=basePriority;
        this.permissionMask=permissionMask;
    }
    public String getAccountId() {
        return accountId;
    }
    public int getBasePriority() {
        return basePriority;
    }
    public int getPermissionMask() {
        return permissionMask;
    }
}
/**
 * Classe che mantiene il nome del tipo di ServiceManager per un determinato Servizio identificato dal nome.
 * @author Mattia Righini Mat 170738
 */
class ServiceInfo
{
    protected String serviceName;
    protected String managerType;
    public ServiceInfo(String serviceName,String managerType)
    {
        this.serviceName=serviceName;
        this.managerType=managerType;
    }
    public String getManagerType() {
        return managerType;
    }
    public String getServiceName() {
        return serviceName;
    }
}
/**
 * Classe che gestisce un Account attivo sul Broker.
 * Mantiene l'istanza di Account, l'elenco dei servizi lockati e il Thread per il controllo del timeout
 * della sessione.
 * @author Mattia Righini Mat 170738
 */
class ActiveAccountDescriptor
{
	protected Account account;
	protected Vector lockedService;
	protected Thread sessionKiller;
	public ActiveAccountDescriptor(Account account,IServiceBroker broker)
	{
		this.account=account;
		lockedService=new Vector();
		sessionKiller=new Thread(new SessionKiller(broker,account));
		sessionKiller.start();
	}
	public void resetTimer()
	{
		sessionKiller.interrupt();
	}
	public Account getAccount(){return account;}
	public void addLock(String serviceName){lockedService.add(serviceName);}
	public void removeLock(String serviceName){lockedService.remove(serviceName);}
	public boolean verifyPermission(int request)
	{
		return (request&(account.getPermissionsMask()))!=0;
	}
	public String[] getLockedService()
	{
		String[] ret=new String[lockedService.size()];
		for(int i=0;i<lockedService.size();i++)
			ret[i]=(String)lockedService.get(i);
		return ret;
	}
	public boolean equals(ActiveAccountDescriptor ad)
	{
		return account.equals(ad.account);
	}
	public String toString()
	{
	    String ret="AccountId="+account.getAccountId();
	    ret+="\nBasePriority="+account.getBasePriority();
	    ret+="\nPermissionMask="+account.getPermissionsMask();
	    ret+="\nLocks:";
	    String[]lockeds=getLockedService();
	    for(int i=0;i<lockeds.length;i++)
	        ret+="\n\t"+lockeds[i];
	    return ret;
	}
}
/**
 * Thread per eseguire il logout forzato allo scadere della sessione
 * @author Mattia Righini Mat 170738
 */
class SessionKiller implements Runnable
{
	protected IServiceBroker broker;
	protected Account accountToKill;
	public SessionKiller(IServiceBroker broker,Account accountToKill)
	{
		this.broker=broker;
		this.accountToKill=accountToKill;
	}
	public void run() {
		while(true){
			try{
				Thread.sleep(Configuration.SESSIONTIMEOUT);
				Configuration.DebugOutput("Logout forzato per "+accountToKill);
				broker.logOut(accountToKill);
				break;
			}catch(InterruptedException e)
			{
				continue;
			}
			catch(RemoteException re)
			{
				break;
			}
		}
	}
}
/**
 * Interfaccia grafica per il ServiceBroker
 * @author Mattia Righini Mat 170738
 */
class ServiceBrokerFrame extends JFrame implements ListSelectionListener,WindowListener,DebugListener,
AccountListener,ServiceListener
{
    ServiceBroker broker;
    String bindAddress;
    JList serviceList;
    JTextArea serviceDetail;
    JList accountList;
    JTextArea accountDetail;
    JTextArea debugOutput;
    public ServiceBrokerFrame(ServiceBroker broker,String bindAddres)
    {
        super();
        this.bindAddress=bindAddres;
        this.broker=broker;
        try{
            Naming.bind(bindAddres,broker);
        }catch(Exception e)
        {
            System.out.println(e);
            System.exit(1);			
        }
        setTitle(bindAddres);
        getContentPane().setLayout(new GridLayout(3,1));
        //Area Servizi
        serviceList=new JList();
        serviceList.addListSelectionListener(this);
        serviceDetail=new JTextArea();
        serviceDetail.setEditable(false);
        Box b=Box.createHorizontalBox();
        b.setAlignmentX(Box.LEFT_ALIGNMENT);
        Box vb=Box.createVerticalBox();
        vb.setMaximumSize(new java.awt.Dimension(100,800));
        vb.add(new JLabel("Servizi"));
        JScrollPane sp=new JScrollPane(serviceList);
        sp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        sp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        vb.add(sp);
        b.add(vb);
        vb=Box.createVerticalBox();
        vb.add(new JLabel("Dettaglio Servizio"));
        sp=new JScrollPane(serviceDetail);
        sp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        sp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);        
        vb.add(sp);
        b.add(vb);
        getContentPane().add(b);
        //Area Account
        accountList=new JList();
        accountList.addListSelectionListener(this);
        accountDetail=new JTextArea();
        accountDetail.setEditable(false);
        b=Box.createHorizontalBox();
        b.setAlignmentX(Box.LEFT_ALIGNMENT);
        vb=Box.createVerticalBox();
        vb.setMaximumSize(new java.awt.Dimension(100,800));
        vb.add(new JLabel("Accounts"));
        sp=new JScrollPane(accountList);
        sp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        sp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        vb.add(sp);
        b.add(vb);
        vb=Box.createVerticalBox();
        vb.add(new JLabel("Dettaglio Account"));
        sp=new JScrollPane(accountDetail);
        sp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        sp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        vb.add(sp);
        b.add(vb);
        getContentPane().add(b);
        //Area Debug
        vb=Box.createVerticalBox();
        vb.add(new JLabel("Debug Window"));
        debugOutput=new JTextArea();
        debugOutput.setEditable(false);
        debugOutput.setBackground(Color.black);
        debugOutput.setForeground(Color.white);
        sp=new JScrollPane(debugOutput);
        sp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        sp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        vb.add(sp);
        getContentPane().add(vb);
        setExtendedState(JFrame.MAXIMIZED_BOTH);
        addWindowListener(this);
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        if(Configuration.DEBUG)
            Configuration.addDebugListener(this);
        this.broker.addAccountListener(this);
        this.broker.addServiceListener(this);
        try{
            ServiceList sl=broker.getServiceList();
            String[] sls=new String[sl.getServiceCount()];
            for(int i=0;i<sl.getServiceCount();i++)
                sls[i]=sl.getService(i).getServiceName();
            serviceList.setListData(sls);
        }catch(Exception e){}        
    }
    public void printDebugString(String s)
    {
        debugOutput.append("\n"+s);
    }
    public void valueChanged(ListSelectionEvent arg0) {
        if(arg0.getSource()==accountList)
        {
            if(accountList.getSelectedValue()==null)return;
            String accountId=(String)accountList.getSelectedValue();
            ActiveAccountDescriptor ad=broker.findAccountDescriptor(accountId);
            accountDetail.setText(ad.toString());
        }
        if(arg0.getSource()==serviceList)
        {
            if(serviceList.getSelectedValue()==null)return;
            String serviceName=(String)serviceList.getSelectedValue();
            AServiceManager sm=broker.findServiceManager(serviceName);
            serviceDetail.setText(sm.toString());
        } 
    }
    public void AccountEvent(AccountsStateChangeEvent event){
        if(event.Updated()&& (!accountList.isSelectionEmpty()))
        {
            String selectedAccount=(String)accountList.getSelectedValue();
            if(selectedAccount.equals(event.getAccount().getAccountId()))
            {
                accountDetail.setText(broker.findAccountDescriptor(selectedAccount).toString());
            }
        }else
        {
            accountList.clearSelection();
            accountDetail.setText("");
            accountList.setListData(broker.getActiveAccounts());
        }
    }
    public void ServiceEvent(ServiceChangeEvent event){
        if((event.getEvent()==ServiceChangeEvent.SERVICE_CHANGED)&&(!serviceList.isSelectionEmpty()))
        {
            String selectedService=(String)serviceList.getSelectedValue();
            if(selectedService.equals(event.getServiceName()))
            {
                serviceDetail.setText(broker.findServiceManager(selectedService).toString());
            }
        }else
        {
            serviceList.clearSelection();
            serviceDetail.setText("");
            try{
                ServiceList sl=broker.getServiceList();
                String[] sls=new String[sl.getServiceCount()];
                for(int i=0;i<sl.getServiceCount();i++)
                    sls[i]=sl.getService(i).getServiceName();
                serviceList.setListData(sls);
            }catch(Exception e){}
        }
    }
    public void windowActivated(WindowEvent arg0) {
    }
    public void windowClosed(WindowEvent arg0) {
    }
    public void windowClosing(WindowEvent arg0) {
        try{
            Naming.unbind(bindAddress);
        }catch(Exception e)
        {
            System.out.println("UNBIND non riuscita.");
        }
        broker.disableBroker();
    }
    public void windowDeactivated(WindowEvent arg0) {
    }
    public void windowDeiconified(WindowEvent arg0) {
    }
    public void windowIconified(WindowEvent arg0) {
    }
    public void windowOpened(WindowEvent arg0) {
    }
}