package JSR;

import java.rmi.Remote;
import java.rmi.RemoteException;

/**
 * <p>Interfaccia da implementare per realizzare un gestore di servizi replicati.
 * L'interfaccia viene usata dai Providers e dagli altri ReplicationManager
 * @author Mattia Righini Mat 170738
 */
public interface IReplicationManager extends Remote {
	//Metodi per i Providers
	/**
	 * Metodo invocato per registrare un servizio presso il manager.
	 * @param serviceName Nome del servizio da registrare
	 * @param provider Provider per il servizio.
	 * @return Risposta alla registrazione vedi {@link JSR.RegisterResponse RegisterResponse}, null nel caso in
	 * cui non sia possibile effettuare la registrazione a causa del fatto che sul manager esiste gi un servizio
	 * con il nome indicato ma con interfaccia diversa da quella proposta.
	 * @throws RemoteException
	 */
	RegisterResponse registerService(String serviceName,IReplicableService provider)throws RemoteException;
	/**
	 * Metodo invocato per rimuovere un provider da un servizio.
	 * @param serviceName Nome del servizio a cui fa riferimento il provider
	 * @param provider Provider che intende deregistrarsi
	 * @throws RemoteException
	 */
	void unRegisterService(String serviceName,IReplicableService provider)throws RemoteException;
	/**
	 * Metodo invocato dai Provider quando vengono attivati. L'attivazione del provider pu seguire 
	 * la registrazione nel caso in cui il provider sia l'unico disponibile per il servizio o nel caso in cui
	 * un manager rileva un malfunzionamento del provider attivo correntemente.
	 * @param serviceName Servizio di riferimento
	 * @param provider Provider che  stato attivato
	 * @param currentStatus Stato corrente del provider
	 * @throws RemoteException
	 */
	void notifyActivation(String serviceName,IReplicableService provider,Status currentStatus)throws RemoteException;
	/**
	 * Metodo incovcato dal provider attivo quando lo stato del servizio  cambiato ed  quindi
	 * necessario aggiornare la copia dello stato sul manager.
	 * Il Manager accetta aggiornamenti dello stato solo dal provieder correntemente attivo.
	 * @param serviceName Servizio di riferimento
	 * @param provider Provider che effettua l'update
	 * @param newStatus nuovo stato per il servizio
	 * @throws RemoteException
	 */
	void updateStatus(String serviceName,IReplicableService provider,Status newStatus)throws RemoteException;	
	/**
	 * Metodo invocato da un provider a seguito dell'attivazione e nel caso in cui il provider rilevi la presenza
	 * di un lock. Con questo metodo il provider richiede l'autorizzazione al manager a rimuovere il lock,  possibile
	 * che a seguito di una indisponibilit di provider per un servizio il manager non riesca a rimuovere il lock e quindi
	 * che questa operazione venga differita nel tempo.
	 * @param serviceName Servizio di riferimento
	 * @param provider Provider che richiede la rimozione del lock
	 * @return True se un operazione di lock  stata differita nel tempo e quindi il lock deve essere rimosso,
	 * False se il lock deve essere mantenuto.
	 * @throws RemoteException
	 */
	boolean canRemoveLock(String serviceName,IReplicableService provider)throws RemoteException;
	/**
	 * Operazione mediante la quale il provider attivo interroga il managr sullo stato del servizio. A seguito
	 * dell'attivazione il provider verifica lo stato del servizio presso tutti i manager a cui fa riferimento e
	 * nel caso vi siano delle inconsistenze setta lo stato del servizio alla versione pi recente e aggiorna lo 
	 * stato su tutti i manager
	 * @param serviceName Servizio di riferimento
	 * @return Stato corrente del servizio
	 * @throws RemoteException
	 */
	Status getCurrentStatus(String serviceName)throws RemoteException;
	/**
	 * Viene reimpostato lo stato del servizio senza effettuare controlli sllo stesso, metodo invocato se si
	 * vuole reimpostare un servizio gi presente su un manager.
	 * @param serviceName Servizio di riferimento
	 * @param provider Provider che effettua l'update 
	 * @param s Stato da impostare sul manager.
	 * @throws RemoteException
	 */
	void resetStatus(String serviceName,IReplicableService provider,Status s)throws RemoteException;	
	
	//Metodi per gli altri Replication Managers
	/**
	 * Metodo invocato da un altro ReplicationManager per ottenere il provider attivo per un determinato 
	 * servizio. Questo metodo insieme a {@link JSR.IReplicationManager#getInactiveProviders getInactiveProviders} 
	 * viene utilizzato dai manager in fase di creazione per clonare un manager esistente.
	 * @param serviceName Servizio di riferimento
	 * @return Provider correntemente attivo per il servizio o null se il servizio non  attualmente disponibile
	 * @throws RemoteException
	 */
	IReplicableService getActiveProvider(String serviceName)throws RemoteException;
	/**
	 * Metodo invocato da un altro ReplicationManger per ottenere l'insieme dei provider inattivi per un 
	 * determinato servizio.Questo metodo insieme a {@link JSR.IReplicationManager#getActiveProvider getActiveProvider} 
	 * viene utilizzato dai manager in fase di creazione per clonare un manager esistente.
	 * @param serviceName Servizio di riferimento
	 * @return vettore di provider inattivi gestiti dal manager per il servizio.
	 * @throws RemoteException
	 */
	IReplicableService[] getInactiveProviders(String serviceName)throws RemoteException;
	/**
	 * Metodo per ottenere la lista dei servizi accessibili dal manager.
	 * Vedere {@link JSR.ServiceList ServiceList} per maggiori informazioni.
	 * @return Lista dei servizi accessibili.
	 * @throws RemoteException
	 */
	ServiceList getServiceList()throws RemoteException;
}
