package JSR;

import java.rmi.Remote;
import java.rmi.RemoteException;

/**
 * <p>Interfaccia da implementare per realizzare un servizio replicabile.
 * @author Mattia Righini Mat: 170738
 */
public interface IReplicableService extends Remote {
	
	/**
	 * Metodo che verifica se il provider  vivo o meno
	 * @return true se il provider  vivo
	 * @throws RemoteException
	 */
	public boolean alive()throws RemoteException;
	/**
	 * Viene restituito l'id globale associato al provider
	 * @return Id del provider
	 * @throws RemoteException
	 */
	public int getId()throws RemoteException;
	/**
	 * Viene restituita l'interfaccia esposta dal provider.
	 * @return L'interfaccia del provider
	 * @throws RemoteException
	 */
	public Interface getInterface()throws RemoteException;
	/**
	 * Restituisce lo stato corrente del servizio
	 * @return Stato corrente del servizio
	 * @throws RemoteException
	 */
	public Status getStatus()throws RemoteException;
	/**
	 * Il provider viene lockato dall'account passato in ingresso, successivamente
	 * all'invocazione di questo metodo e fino alla chiamata del metodo unLock(Account)
	 * solo le invocazioni provenienti dall'account che ha eseguito la lock vengono
	 * servite, invocazioni da parte di altri account vengono sospese in attesa della
	 * Unlock
	 * @param lockAccount Account che vuole ottenere il lock
	 * @return true se il lock  stato settato
	 * @throws RemoteException
	 */
	public boolean lock(Account lockAccount)throws RemoteException;
	/**
	 * Viene inviata ed eseguita una richiesta di servizio
	 * @param r Richiesta da elaborare
	 * @param account Account richiedente
	 * @return la risposta alla richiesta
	 * @throws RemoteException
	 */
	public Object sendRequest(Request r,Account account)throws RemoteException;
	/**
	 * Il Provider viene attivato, per un dato servizio e un dato insieme di manager che
	 * condividono questo servizio esiste al pi un provider attivo. Al momento dell'attivazione
	 * del provider da parte di un manager il provider notifica agli altri manager l'attivazione
	 * @param activator Manager che ha attivato il provider
	 * @throws RemoteException
	 */
	public void activate(IReplicationManager activator)throws RemoteException;
	/**
	 * Effettua l'aggiornamento dello stato
	 * @param s Nuovo stato
	 * @param updater Manager che effettua l'update
	 * @throws RemoteException
	 */
	public void setStatus(Status s,IReplicationManager updater)throws RemoteException;
	/**
	 * Viene rimosso il lock dal provider se l'account corrisponde all'account che detiene
	 * il lock.
	 * @param lockAccount Account che vuole rimuovere il lock
	 * @return true se il lock  stato rimosso
	 * @throws RemoteException
	 */
	public boolean unLock(Account lockAccount)throws RemoteException;
	/**
	 * Aggiunge un nuovo ReplicationManager, metodo invocato dai ReplicationManager che replicano altri
	 * ReplicationManager o semplicemente quando intendono gestire un servizio gi esistente di cui 
	 * conoscono il riferimento.
	 * @param manager Nuovo manager da aggiungere ai manager a cui il provider  collegato.
	 * @throws RemoteException
	 */	
	public void registerReplicationManager(IReplicationManager manager)throws RemoteException;
	/**
	 * Il Manager passato in ingresso viene rimosso dai manager collegati al provider, non sar quindi pi 
	 * notificato al variare dello stato del servizio.
	 * @param manager Manager che intende essere rimosso.
	 * @throws RemoteException
	 */	
	public void unRegisterReplicationManager(IReplicationManager manager)throws RemoteException;
}
