package JSR;
import JSR.Exceptions.*;

import java.io.IOException;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
/**
 * Classe Astratta che fornisce le funzionalit base per la realizzazione di un Servizio Replicabile
 * @author Mattia Righini Mat: 170738
 */
public abstract class AReplicableService extends UnicastRemoteObject implements
		IReplicableService {
	/**
	 * Comment for <code>id</code> Codice univoco legato al provider del servizio, settato in fase di registrazione.
	 */
	protected int id;
	/**
	 * Comment for <code>active</code> Indica se il provider  attivo (cio i ReplicationManager inviano le richieste di
	 * servizio a questo provider per il servizio corrispondente).
	 */
	boolean active;
	/**
	 * Comment for <code>serviceStatus</code> Stato corrente del servizio mantenuto dal provider, in caso di provider attivo  anche lo stato del 
	 * servizio altrimenti  lo stato ricevuto in fase di registrazione e non necessariamente aggiornato.
	 */
	protected Status serviceStatus;
	/**
	 * Comment for <code>serviceName</code> Nome del servizio offerto.
	 */
	protected String serviceName;
	/**
	 * Comment for <code>serviceInterface</code> Interfaccia del servizio offerto.
	 */
	protected Interface serviceInterface;
	
	/*Funzioni astratte specifiche del servizio*/
	/**
	 * Restituisce il nome del servizio
	 * @return Nome del servizio
	 */
	abstract public String getServiceName();
	/**
	 * Costruisce e restituisce l'interfaccia del servizio.
	 * @return Interfaccia del servizio.
	 */
	abstract protected Interface buildInterface();
	/**
	 * Costruisce e restituisce lo stato iniziale del servizio.
	 * @return Stato iniziale.
	 */
	abstract protected Status buildInitialStatus();
	/**
	 * Indica se il metodo il cui nome  passato in ingresso  un metodo che modifica lo stato.
	 * @param methodName Nome del metodo da considerare.
	 * @return true se il metodo modifica lo stato false altrimenti
	 */
	abstract protected boolean needToRefreshStatusAfter(String methodName); 
	//Fine Funzioni Astratte
	/**
	 * Costruttore, Fallisce e termina il processo chiamante se non riesce a inizializzare l'interfaccia del servizio con i metodi base.
	 * @throws RemoteException 
	 */
	protected AReplicableService()throws RemoteException
	{
		super();
		id=-1;
		active=false;
		serviceStatus=buildInitialStatus();
		serviceName=getServiceName();
		serviceInterface=buildInterface();
	}
	synchronized public Status getStatus() throws RemoteException {
		Configuration.DebugOutput("AReplicableService.getStatus()");
		return serviceStatus;
	}
	synchronized public void setStatus(Status s, IReplicationManager updater)
			throws RemoteException {
		Configuration.DebugOutput("AReplicableService.setStatus("+s+","+updater+")");
		if(serviceStatus.getStatusCounter()<s.getStatusCounter()){
			serviceStatus=s;
			if(active){
				for(int i=0;i<serviceStatus.getManagersCount();i++)
				{
				IReplicationManager current=serviceStatus.getManager(i);
					if(!current.equals(updater))
					{
						try{
							current.updateStatus(serviceName,this,serviceStatus);
						}
						catch(IOException e)
						{
							serviceStatus.removeManager(i);
						}
					}
				}
			}
		}
	}
	synchronized public void activate(IReplicationManager activator) throws RemoteException {
		Configuration.DebugOutput("AReplicableService.setActive("+activator+")");
		if(active)return;
		active=true;
		boolean needupdate=false;
		for(int i=0;i<serviceStatus.getManagersCount();i++)
		{
			IReplicationManager current=serviceStatus.getManager(i);
			try
			{
				if(!current.equals(activator)){
					current.notifyActivation(serviceName,this,serviceStatus);
					Status s=current.getCurrentStatus(serviceName);
					if((s!=null)&&(serviceStatus.getStatusCounter()<s.getStatusCounter()))
					{
						serviceStatus=s;
						needupdate=true;
					}
					Configuration.DebugOutput("NotifyActivation on "+current+" Ok");
				}
			}catch(IOException e)
			{
				serviceStatus.removeManager(i);
			}
		}
		if(needupdate)updateBrokerStatus();
		if(serviceStatus.isLock()){
			boolean remove=false;
			try{
				remove=serviceStatus.getLockAccount().getRefManager().canRemoveLock(getServiceName(),this);
			}catch(IOException e)
			{
				serviceStatus.removeManager(serviceStatus.getLockAccount().getRefManager());
			}
			if(remove)
			{
				serviceStatus.setLock(false);
				serviceStatus.setLockAccount(null);
				serviceStatus.nextStatus();
				updateBrokerStatus();
			}
		}
	}

	synchronized public boolean lock(Account lockAccount) throws RemoteException {
		if(!active)throw new JSRProviderNotActive(this);
		Configuration.DebugOutput("AReplicableService.lock("+lockAccount+")");
		while(serviceStatus.isLock() && (!serviceStatus.getLockAccount().equals(lockAccount)))
		{
			try{
				wait();
			}catch(Exception e)
			{
				throw new RemoteException(e.getMessage());
			}
		}
		serviceStatus.setLock(true);
		serviceStatus.setLockAccount(lockAccount);
		serviceStatus.nextStatus();
		updateBrokerStatus();
		return true;
	}

	synchronized public boolean unLock(Account lockAccount) throws RemoteException {
		if(!active)throw new JSRProviderNotActive(this);
		Configuration.DebugOutput("AReplicableService.unLock("+lockAccount+")");
		if((!serviceStatus.isLock())||(!serviceStatus.getLockAccount().equals(lockAccount)))return false;
		serviceStatus.setLockAccount(null);
		serviceStatus.setLock(false);
		serviceStatus.nextStatus();
		updateBrokerStatus();
		notifyAll();
		return true;
	}
	synchronized public Object sendRequest(Request r, Account account)
			throws RemoteException {
		if(!active)throw new JSRProviderNotActive(this);
		Configuration.DebugOutput("AReplicableService.sendRequest("+r+","+account+")");
		while(serviceStatus.isLock() && (!serviceStatus.getLockAccount().equals(account)))
		{
			try{
				wait();
			}catch(Exception e)
			{
				throw new RemoteException(e.getMessage());
			}
		}
		try{
			Method m=r.getMethod();
			Object ret=getClass().getMethod(m.getName(),m.getArgsType()).invoke(this,r.getArgs());
			if(needToRefreshStatusAfter(m.getName())){
				serviceStatus.nextStatus();
				updateBrokerStatus();
			}
			return ret;
		}catch(Exception e)
		{
			throw new RemoteException(e.getMessage());
		}
	}
	public boolean alive() throws RemoteException {
		return true;
	}
	public Interface getInterface() throws RemoteException {
		return serviceInterface;
	}
	public int getId() throws RemoteException {
		return id;
	}
	synchronized public void registerReplicationManager(IReplicationManager manager)throws RemoteException
	{
	    if(!active)throw new JSRProviderNotActive(this);
	    if(manager==null)throw new JSRGenericException("Manager=null");
		Configuration.DebugOutput("AReplicableService.registerReplicationManager("+manager+")");	
		serviceStatus.addReplicationManager(manager);
		serviceStatus.nextStatus();
		updateBrokerStatus();
	}
	synchronized public void unRegisterReplicationManager(IReplicationManager manager)throws RemoteException 
	{
	    if(!active)throw new JSRProviderNotActive(this);
	    if(manager==null)throw new JSRGenericException("Manager=null");
		Configuration.DebugOutput("AReplicableService.unRegisterReplicationManager("+manager+")");	
		serviceStatus.removeManager(manager);
		serviceStatus.nextStatus();
		updateBrokerStatus();		
	}
	/**
	 * Metodo invocato quando si ha la necessit di notificare un cambio di stato a tutti i manager
	 * collegati al provider.
	 */
	protected void updateBrokerStatus()
	{
		for(int i=0;i<serviceStatus.getManagersCount();i++)
		{
			IReplicationManager current=serviceStatus.getManager(i);
			try{
				current.updateStatus(serviceName,this,serviceStatus);
			}catch(IOException e)
			{
				serviceStatus.removeManager(i);
			}
		}		
	}
	/**
	 * Metodo non esposto ai clienti per aggiungere un nuovo manager al provider, invocato solitamente all'atto
	 * della creazione del servizio se si conoscono gi managers disponibili.
	 * @param manager Manager da aggiungere.
	 */
	public void addReplicationManager(IReplicationManager manager)
	{
		serviceStatus.addReplicationManager(manager);
	}
	/**
	 * Metodo invocato in fase di chiusura del provider, deregistra il provider presso tutti i manager collegati
	 */
	synchronized public void unRegisterService()
	{
	    for(int i=0;i<serviceStatus.getManagersCount();i++)
	    {
			IReplicationManager current=serviceStatus.getManager(i);
			try
			{
			    current.unRegisterService(getServiceName(),this);
			}catch(Exception e){continue;}
	    }
	}
	/**
	 * Effettua la registrazione presso tutti i manager aggiunti in precedenza,metdodo invocato in fase di creazione
	 * del servizio. Successivamente alla registrazione e su indicazione dei provider il provider eventualmente si 
	 * attiva.
	 * @return true se la registrazione ha avuto successo presso almeno un server, false se il provider era gi
	 * registrato o se la registrazione  fallita presso tutti i manager.
	 */
	synchronized public boolean registerService()
	{
		if(id>=0)return false;
		Configuration.DebugOutput("AReplicableService.registerService()");
		boolean activate=false;
		for(int i=0;i<serviceStatus.getManagersCount();i++)
		{
			IReplicationManager current=serviceStatus.getManager(i);
			try{
				RegisterResponse r=current.registerService(serviceName,this);
				if(r==null)
				{
				    Configuration.DebugOutput(current.toString()+" Ha rifiutato la registrazione, Interfaccia scorretta per il servizio");
				    serviceStatus.removeManager(i);
				    i--;
				}
				else{
				    Configuration.DebugOutput(current.toString()+" Response "+r+" Current ID"+id);
				    if(id<0)id=r.id;
				    activate=activate|r.activate;
				    serviceStatus.updateManagers(r.status);
				}
			}catch(IOException e)
			{
				serviceStatus.removeManager(i);
				i--;
			}
		}
		if(activate)
		{
			try{
				activate(null);
			}catch(RemoteException e){}
		}
		return serviceStatus.getManagersCount()>0;
	}	
}
