/*
 * NodoAD.java
 *
 * Created on 28 febbraio 2004, 9.31
 */

package Src.Logica;

/**
 *
 * @author  Monaco Luca
 */
import java.util.*;
import Src.Servizi.Exception.*;
import javax.xml.parsers.*;
import org.xml.sax.SAXException;
import java.io.*;


public class NodoAD implements Nodo
{
    /**Il nome di un nodo  formata dall'associazione attribuo,valore che tutti gli esempi
     * associati al nodo condividono.*/
    private String nome;
    
    /**Riferimento al nodo padre.*/
    private NodoAD nodoPadre;
    
    /**Riferimenti ai nodi figli.*/
    private ArrayList nodiFigli;
    
    /**Insieme di esempi su cui opera il nodo.*/
    private ExampleSet esempi;
    
    /**Riferimento alla dichiarazione degli attributi e delle classe.*/
    private Dichiarazione dichiarazione;
    
    /**Nome dell'attributo sul quale il nodo effettua la selezione.*/
    private String nomeAttributo;
    
    /**Riferimento all'albero a cui il nodo appartiene.*/
    private AlberoDecisionale albero;
    
    /**Lista di attributi disponibili per la scelta al nodo.*/
    private ArrayList attributiDisponibili;
    
    /**Itero che esprime il livello del nodo nell'albero*/
    private int altezzaNodo;
    
    
    /** Creates a new instance of NodoAD */
    public NodoAD(NodoAD padre, ExampleSet eSet, Dichiarazione dich,ArrayList attrDisp,String nome)
    throws FileNotFoundException, ParserConfigurationException, IOException, SAXException, AttributeNotFoundException
    {
        this.nodoPadre = padre;
        this.nome = nome;
        this.altezzaNodo = padre.getAltezzaNodo()+1;
        this.esempi = eSet;
        this.dichiarazione = dich;
        this.nodiFigli = new ArrayList();
        this.albero = padre.getAlbero();
        this.attributiDisponibili = new ArrayList(attrDisp);
        this.generaFigli();
    }
    
    
    public NodoAD(AlberoDecisionale ad, ExampleSet eSet, Dichiarazione dich,int h)
    throws FileNotFoundException, ParserConfigurationException, IOException, SAXException, AttributeNotFoundException
    {
        this.nodoPadre = null;
        this.nome = "RADICE";
        this.altezzaNodo = h;
        this.esempi = eSet;
        this.dichiarazione = dich;
        this.nodiFigli = new ArrayList();
        this.attributiDisponibili = new ArrayList(this.dichiarazione.getNomeAttributi());
        this.albero = ad;
        this.generaFigli();
    }
    
    /**Recupera il nodo figlio in posizione index.*/
    public NodoAD getFiglio(int index)
    {
        return (NodoAD)this.nodiFigli.get(index);
    }
    
    
    /**Per il recupero del riferimento al nodo padre.*/
    public AlberoDecisionale getAlbero()
    {
        return this.albero;
    }
    
    /**Per il recupero del nome del nodo.*/
    public String getNome()
    {
        return this.nome;
    }
    
    public ExampleSet getEsempi()
    {
        return this.esempi;
    }
    
    public ArrayList getAttributiDisponibili()
    {
        return this.attributiDisponibili;
    }
    
    /**Permette di recuperare la classe pi frequente all'interno degli esempi in input al nodo.*/
    public String getClassePiFrequente()
    throws FileNotFoundException, ParserConfigurationException, IOException, SAXException
    {
        int max = 1000;  //Inizializzato a infinito
        int indice = -1;
        ArrayList esempiRaggruppati = this.albero.raggruppaEsempi(this.esempi);
        for(int i=0;i<esempiRaggruppati.size();i++){
            ExampleSetOmogeneo eso = (ExampleSetOmogeneo)esempiRaggruppati.get(i);
            if (i==0) {
                max = eso.getSize();
                indice=0;
            }
            else {
                if (eso.getSize()>max) {
                    max = eso.getSize();
                    indice = i;
                }
            }
        }
        return ((ExampleSetOmogeneo)esempiRaggruppati.get(indice)).getClasse();
    }
    
    
    /**Partiziona l'insieme degli esempi in ingresso al nodo fra i suoi nodi figli, in base al
     * valore assunto dall'attributo specificato.
     * Ritorna il numero di partizioni effettuate.*/
    public int partizionaEsempi(String nomeAttr)
    throws FileNotFoundException, ParserConfigurationException, IOException, SAXException, AttributeNotFoundException
    {
       
        ExampleSet[] gruppiPartizionati = this.albero.applicaPartizionamento(this.esempi, nomeAttr);
       
        //Determinazione degli attributi disponibila ai figli
        ArrayList attrDisp = new ArrayList(this.attributiDisponibili);
        attrDisp.remove(this.nomeAttributo);
        
        for(int x=0;x<gruppiPartizionati.length;x++){
            
            String nome;
            if (this.dichiarazione.getTipoAttributo(nomeAttr).equals("discreto")) 
                nome = nomeAttr+"="+(String)this.dichiarazione.getValoriPossibiliAttributoDiscreto(nomeAttr).get(x); 
            else   
            {
                if (x==0) nome=nomeAttr+"<="+this.dichiarazione.getSogliaAttributoContinuo(nomeAttr);
                else nome=nomeAttr+">"+this.dichiarazione.getSogliaAttributoContinuo(nomeAttr);
            } 
            
            
            
            ArrayList esempiRaggruppati = this.albero.raggruppaEsempi(gruppiPartizionati[x]);
            
            
            if(esempiRaggruppati.size()==1) {
                String classe = ((ExampleSetOmogeneo)(esempiRaggruppati.get(0))).getClasse();
                this.nodiFigli.add(new FogliaAD(this,classe,nome));
            }
            else {
                NodoAD nodo = new NodoAD(this,gruppiPartizionati[x], this.dichiarazione,attrDisp,nome);
                this.nodiFigli.add(nodo);
            }
        }
        
        return gruppiPartizionati.length;
    }
    
    
    /**Il metodo permette di scegliere l'attributo su cui il nodo effettua la selezione degli esempi.*/
    public String scegliAttributo()
    throws FileNotFoundException, ParserConfigurationException, IOException, SAXException,AttributeNotFoundException
    {
        String nomeAttr = this.albero.selezionaAttributo(this);
        this.nomeAttributo = nomeAttr;
        return nomeAttr;
    }
    
    /**Il metodo implementa l'algoritmo da usare per la creazione dei figli del nodo.
     * Ritorna il numero di figli generati:
     * # 1 --> Generato un nodo foglia.*/
    public int generaFigli()
    throws FileNotFoundException, ParserConfigurationException, IOException, SAXException, AttributeNotFoundException
    {
        /*In primis si controlla che il nodo abbia esempi in ingresso:*/
        if (this.esempi.getSize()==0){
            
            //Recupero dalla folgia padre la classe pi frequente.
            String classe = this.nodoPadre.getClassePiFrequente();
            
            //Blocco la catena di generazione con la creazione di una foglia etichettata con il nome
            //della classe pi frequente del nodo padre
            this.nodiFigli.add(new FogliaAD(this,classe,"Nodo senza esempi"));
            return 1;
        }
        
        /*Si controlla che gli il nodo abbia a disposizione degli attributi per poter effettuare una 
         partizione: */
        if(this.attributiDisponibili.size()==0){ 
                //Creo un nodo foglia etichettato con il nome della classe pi frequente nel nodo
                this.nodiFigli.add(new FogliaAD(this,this.getClassePiFrequente(),"Attrbuti finiti"));
                return 1;
        }     
        
        /*Raggruppo gli esempi in basse alla loro classe:*/
        ArrayList esempiRaggruppati = this.albero.raggruppaEsempi(this.esempi);
       
       /*Se gli esempi sono tutti della stessa classe creo un nodo foglia:*/ 
       if(esempiRaggruppati.size()==1) {
            //Recupero il nome della classe che caratterizza il gruppo.          
            String classe = ((ExampleSetOmogeneo)(esempiRaggruppati.get(0))).getClasse();
            
            //Blocco la catena di generazione con la creazione di una foglia
            this.nodiFigli.add(new FogliaAD(this,classe,"foglia"));
            return 1;
        }
        /*altrimenti gli esempi sono di classi diverse, occorrono nuove partizionamenti:*/
        else {
             //Scelgo l'attributo su cui partizionare
             String nomeAttr = this.scegliAttributo();
             //Partiziono gli esempi in ingresso secondo questo attributo
             return this.partizionaEsempi(nomeAttr);
        }
        
    }
    
    
    /**Il metodo permette di valutare un Istanza.*/
    public String valuta(Istanza ist)
    throws AttributeNotFoundException
    {
        //System.out.println("Valutazione dell'istanza "+ist);
        //System.out.println("Nodo di valutazione:"+this.nome);
        //System.out.println("Prossimo attributo valutato:"+this.nomeAttributo);
        String tipoAttr = this.dichiarazione.getTipoAttributo(this.nomeAttributo);
        if (tipoAttr==null) return ((Nodo)this.nodiFigli.get(0)).valuta(ist);
        else if (tipoAttr.equals("discreto"))
        {
            String val = ((AttributoDiscreto)ist.getAttributo(this.nomeAttributo)).getValue();
            //System.out.println("Valore dell'attributo discreto "+this.nomeAttributo+": "+val);
            ArrayList valoriPossibili = this.dichiarazione.getValoriPossibiliAttributoDiscreto(this.nomeAttributo);
            //System.out.println("Valori possibili dell'attributo "+this.nomeAttributo+": "+valoriPossibili);
            for (int i=0;i<valoriPossibili.size();i++){
                if (val.equals((String)valoriPossibili.get(i))){
                    //System.out.println("-->Match con il ramo "+i);
                    return ((Nodo)this.nodiFigli.get(i)).valuta(ist);
                }
            }
            //System.out.println("L'istanza assume per l'attributo "+ this.nomeAttributo+" un valore non contemplato");
        }
        else{
            int val = ((AttributoContinuo)ist.getAttributo(this.nomeAttributo)).getValue();
            //System.out.println("Valore dell'attributo continuo "+this.nomeAttributo+": "+val);
            int soglia = this.dichiarazione.getSogliaAttributoContinuo(this.nomeAttributo);
            //System.out.println("Valore soglia dell'attributo "+this.nomeAttributo+": "+soglia);
            if (val<=soglia) return ((Nodo)this.nodiFigli.get(0)).valuta(ist);
            else return ((Nodo)this.nodiFigli.get(1)).valuta(ist);
        }    
        return "Errore";
    }
    
    public int getAltezzaNodo() 
    {
        return this.altezzaNodo;
    }
    
    public String getNomeAttributo()
    {
        return this.nomeAttributo;
    }
    
    /**Permette di calcolare il cammino tra la radice dell'albero e il nodo.*/
   public ArrayList getPath()
   {
       if (this.altezzaNodo==0){
           ArrayList list = new ArrayList();
           list.add(this.nome);
           return list;
       }
        else{
            ArrayList list = new ArrayList(this.nodoPadre.getPath());
            list.add(this.nome);
            return list;
        }
   } 
   
   
      
    public String toString()
    {
       return this.nome;
    }
}
