package SOMA.resourceManagement;

import SOMA.Environment;
import SOMA.explorer.*;
import SOMA.naming.*;
import SOMA.network.connection.*;
import SOMA.gui.Debug;

import java.util.*;

/** 
 * Classe di interfaccia fra il place e l'infrastruttura di gestione delle risorse; 
 * definisce i metodi per ricavare informazioni sull'occupazione delle risorse da parte
 * degli agenti e sugli allarmi che si sono verificati nel place, e i metodi per conoscere 
 * e settare i valori delle soglie, 
 * Fa parte dell'Environment. 
 * Contiene il Monitoring Manager e i vari Alarm Handler.
 * 
 * @see SOMA.resourceManagement.MonitoringManager
 * @see SOMA.resourceManagement.CpuAlarmsHandler
 * @see SOMA.resourceManagement.FileAlarmsHandler
 * @see SOMA.resourceManagement.MemAlarmsHandler
 * @see SOMA.resourceManagement.NetAlarmsHandler
 * 
 *
 * @author Silvia Vecchi
 */

public class PlaceResourceManager{
	
	Environment env;
	public MonitoringManager monitoringManager; 
	CpuAlarmsHandler cpuAH;
	MemAlarmsHandler memAH;
	NetAlarmsHandler netAH;
	FileAlarmsHandler fileAH;
	

	/** Consultata dagli Alarm Handler per evitare di uccidere due volte lo stesso agente */
	Hashtable killedAgents = new Hashtable(); 
	// qua perch deve essere visibile a tutti i vari AlarmsHandler	
	DirExplorerItem placeResourceManagerDir;
	
	/** Costruttore: crea l'oggetto e tutti gli elementi che lo compongono
	  * inserendoli nella gerarchia a direttori del sistema
	  */
	public PlaceResourceManager (Environment env){
		
		this.env = env;
		
		placeResourceManagerDir = new DirExplorerItem("placeResourceManager");
		if (placeResourceManagerDir != null)
		  Debug.outln("PlaceResourceManager: creato placeResourceManagerDir");
		else   Debug.outln("PlaceResourceManager: problemi nella creazione di placeResourceManagerDir");
		env.dir.addItem(placeResourceManagerDir);
		
		monitoringManager = new MonitoringManager(this);
		placeResourceManagerDir.addItem("monitoringManager", new DaemonExplorerItem(monitoringManager));
		if (monitoringManager != null) Debug.outln("PlaceResourceManager: creato monitoringManager");
		else  Debug.outln("PlaceResourceManager: problemi nella creazione di monitoringManager");
		
		//try { monitoringManager.start();}
		//catch(Exception ex)	{ Debug.outln(ex.toString());}
		
		cpuAH = new CpuAlarmsHandler(this);
		memAH = new MemAlarmsHandler(this);
		netAH = new NetAlarmsHandler(this);
		fileAH = new FileAlarmsHandler(this);
		
		if (cpuAH != null && memAH != null && netAH != null && fileAH != null)
		  Debug.outln("PlaceResourceManager: creati gli Alarms Handlers");
		else Debug.outln("PlaceResourceManager: problemi nella creazione degli Alarms Handlers");  
	}
	
	
	
	
	/* Funzioni che restituiscono all'agente, che le invoca tramite l'AgentSystem,
       le info su soglie e consumi */
	
	/** Restituisce i valori delle soglie */
	public Threshold getConsumptionThresholds(){
		return monitoringManager.threshold;
	}
	
	/** Restituisce il valore dell'intervallo di polling */
	public int getPollingTime(){
		return monitoringManager.pollingTime;
	}
	
	/** Restituisce i valori di consumo raggiunti da tutti i thread
    *  di un certo agente, dalla nascita
    */
	public AgentInfo[] getAgentTotConsumptions(AgentID agID){
		
		int index=0;
		LinkedList list = new LinkedList();
		AgentInfo temp = new AgentInfo();
		AgentInfo[] info;
		//AgentWorker agW = env.agentManager.agentWorkerStore.getWorker(agID);
		//ThreadGroup agTG = agW.agentThreadGroup;
		
		Iterator i = monitoringManager.agentsTotConsumptions.iterator();
		
		for (; i.hasNext();)
		 if ((temp=(AgentInfo)(i.next())).agID == agID){
		   // conto i thread dell'agente e metto i dati relativi in una lista
			index++; 
			list.add(temp);
		 }
		
		info = new AgentInfo[index];
		list.toArray(info);
		
		if (info.length > 0)
		   return info;
		else	{
			Debug.outln("Nessun thread appartenente all'agente specificato: " + agID);
			return null;
		}
	}
		  
		
	
	
	/** Restituisce i valori di consumo di tutti i thread di un certo agente, 
	  * nell'ultimo intervallo di polling
    */
	public AgentInfo[] getAgentDiffConsumptions(AgentID agID){
		
		int i, index=0;
		LinkedList list = new LinkedList();
		AgentInfo[] info;
		//AgentWorker agW = env.agentManager.agentWorkerStore.getWorker(agID);
		//ThreadGroup agTG = agW.agentThreadGroup;
		
		for (i=0; i<monitoringManager.agentsDiffConsumptions.length; i++)
		if ( monitoringManager.agentsDiffConsumptions[i].agID == agID ){
			// conto i thread dell'agente e metto i dati relativi in una lista
			index++; 
			list.add(monitoringManager.agentsDiffConsumptions[i]);
		}
		
		info = new AgentInfo[index];
		list.toArray(info);
		
		if (info.length > 0)
		   return info;
		else{
			Debug.outln("Nessun thread appartenente all'agente specificato: " + agID);
			return null;
		}
		
	}
	
	
    /** Restituisce i valori di consumo raggiunti da tutti i thread
      *  di tutti gli agenti, dalla nascita
      */
	public AgentInfo[] getAgentTotConsumptions(){
		
		AgentInfo[] info;
		int size = monitoringManager.agentsTotConsumptions.size();
		info = new AgentInfo[size];
		monitoringManager.agentsTotConsumptions.toArray(info);
		

		if (info.length > 0)
		   return info;
		else{
			Debug.outln("Nessun agente");
			return null;
		}
	}
		  

	/** Restituisce i valori di consumo di tutti i thread di tutti gli agenti, 
	  * nell'ultimo intervallo di polling
    */
	public AgentInfo[] getAgentDiffConsumptions(){
		
		AgentInfo[] info;
		int size = monitoringManager.agentsDiffConsumptions.length;
		info = new AgentInfo[size];
		for (int i=0; i<size; i++)	{
			info[i]=new AgentInfo(monitoringManager.agentsDiffConsumptions[i]);
		}
		if (info.length > 0)
		   return info;
		else{
			Debug.outln("Nessun agente");
			return null;
		}
	}

	
	/** Restituisce il numero di allarmi che si sono verificati relativamente alla cpu */
	public int getCpuAlarmNum(){
		return cpuAH.alarmsNum;
	} 
	
	/** Restituisce il numero di allarmi che si sono verificati relativamente ai file */
  public int getFileAlarmNum(){
	    return fileAH.alarmsNum;
	} 
	
	/** Restituisce il numero di allarmi che si sono verificati relativamente alla memoria */
	public int getMemAlarmNum(){
	    return memAH.alarmsNum;
	} 
	
	/** Restituisce il numero di allarmi che si sono verificati relativamente alla rete */
	public int getNetAlarmNum(){
	    return netAH.alarmsNum;
	} 
	
	
	/** Restituisce le informazioni sugli allarmi relativi alla cpu */
	public Hashtable getCpuAlarmInfo(){
		return cpuAH.alarmsTab;
	} 
	
	/** Restituisce le informazioni sugli allarmi relativi ai file */
	public Hashtable getFileAlarmInfo(){
	    return fileAH.alarmsTab;
	} 
	
	/** Restituisce le informazioni sugli allarmi relativi alla memoria */
	public Hashtable getMemAlarmInfo(){
	    return memAH.alarmsTab;
	} 
	
	/** Restituisce le informazioni sugli allarmi relativi alla rete */
	public Hashtable getNetAlarmInfo(){
	    return netAH.alarmsTab;
	} 
	
  
  
  
  // Funzioni di settaggio delle soglie 
	 
  /** Setta tutte soglie ai valori passati come argomenti */
  public void setAllThresholds(float cpu, float mem, 
  		                        float file_in, float file_out,
								              float band_width)
  {
  	monitoringManager.threshold.setAll(cpu, mem, file_in, file_out, band_width);
  }
  
  /** Setta tutte le soglie ai valori di default */
  public void setDefaultAllThresholds(){
	monitoringManager.threshold.setDefaultAll();
  }
  
  /** Setta il valore di soglia della cpu */
  public void setCpuThreshold(float val){
  	monitoringManager.threshold.setCpu(val);
  }
  
  /** Setta il valore di soglia della cpu a quello di default */
  public void setDefaultCpuThreshold(){
  	monitoringManager.threshold.setDefaultCpu();
  }
  
  /** Setta il valore di soglia della memoria */
  public void setMemThreshold(float val){
  	monitoringManager.threshold.setMem(val);
  }
  
  /** Setta il valore di soglia della memoria a quello di default */
  public void setDefaultMemThreshold(){
  	monitoringManager.threshold.setDefaultMem();
  }
  
  /** Setta il valore di soglia del numero di operazioni di lettura da file */
  public void setFile_inThreshold(float val){
  	monitoringManager.threshold.setFile_in(val);
  }
  
  /** Setta il valore di soglia del numero di operazioni di lettura da file 
    * a quello di default
	  */
  public void setDefaultFile_inThreshold(){
  	monitoringManager.threshold.setDefaultFile_in();
  }
  
  /** Setta il valore di soglia del numero di operazioni di scrittura su file */
  public void setFile_outThreshold(float val){
  	monitoringManager.threshold.setFile_out(val);
  }
  
  /** Setta il valore di soglia del numero di operazioni di scrittura su file
    * a quello di default 
    */
  public void setDefaultFile_outThreshold(){
  	monitoringManager.threshold.setDefaultFile_out();
  }
  
  /** Setta il valore di soglia dell'occupazione di banda */
  public void setBand_widthThreshold(float val){
  	monitoringManager.threshold.setBand_width(val);
  }
  
  /** Setta il valore di soglia dell'occupazione di banda 
    * a quello di default */
  public void setDefaultBand_widthThreshold(){
  	monitoringManager.threshold.setDefaultBand_width();
  }
  
  
  // Funzioni di settaggio dell'intervallo di polling  
  
  /** Setta la durata dell'intervallo di polling al valore passato come argomento */
  public synchronized void setPollingTime (int val){
  	monitoringManager.pollingTime = val;
  }
  
  /** Setta la durata dell'intervallo di polling al valore di default */
  public synchronized void setDefaultPollingTime(){
  	monitoringManager.pollingTime = monitoringManager.POLLING_TIME;
  }
	
	
	
	
	/** Attiva/Disattiva il ciclo di controllo */
	public void setCtrlStatus(boolean s){
		monitoringManager.setCtrl(s);
	}
	
	/** Restituisce lo stato del ciclo di controllo */
	public boolean getCtrlStatus(){
		return monitoringManager.ctrl;
	}
	
	
	
	/** Restituisce la rappresentazione in stringa */
    public String toString(){
		  return "ResourceManager";  
    }
}