package SOMA.resourceManagement;

import SOMA.resourceManagement.windows.*;
import SOMA.network.connection.*;
import SOMA.naming.*;
import SOMA.gui.Debug;
import SOMA.resourceManagement.res.*;

import java.util.*;
import java.io.*;
import java.text.*;

/**
 * Classe che coordina il monitoraggio e l'accounting.
 * E' un componente del Place Resource Manager.
 * Un demone, quando avviato (cio messo in stato ON), esegue un ciclo
 * in cui si alternano fasi di lettura dal monitor, di aggiornamento dei dati e delle
 * finestre (se sono aperte), e di controllo dei consumi effettuati dagli agenti
 * (se ci sono agenti da controllare): se si rileva un superamento di soglia viene
 * generato un evento di allarme consegnato all'opportuno gestore.
 * Al momento della creazione  in stato OFF.
 * E' prevista un'autoregolazione della frequenza dei controlli:
 * se il numero di allarmi  sufficientemente basso viene allentata,
 * se invece  elevato viene aumentata.
 * I controlli possono essere disattivati, mantenendo per attive le letture e
 * gli aggiornamenti, settando un opportuno flag.
 *
 * ATTENZIONE: il monitoraggio  effettuabile solamente su un place per host, in caso
 * di configuarazione avanzata si creano dei conflitti al momento ancora non risolti.
 *
 * @see SOMA.resourceManagement.PlaceResourceManager
 * @see SOMA.resourceManagement.Threshold
 * @see res.JvmMonitor
 * @see res.ProcessMonitor
 *
 * @author Silvia Vecchi
 */



public class MonitoringManager implements Runnable, Daemon{

	/** Demone che esegue il ciclo di gestione del monitoraggio */
	Thread myMonitoringDaemon = null;
	/** Vettore dei listener (gestori di allarme e finestre) */
	Vector listeners = null;  // potrei usare anche ArrayList, non so quale sia pi efficiente...
	/** Flag per segnalare la presenza della finestra degli agenti,
	  * e, di conseguenza, la necessit di aggiornarla
	  */
	boolean agentWin = false;
	/** Flag per segnalare la presenza della finestra dei processi,
	  * e, di conseguenza, la necessit di aggiornarla
	  */
	boolean processWin = false;
	/** Flag per segnalare la presenza della finestra della rete,
	  * e, di conseguenza, la necessit di aggiornarla
	  */
	boolean networkWin = false;

	/** Flag per stabilire se effettuare o meno i controlli */
	boolean ctrl = false;

	/** Stato del demone */
	Object status = OFF;
	String ErrorDescription = " ";

	/** Struttura contenente i consumi di tutti gli agenti nell'ultimo intervallo di polling */
	FilteredInfo[] agentsDiffConsumptions = new FilteredInfo[0];
	/** Struttura contenente i consumi di tutti gli agenti dalla nascita */
  HashSet agentsTotConsumptions = new HashSet(); //potrei usare anche una linkedList, non so quale sia pi efficiente...
	/** Intervallo di polling */
	int pollingTime;
	/** Valore di default dell'intervallo di polling */
	static final int POLLING_TIME = 5;
	/** Intervallo fra le due letture differenziali */
	int delta;
	/** Valore di default dell'intervallo fra le due letture differenziali */
  static final int DELTA = 3;
	/** Soglie dei consumi usate per effettuare i controlli */
	Threshold threshold;
	/** Numero di allarmi relativo agli ultimi tre cicli di controllo */
	int alarmsNum;
	/** Numero massimo di allarmi al di sopra del quale si intensifica la frequenza dei controlli */
	static final int ALARMS_NUM_MAX = 7;
  /** Numero minimo di allarmi al di sotto del quale si allenta la frequenza dei controlli */
	static final int ALARMS_NUM_MIN = 3;

	/** Riferimento al Place Resource Manager */
	PlaceResourceManager prm;
	/** Oggetto di interfaccia al Monitor della JVM */
	JvmMonitor jvmMonitor = null;
	/** Oggetto di interfaccia al Monitor della macchina (processi e rete) */
	ProcessMonitor processMonitor = null;

	FileWriter out;
  DateFormat timeStampFormat;


	/** Costruttore: crea gli oggetti d'interfaccia al monitor e
	 *  setta soglie e intervallo di polling
	 */
	public MonitoringManager(PlaceResourceManager prm){
	  listeners = new Vector();
    try{
    	out = new FileWriter("Consumptions.txt");
      out.write("********** CONSUMI DEGLI AGENTI NELL'ULTIMA SESSIONE **********\n\n");
      timeStampFormat = DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.LONG);
    }
    catch(Exception e){Debug.outln(e.toString());}
		this.prm = prm;
		jvmMonitor = new JvmMonitor();
		processMonitor = new ProcessMonitor();    // per il momento lo uso solo per le finestre
		if (jvmMonitor != null && processMonitor != null)
		  Debug.outln("MonitoringManager: costruttore -> istanziati gli oggetti di interfaccia al monitor");
		else Debug.outln("MonitoringManager: costruttore -> problemi nella creazione di jvmMonitor e processMonitor");
		pollingTime=POLLING_TIME;
		Debug.outln("MonitoringManager: costruttore -> settato l'intervallo di polling al valore di default: " + pollingTime);
		delta=DELTA;
		Debug.outln("MonitoringManager: costruttore -> settato l'intervallo di lettura differenziale al valore di default: " + delta);
		threshold = new Threshold(); // il costruttore setta i valori di default
		Debug.outln("MonitoringManager: costruttore -> settate le soglie ai valori di default:");
		Debug.outln(threshold.printString());
	}


	 /** Restituisce lo stato */
    public Object getStatus()
    {return status;}

	/** Restituisce la rappresentazione in stringa */
    public String toString(){
	  	return "[MonitoringManager: status = " + status +
           (status == ERROR ?  " " + ErrorDescription : "") + "]";
    }

	/** Mette lo stato a ON e avvia il demone che fa polling  */
  public synchronized void start() throws MonitoringManagerException, Exception{
    if( !status.toString().equals("ON") ){
      try{
        myMonitoringDaemon = new Thread( this, toString() );
		    status = ON;
        myMonitoringDaemon.start();
	   	  Debug.outln("MonitoringManager -> start!");
      }
      catch( Exception ex ){
        status = ERROR;
        ErrorDescription = ex.toString();
        throw( ex );
      }
    }
    else throw new MonitoringManagerException( "MonitoringManager already ON" );
  }


  /** Mette lo stato a OFF, e quindi sospende il polling */
  public synchronized void stop() throws MonitoringManagerException, Exception{
    if( !status.toString().equals("OFF")  ){
      try{
        status = OFF;
        Debug.outln("MonitoringManager -> stop!");
        // creo eventuale logfile ?
      }
      catch( Exception ex ){
        status = ERROR;
        ErrorDescription = ex.toString();
        throw( ex );
      }
    }
    else throw new MonitoringManagerException( "MonitoringManager already OFF" );
  }

 /* NOTA sui metodi "start" e "stop": quando viene invocato start (stop) e lo stato era
    OFF (ON), viene stampata l'eccezione "MonitoringManager already ON (OFF)" seguita da
    "MonitoringManager -> start! (stop!)". Il comportamento risulta corretto, il monitoraggio
    viene effettivamente avviato o sospeso, ma il fatto che venga lanciata l'eccezione, peraltro
    quella di stop quando esegue start e viceversa, non ha senso...
    Quando invece si invoca start (stop) e lo stato era ON (OFF), stampa correttamente
    "MonitoringManager already ON (OFF)".   */



  /** Registra il listener passato come argomento */
  public void addMonitoringManagerListener(MonitoringManagerListener listener){
  	if (listener != null)
		if (!listeners.contains(listener))
			if ( listeners.add(listener) )
	           Debug.outln("MonitoringManager: listener " + listener + " registrato");
	        else
               Debug.outln("MonitoringManager: listener " + listener + " non registrato");
		else
		   Debug.outln("MonitoringManager: listener " + listener + " gi registrato");
    else
	   Debug.outln("MonitoringManager: errore listener=null");

	  if (listener instanceof AgentWinData)
	  	agentWin=true;
	  if (listener instanceof ProcessWinData)
	  	processWin=true;
	  if (listener instanceof NetworkWinData)
	  	networkWin=true;
  }



  /** Rimuove il listener passato come argomento */
  public void removeMonitoringManagerListener(MonitoringManagerListener listener){
    if (listener != null)
	   if (listeners.contains(listener))
	   	   if (listeners.removeElement(listener))
		      Debug.outln("MonitoringManager: listener " + listener + " rimosso");
		     else
		      Debug.outln("MonitoringManager: listener " + listener + " non rimosso");
	   else
	      Debug.outln("MonitoringManager: listener " + listener + " gi rimosso");
	  else
	   Debug.outln("MonitoringManager: errore listener=null");

	  if (listener instanceof AgentWinData)
	  	agentWin=false;
	  if (listener instanceof ProcessWinData)
	  	processWin=false;
	  if (listener instanceof NetworkWinData)
	  	networkWin=false;

  }




  /** Ciclo principale di funzionamento */
  public void run(){

  	Debug.outln("MonitoringManager ->  run");
	  int count=0, alarmsNumNew=0, alarmsNumOld=0;

	  while (status == ON){

	  /* Prelievi e aggiornamenti sono fatti comunque, ma la loro frequenza subisce
       la stessa autoregolazione dei controlli, sarebbe meglio separare le due cose,
       mantenendo costante la frequenza di prelievi e aggiornamenti... */
		getProcessInfo();
	  getNetworkInfo();
	  getAgentsInfo();

		/* I controlli sono fatti solo quando sono presenti agenti e ctrl=true;
		   ponendo quindi ctrl=false  possibile disattivare i controlli */
		if ((prm.env.agentManager.agentsNumber() > 0) && ctrl){
		   controlAgentsInfo(agentsDiffConsumptions);
		  /* Autoregolazione della frequenza dei controlli */
	  	if (count == 3){
		    Debug.outln("MonitoringManager: run -> 4 ciclo, verifico se la frequenza dei controlli deve essere modificata");
		    alarmsNumNew = (prm.cpuAH.alarmsNum+prm.netAH.alarmsNum+prm.memAH.alarmsNum+prm.fileAH.alarmsNum);
		    alarmsNum=(alarmsNumNew-alarmsNumOld);
		    Debug.outln("MonitoringManager: run -> numero di allarmi negli ultimi  3 cicli: "+alarmsNumNew+"-"+alarmsNumOld+"="+alarmsNum);
		    if (alarmsNum>ALARMS_NUM_MAX && pollingTime>1){
		      pollingTime-=1;
			    Debug.outln("MonitoringManager: run -> Numero di allarmi troppo elevato, intensifico i controlli: pollingTime="+pollingTime);
		    }
		    else if (alarmsNum<ALARMS_NUM_MIN && pollingTime<10){
		  	        pollingTime+=1;
				        Debug.outln("MonitoringManager: run -> Numero di allarmi sufficientemente basso, allento i controlli: pollingTime="+pollingTime);
		           }
				   else Debug.outln("MonitoringManager: run -> Frequenza di controllo inalterata: pollingTime="+pollingTime);
	      alarmsNumOld=alarmsNumNew;
		    count=0;
		  }
		  count+=1;
		}// fine autoregolazione
		else if (ctrl) Debug.outln("MonitoringManager: run -> Nessun agente: non controllo i consumi dei thread di sistema");


		try {Thread.sleep(pollingTime*1000);}
		catch(Exception ex){Debug.outln(ex.toString());}
	 }
  }




  // Funzioni di prelievo dati dal monitor e controllo dei dati prelevati


  /** Preleva le informazioni sui processi interrogando il monitor relativo al SO */
  public void getProcessInfo (){

  	ProcessInfo[] procInfo;

	  Debug.outln("MonitoringManager -> getProcessInfo ");

	  procInfo = processMonitor.getProcessInfo(delta);


	  if (processWin){
			Enumeration en = listeners.elements();
	    while (en.hasMoreElements()){
		    	MonitoringManagerListener listener = (MonitoringManagerListener)en.nextElement();
		      if (listener instanceof ProcessWinData)
		      	listener.update(procInfo);
		     }
  	}

	}


  /** Preleva le informazioni sulla rete interrogando il monitor relativo al SO */
  public void getNetworkInfo (){

  	NetworkInfo netInfo;

	  Debug.outln("MonitoringManager -> getNetworkInfo ");

	  netInfo = processMonitor.getNetworkInfo(delta);


	  if (networkWin){
			Enumeration en = listeners.elements();
	    while (en.hasMoreElements()){
		    	MonitoringManagerListener listener = (MonitoringManagerListener)en.nextElement();
		      if (listener instanceof NetworkWinData)
		      	listener.update(netInfo);
		  }
  	}

  }





  /** Preleva le informazioni sui thread Java interrogando il monitor relativo alla JVM.
    * Aggiorna le tabelle agentsDiffConsumptions (consumi agenti nell'ultimo intervallo di
    * polling) e agentsTotConsumptions (consumi totali degli agenti dalla loro nascita);
	  * se non ci sono agenti la prima viene svuotata
	  */
  public void getAgentsInfo () {

  	LinkedList list = new LinkedList();
	  ThreadStat[] data1, data2;
	  AgentID agID;
	  int i, j, index=0;


	Debug.outln("MonitoringManager -> getAgentsInfo -> inizio");

	data1 = jvmMonitor.getThreadStat();
	Debug.outln("MonitoringManager: getAgentsInfo -> 1prelievo");

	try { Thread.sleep(delta*1000);}
	catch (Exception ex) {Debug.outln(ex.toString());}

    data2 = jvmMonitor.getThreadStat();
	  Debug.outln("MonitoringManager: getAgentsInfo -> 2prelievo");

	/* Conteggio e controllo solo i thread che non appartengono n all'environment n al main
	   mi sembra che questi due casi esauriscano i thread di sistema che, qualunque cosa stiano facendo,
	   non voglio assolutamente toccare; tutti gli altri dovrebbero appartenere agli agenti.
	   In realt farebbe comodo escludere dai controlli anche determinati tipi di agenti, per esempio
	   quelli di monitoraggio...campo SPECIAL ? */
	for (i=0; i<data1.length; i++)
		for (j=0; j<data2.length; j++)
			if ( !prm.env.threadGroup.getName().equals(data2[j].thread.getThreadGroup().getName()) &&
				 !data2[j].thread.getThreadGroup().getName().equals("main")){
				try{
					agID = new AgentID(data2[j].thread.getThreadGroup().getName());
					upDateAgentsTotConsumptions(data2[j], agID);

					if (data1[i].thread == data2[j].thread){
					   index++;
					   FilteredInfo info=new FilteredInfo();
					   info.agID=agID;
					   info.thread=data2[j].thread;
					   info.cpu = data2[j].cpu;
				     /* cpu  gi relativo ad un intervallo, non faccio differenze */
			       //info.time = (data2[j].time-data1[i].time);
				     info.mem = (data2[j].obj_size-data1[i].obj_size)/delta;

				     info.file_in = (data2[j].file_in-data1[i].file_in)/delta;
				     info.file_out = (data2[j].file_out-data1[i].file_out)/delta;
				     /* in questa versione si contano solo le op.file al sec, non i bytes medi al sec perch
				        il campo di variabilit della dimensione di ogni op.file  troppo ampio,
				        quindi usare un valore medio avrebbe poco senso, servirebbero valutazioni statistiche */
				     info.tcp_in = ((float)(data2[j].tcp_in-data1[i].tcp_in)*512)/delta;
				     info.tcp_out = ((float)(data2[j].tcp_out-data1[i].tcp_out)*512)/delta;
				     info.udp_in = ((float)(data2[j].udp_in-data1[i].udp_in)*773)/delta;
				     info.udp_out = ((float)(data2[j].udp_out-data1[i].udp_out)*773)/delta;
			        // tutti questi passaggi per avere bytes medi al secondo

					   list.add(info);
					}//if
				}//try
				catch(Exception ex){Debug.outln(ex.toString());}
			}//if
		Debug.outln("MonitoringManager: getAgentsInfo -> rilevati " + data2.length + " thread attivi di cui " + index + " non di sistema");

		agentsDiffConsumptions = new FilteredInfo[index];
		list.toArray(agentsDiffConsumptions);

	  if (agentWin){
			Enumeration en = listeners.elements();
	    while (en.hasMoreElements()){
	    	  MonitoringManagerListener listener = (MonitoringManagerListener)en.nextElement();
		      if (listener instanceof AgentWinData)
		      	 listener.update(agentsDiffConsumptions);
		  }
  	}

		for (index=0; index < agentsDiffConsumptions.length; index++)
		  Debug.outln(agentsDiffConsumptions[index].printString());

	}



 /** Aggiorna la tabella in cui vengono memorizzati i consumi di tutti i thread
   * degli agenti, ancora presenti o no, dalla loro nascita
   */
  public void upDateAgentsTotConsumptions(ThreadStat data, AgentID agID){

  	FilteredInfo info = new FilteredInfo();
	  info.agID = agID;
	  info.thread = data.thread;
	  info.cpu = 0;
    info.time = data.time;
	  info.mem = data.obj_size;
	  info.file_in = data.file_in;
	  info.file_out = data.file_out;
	  info.tcp_in = data.tcp_in*512;
	  info.tcp_out = data.tcp_out*512;
	  info.udp_in = data.udp_in*773;
	  info.udp_out = data.udp_out*773;


	/* Mi sincronizzo su agentsTotConsumptions per rendere atomico inserimento di
	   un nuovo dato e rimozione di una sua eventuale versione pi vecchia, per evitare
	   che in fase di lettura vengano letti sia il nuovo che il vecchio */
	synchronized(agentsTotConsumptions){
	   agentsTotConsumptions.add(info);
	   Iterator i = agentsTotConsumptions.iterator();
	   /* Aggiungo il nuovo elemento alla lista poi controllo se era gi presente
	      una versione pi vecchia dello stesso, se si la elimino */
	   for (; i.hasNext();)
	   	if (((FilteredInfo)(i.next())).thread == info.thread)
			i.remove();
	}


	/* Aggiorno anche il file di testo con i dati relativi ai consumi nella sessione*/

	//out.println(timeStampFormat.format(timeStamp));

	try{
	      out.write(timeStampFormat.format(new Date()));
        out.write("\n");
		    out.write(info.printString());
		    out.write("\n");

	}
	catch(Exception e){Debug.outln(e.toString());}
}








  /** Confronta i consumi rilevati con i valori delle soglie corrispondenti,
    * lanciando un evento in caso di superamento
	  */
  public void controlAgentsInfo (FilteredInfo[] data) {
    Debug.outln("MonitoringManager -> controlAgentsInfo");
  	for (int i=0; i<data.length; i++){
		 if ( data[i].cpu > threshold.cpu ){
		 	try { sendEvent( new CpuEvent(this, data[i].agID, data[i].thread, data[i].cpu));}
			catch(MonitoringManagerException ex) { Debug.outln(ex.toString());}
		 }
		 float bwCons = ( data[i].tcp_in + data[i].tcp_out + data[i].udp_in + data[i].udp_out);
		 if (bwCons > threshold.band_width){
		 	try { sendEvent( new NetEvent(this, data[i].agID, data[i].thread, bwCons));}
			catch(MonitoringManagerException ex) { Debug.outln(ex.toString());}
		 }
		 if ( data[i].mem > threshold.mem ){
		 	try { sendEvent( new MemEvent(this, data[i].agID, data[i].thread, data[i].mem));}
			catch(MonitoringManagerException ex) { Debug.outln(ex.toString());}
		 }
		 if ( data[i].file_in > threshold.file_in || data[i].file_out > threshold.file_out ){
		 	try { sendEvent( new FileEvent(this, data[i].agID, data[i].thread, (data[i].file_in+data[i].file_out)));}
			catch(MonitoringManagerException ex) { Debug.outln(ex.toString());}
		 }

  	}
  }

  // da rivedere eventuali sincronizzazioni...



  /** Invia l'evento passato come argomento al listener corrispondente */
  public void sendEvent(MonitoringManagerEvent ev) throws MonitoringManagerException{

    Debug.outln("MonitoringManager -> send " + ev);

  	if ( !listeners.isEmpty() ){
	    synchronized(listeners){
	    	Enumeration list = listeners.elements();
	      while (list.hasMoreElements()){
	      	MonitoringManagerListener listener = (MonitoringManagerListener)list.nextElement();
		      checkEventListener(listener, ev);
		    }
	    }

	  }
	else throw new MonitoringManagerException("MonitoringManager: sendEvent -> nessun listener registrato");
  }



  /** Associa l'evento al listener corrispondente.
    * L'ordine con cui viene fatto il check definisce una sorta di priorit
    * degli allarmi da gestire
    */
  public void checkEventListener(MonitoringManagerListener lis, MonitoringManagerEvent ev){

    Debug.outln("MonitoringManager -> checkEventListener");
  	if ((lis instanceof CpuAlarmsHandler) && (ev instanceof CpuEvent))
		  lis.alarm(ev);
	  if ((lis instanceof NetAlarmsHandler) && (ev instanceof NetEvent))
		  lis.alarm(ev);
	  if ((lis instanceof MemAlarmsHandler) && (ev instanceof MemEvent))
		  lis.alarm(ev);
	  if ((lis instanceof FileAlarmsHandler) && (ev instanceof FileEvent))
	 	  lis.alarm(ev);
  }



 /** Permette di attivare/disattivare i controlli */
 public void setCtrl(boolean s){

 	if (ctrl && !s) {
 		pollingTime=POLLING_TIME; // lo riporto al valore di default
 		Debug.outln("MonitoringManager -> controlli disattivati");
 		ctrl=s;
 		}
 	else  if (!ctrl && s){
 		 Debug.outln("MonitoringManager -> controlli attivati");
 		 ctrl=s;
 	}
 }



  /** Eccezione lanciata da MonitoringManager */
  public static class MonitoringManagerException extends Exception{

	  public MonitoringManagerException (String s){
	  	  super(s);
	  }
  }




}