package SOMA.network;

import java.io.*;
import java.net.*;
import java.util.*;
import SOMA.Environment;
import SOMA.network.connection.*;
import SOMA.explorer.*;
import SOMA.ext.Main.*;
import SOMA.ext.NetManagement.*;
import SOMA.naming.*;
import SOMA.naming.place.*;
import SOMA.naming.domain.*;
import SOMA.mobilePlace.*;

/** 
 * Gestore delle comunicazioni di rete di un {@link SOMA.Environment place}.
 * 
 * @author Livio Profiri
 * Revised by Alessandro Ghigi
 */

public class NetworkManager {
	
	Environment env;
	ConnectionServer connectionServer = null;
	protected Hashtable permanentConnections = new Hashtable();
	/** Informazioni sul place. */
	public PlaceInfo placeInfo;
	DirExplorerItem networkManagerDir;
	public DirExplorerItem connectionsDir;
	public int connectionCount = 1;
	/** Memorizza le connessioni con gli altri place dello stesso dominio. */
	public ConnectionStore connectionStore = new ConnectionStore();
	//public Vector clientConnections = new Vector();
	
	/** Costruttore. */
	public NetworkManager(Environment env,int port) throws IOException, ConnectionServer.ConnectionServerException, NameException {
		this.env = env;
		if(env.placeID.isDomain()) placeInfo = new DomainInfo(env.placeID,InetAddress.getLocalHost(),port);
		else placeInfo = new PlaceInfo(env.placeID,InetAddress.getLocalHost(),port);
		//Aggiungo 2 voci alla struttura delle directory.
		networkManagerDir = new DirExplorerItem("netManager");
		env.dir.addItem(networkManagerDir);
		networkManagerDir.addItem("placeInfo",new ObjectExplorerItem(placeInfo));
		// Eliminare questa: la lista si allunga ad ogni connessione.
		connectionsDir = new DirExplorerItem("connections");
		networkManagerDir.addItem( connectionsDir );
		// Aggiungo un'autoconnessione, in modo che i comandi spediti verso lo stesso place vengano eseguiti in loco
		Connection selfConnection = new SelfConnection(env);
		connectionStore.putConnection(env.placeID,selfConnection);
		connectionsDir.addItem("selfConnection",new ConnectionExplorerItem(selfConnection));
		// Avvio il server per le connessioni
		connectionServer = new ConnectionServer(port,100,new ExplorableConnectionFactory(env,connectionsDir,"servConn"));
		networkManagerDir.addItem("server",new DaemonExplorerItem(connectionServer));
		connectionServer.start();
		// Registrazione presso PNS ed eventualmente DNS
		if(env.placeID.isDomain()) env.domainNameService.putDomain((DomainInfo)placeInfo);
		env.placeNameService.putPlace(placeInfo);
		networkManagerDir.addItem("connList",new ExplorerItem("List of connections") {
			public Object Execute(Collection Parameters,PrintStream out) {
				connectionStore.printConnections(out);
				return null;
			}
		});
		networkManagerDir.addItem("send",new ExplorerItem("<PlaceID> <Message>") {
			public Object Execute(Collection Parameters,PrintStream out) {
				if(Parameters.size() > 1) {
					Iterator i = Parameters.iterator();
					PlaceID dest = null;
					try {
						dest = MobilePlaceID.parsePlaceID((String)i.next());
					}
					catch(NameException e) {
						out.println(e);
						return e;
					}
					SendMessageCommand Message = new SendMessageCommand((String)i.next());
					out.println("Sending command " + Message + " to " + dest);
					if(sendCommand(dest,Message)) {
						out.println("Message " + Message + " sent!");
						return new Boolean(true);
					}
					else {
						out.println("ERROR: Message " + Message + " NOT SENT!");
						return new Boolean(true);
					}
				}
				out.println("incorrect number of parameters");
				return null;
			}
		});
		networkManagerDir.addItem("greet",new ExplorerItem("<PlaceID>") {
			public Object Execute(Collection Parameters,PrintStream out) {
				if(Parameters.size() == 1) {
					Iterator i = Parameters.iterator();
					PlaceID dest = null;
					try {
						dest = MobilePlaceID.parsePlaceID((String)i.next());
					}
					catch(NameException e) {
						out.println(e);
						return e;
					}
					GreetingsCommand greet = new GreetingsCommand();
					if(sendCommand(dest,greet)) {
						return new Boolean(true);
					}
					else {
						out.println("ERROR: Message NOT SENT!");
						return new Boolean(false);
					}
				}
				out.println("incorrect number of parameters");
				return null;
			}
		});
		networkManagerDir.addItem("perm",new PermanentConnectionsExplorerItem());
	}
	
	/** 
	 * Spedizione di un comando ad un place.
	 */
	public boolean sendCommand(PlaceID destID,Command aCommand) {
		boolean answer = false;
		PlaceID myDomainID;
		if(env.placeID instanceof MobilePlaceID) myDomainID = ((MobileEnvironment)env).currentDomainID;
		else myDomainID = env.placeID.getDomainID();
		if(destID instanceof MobilePlaceID) answer = sendCommandToMobilePlace((MobilePlaceID)destID,aCommand);
		// Stesso dominio
		else if(myDomainID.sameDomain(destID)) {
			// Sono gi nel dominio di destinazione
			answer = sendCommandToPlace(destID,aCommand);
			// Prova a spedirlo come se fosse un MobilePlaceID
			// questo serve per  spedire verso un place mobile i comandi inviati
			// erroneamente all'indirizzo giusto, ma di tipo PlaceID invece che MobilePlaceID
			// Non  necessario, quindi si pu togliere se crea problemi!!!
			if(answer == false) answer = sendCommandToMobilePlace(new MobilePlaceID(destID),aCommand);
		}
		// Se il place  registrato nel pns, ma non appartiene al dominio,
		// prova a spedirlo come se fosse un MobilePlaceID.
		// Questo serve per  spedire verso un place mobile i comandi inviati
		// erroneamente all'indirizzo giusto, ma di tipo PlaceID invece che MobilePlaceID
		// Non  necessario, quindi si pu togliere se crea problemi!!!
		else if(env.placeNameService.getPlace(destID) != null) answer = sendCommandToMobilePlace(new MobilePlaceID(destID),aCommand);
		else if(env.placeID.isDomain()) { // Sono in un default place
			Command toSend;
			if(destID.isDomain()) toSend = aCommand; // destinazione: un altro dominio
			else toSend = new TransportCommand(destID,aCommand); // destinazione: un place di un altro dominio
			answer = sendCommandToDomain(destID.getDomainID(),toSend);
		}
		// Sono in un place normale e devo spedire un messaggio ad un altro dominio
		else answer = sendCommandToPlace(myDomainID,new TransportCommand(destID,aCommand));
		return answer;
	}
	
	/** 
	 * Spedizione di un comando ad un place mobile.
	 * Restituisce true se la spedizione ha avuto successo.
	 */
	private boolean sendCommandToMobilePlace(MobilePlaceID destID,Command aCommand) {
		boolean answer = sendCommandToPlace(destID,aCommand); // Potrebbe essere in questo dominio!
		if(answer == false) { // Il place non era nel dominio corrente
			if(!destID.getHome().equals(env.placeID))  // Spedisce il comando alla Home del place mobile
				answer = sendCommand(destID.getHome(),new TransportCommand(destID,aCommand));
			else { // Comando arrivato nella Home del place mobile.
				PlaceID currentDomainID = env.mobilePlaceManager.getPosition(destID);
				// Se il place mobile non  disconnesso e il suo dominio non  quello attuale (=evito cicli)
				// lo spedisco al dominio di destinazione.
				if( currentDomainID != null && !currentDomainID.equals(MobilePlaceManager.DISCONNECTED) && !currentDomainID.equals(env.placeID))
					answer = sendCommand(currentDomainID,new TransportCommand(destID,aCommand));
			}
		}
		return answer;
	}
	
	/**
	 * Sends a command to a place in the same domain.
	 * E' privato perch non testo il rispetto del vincolo di appartenenza allo stesso dominio.
	 * occhio: la connessione rimane attiva e memorizzata nel PlaceNameService
	 */
	private boolean sendCommandToPlace(PlaceID placeID,Command aCommand) {
		boolean returnValue = false;
		Connection connection = connectionStore.getConnection(placeID);
		// Se la connessione non c' o  disattiva, la riattivo
		if(connection == null || connection.getStatus() != Daemon.ON) {
			PlaceInfo pi = env.placeNameService.getPlace(placeID);
			if(pi != null) {
				try {
					connection = new Connection(new Socket(pi.host,pi.port),env);
					connectionStore.putConnection(placeID,connection);
					connectionsDir.addItem(placeID.place + connectionCount++,new ConnectionExplorerItem(connection));
					connection.start();
					connection.send(new ConnectionCommand(env.placeID));
				}
				catch(Exception e) {
					e.printStackTrace(env.err);
					returnValue = false;
				}
			}
		}
		// Se ora  attiva, procedo alla spedizione del comando
		if(connection != null && connection.getStatus() == Daemon.ON) {
			try {
				connection.send(aCommand);
				returnValue = true;
			}
			catch(Exception e) {
				e.printStackTrace(env.err);
				returnValue = false;
			}
		}
		return returnValue;
	}
	
	/**
	 * Spedisco un comando ad un dominio:
	 * - creo una connessione
	 * - NON avvio il thread in ascolto
	 * - spedisco il comando
	 * - chiudo la connessione = le SOCKET
	 * - Restituisco true se la connessione non  in ERROR, ossia se il messaggio  arrivato
	 */
	private boolean sendCommandToDomain(PlaceID domainID,Command aCommand) {
		boolean returnValue = false;
		Connection connection = connectionStore.getConnection(domainID);
		boolean isPermanent = isPermanentConnection(domainID) > 0;
		boolean connectionCreated = false;
		// Se la connessione non c' o  disattiva, la riattivo
		if(connection == null || connection.getStatus() != Daemon.ON) {
			DomainInfo di = env.domainNameService.getDomain(domainID);
			if(di != null) {
				try {
					connection = new Connection(new Socket(di.host,di.port),env);
					connectionsDir.addItem(domainID.domain + connectionCount++,new ConnectionExplorerItem(connection));
					// Solo se la connessione  permanente
					if(isPermanent) {
						connectionStore.putConnection(domainID,connection);
						connection.start();  // La avvio solo se  permanente
						connection.send(new ConnectionCommand(env.placeID));
					}
					connectionCreated = true;
				}
				catch(Exception e) {
					e.printStackTrace(env.err);
					returnValue = false;
				}
			}
		}
		// Se ora  attiva, procedo alla spedizione del comando
		if(connection != null) {
			try {
				connection.send(aCommand);
				returnValue = true;
				// Se non era permanente, non ho fatto lo start, quindi anche se ora lo  diventata
				// la arresto!
				if(!isPermanent && connectionCreated) connection.stop();
				// Interrompo la connessione solo se non  permanente e l'ho creata io.
				// Se non  permanente e non l'ho creata io, qualcun altro si occuper di chiuderla,
				// ad esempio l'altro pari.
			}
			catch(Exception e) {
				e.printStackTrace(env.err);
				returnValue = false;
			}
		}
		return returnValue;
	}
	
	/**
	 * Utilizzata per spedire un comando ad una coppia host:port.
	 * In questo caso la connessione viene attivata e dovr essere disattivata.
	 * Spedizione di un comando ad un server di indirizzo: host:port.
	 * In seguito, magari da un comando apposito:
	 * - creo una connessione
	 * - avvio il thread in ascolto
	 * - spedisco il comando
	 * - NON chiudo la connessione
	 * - Restituisco true in caso di successo
	 */
	public boolean sendCommand(InetAddress host,int port,Command aCommand) {
		boolean returnValue = true;
		try {
			Connection connection = new Connection(new Socket(host,port),env);
			connectionsDir.addItem("clientConn" + connectionCount++,new ConnectionExplorerItem(connection));
			connection.start();
			connection.send(aCommand);
			returnValue = true;
		}
		catch(Exception e) {
			//e.printStackTrace(env.err);
			returnValue = false;
		}
		return returnValue;
	}
	
	/**
	 * Tenta di connettersi al Master in fase di presentazione
	 */
	public void contactMaster(InetAddress host,int port) throws Exception {
		Connection connection = new Connection(new Socket(host,port),env);
		connectionsDir.addItem("masterConnection",new ConnectionExplorerItem(connection));
		connection.start();
		env.repConnection = connection;
		connection.send(new MeetCommand());
	}
	
	/**
	 * Tenta di connettersi al Master in fase di riattivazione
	 */
	public void contactMaster2(InetAddress host,int port) throws Exception {
		Connection connection = new Connection(new Socket(host,port),env);
		connectionsDir.addItem("masterConnection",new ConnectionExplorerItem(connection));
		connection.start();
		env.repConnection = connection;
	}
	
	/** Richiede una connessione stabile con il place specificato.
	 * Questo metodo altera la politica di gestione delle connessioni fra domini.
	 * Normalmente le connessioni fra domini vengono attivate al momento della spedizione di un messaggio
	 * e disattivate subito dopo. Dopo una chiamata a startPermanentConnection, se siamo
	 * in un default place e se il place specificato appartiene ad un altro dominio, allora
	 * la connessione diretta a quel dominio non verra' interrotta dopo la spedizione di un messaggio,
	 * ma rimarra' attiva, pronta per la spedizione di eventuali successivi messaggi.
	 * E' possibile ritornare al funzionamento normale con una chiamata a stopPermanentConnection(placeID).
	 * E' previsto un meccanismo per gestire pi chiamate successive al metodo
	 * startPermanentConnection: ad ogni connessione e' associato un intero
	 * uguale a (NUMERO di start) - (NUMERO di stop). Questo intero corrisponde al numero
	 * di "procedure" che hanno fatto richiesta di connessione stabile e non hanno ancora
	 * terminato il loro compito. Quando l'intero va a zero si comunica all'altro pari
	 * la possibilit di interrompere la connessione. Se anche all'altro capo della connessione
	 * nessuno ne ha piu' bisogno, la connessione viene terminata.
	 * Restituisce (NUMERO di start effettuati) - (NUMERO di stop)
	 */
	public int startPermanentConnection(PlaceID placeID) {
		int returnValue = 0;
		if(env.placeID.isDomain() && !env.placeID.sameDomain(placeID)) {
			PlaceID domainID = placeID.getDomainID();
			Integer i = (Integer)permanentConnections.get(domainID);
			if(i != null) returnValue = i.intValue() + 1;
			else returnValue = 1;
			permanentConnections.put(domainID,new Integer(returnValue));
		}
		if(env.repConnection != null && !env.slave) {
			try {
				env.repConnection.send(new SlavePermConnectionRefreshCommand(placeID,1));
			}
			catch(Exception e) {
				e.printStackTrace();
			}
		}
		return returnValue;
	}
	
	/**
	 * Avvia una connessione stabile con il place specificato.
	 * Restituisce (NUMERO di start effettuati) - (NUMERO di stop)
	 */
	public int stopPermanentConnection(PlaceID placeID) {
		PlaceID domainID = placeID.getDomainID();
		int returnValue = isPermanentConnection(domainID);
		// Se ci sono ancora threads che hanno bisogno della connessione
		if(returnValue > 1) {
			returnValue--;
			permanentConnections.put(domainID,new Integer(returnValue));
		}
		else if(env.placeID.isDomain() && !env.placeID.sameDomain(placeID)) { // Se la conessione non serve a nessuno
			if(returnValue == 1) {
				returnValue--;
				permanentConnections.remove(domainID); // Rimuovo il contatore.
			}
			Connection connection = connectionStore.getConnection(domainID);
			if(connection != null) {
				// Se la connessione  attiva, chiedo all'altro pari se  il caso di disattivarla
				if(connection.getStatus() == Daemon.ON) {
					try {
						// Chiedo all'altro pari se  il caso di disattivare la connessione stabile
						connection.send(new StopConnectionCommand(env.placeID));
					}
					catch(Exception e) {
						e.printStackTrace(env.err);
					}
				}
				// Rimuovo la connessione solo se  inattiva
				else connectionStore.removeConnection(domainID);
			}
		}
		if(env.repConnection != null && !env.slave) {
			try {
				env.repConnection.send(new SlavePermConnectionRefreshCommand(placeID,2));
			}
			catch(Exception e) {
				e.printStackTrace();
			}
		}
		return returnValue;
	}
	
	/**
	 * Restituisce l'intero relativo alla connessione stabile con il place specificato.
	 * Restituisce (NUMERO di start effettuati) - (NUMERO di stop)
	 */
	public int isPermanentConnection(PlaceID placeID){
		int returnValue = 0;
		Integer i = (Integer)permanentConnections.get(placeID.getDomainID());
		if(i != null) returnValue = i.intValue();
		return returnValue;
	}
	
	public String toString() {
		return "[netManager]";
	}
	
	/** 
	 * Gestisce le connessioni permanenti fra domini.
	 */
	public class PermanentConnectionsExplorerItem extends ExplorerItem {
		public PermanentConnectionsExplorerItem() {
			super("[\"placeID\" [start | stop]]");
		}
		public Object Execute(Collection Parameters,PrintStream out) {
			if(Parameters.size() == 0) {
				out.println("List of domain permanent connections:");
				out.println();
				Iterator i = permanentConnections.entrySet().iterator();
				for(int j=1;i.hasNext();j++) {
					Map.Entry e = (Map.Entry)i.next();
					out.println("  " + j + ") " + e.getKey() + " --> " + e.getValue());
				}
				out.println();
			}
			else {
				Iterator i = Parameters.iterator();
				PlaceID placeID = null;
				try {
					placeID = new PlaceID((String)i.next());
				}
				catch(NameException e) {
					out.println( "Invalid place ID: " + e );
					return e;
				}
				int result = 0;
				if(i.hasNext()) {
					String operation = (String)i.next();
					if("start".equals(operation)) result = startPermanentConnection(placeID);
					else if("stop".equals(operation)) result = stopPermanentConnection(placeID);
					else {
						out.println("Unknown operation: " + operation);
						return null;
					}
				}
				else result = isPermanentConnection( placeID );
				out.println("Number of pocesses using connection to " + placeID + ": " + result);
				return new Integer(result);
			}
			return null;
		}
	}
	
	/** Restituisce le connessioni permanenti. */
	public Hashtable getPermanentConnections() {
		return this.permanentConnections;
	}
	
	/** Imposta le connessioni permanenti. */
	public void setPermanentConnections(Hashtable permanentConnections) {
		this.permanentConnections = permanentConnections;
	}
	
	/** Esegue il refresh delle connessioni permanenti uscenti. */
	public void refreshPermanentConnections() {
		Iterator i = permanentConnections.entrySet().iterator();
		for(int j=1;i.hasNext();j++) {
			Map.Entry e = (Map.Entry)i.next();
			PlaceID temp = (PlaceID)e.getKey();
			env.out.println("Refreshing Connection info with " + temp);
			sendCommand(temp,new ReqAliveCommand());
		}
	}
	
	/** Esegue il refresh delle connessioni permanenti entranti. */
	public void refreshPeerPermanentConnections(PlaceID placeID,PlaceID sender) {
		Iterator i = permanentConnections.entrySet().iterator();
		for(int j=1;i.hasNext();j++) {
			Map.Entry e = (Map.Entry)i.next();
			PlaceID temp = (PlaceID)e.getKey();
			if(temp.equals(placeID)) {
				env.out.println("Refreshing Connection info with " + temp);
				sendCommand(temp,new ReqAliveCommand());
			}
		}
		env.domainNameService.sendToAllDomains(new SlavePeerConnectionRefreshCommand(placeID,env.placeID),sender);
	}
	
}