package SOMA.naming.place;

import java.io.*;
import java.util.*;
import java.net.InetAddress;
import SOMA.ext.PNSManagement.*;
import SOMA.naming.*;
import SOMA.network.connection.*;
import SOMA.Environment;

/** 
 * Servizio di nomi di place, PNS.
 * Il servizio realizza la metafora di una tabella in cui vengono memorizzate
 * le PlaceInfo. La chiave di ricerca e' di tipo PlaceID, identificatore del place.
 * Affinche' un place entri a far parte di un dominio e' necessario che il PNS del place
 * si registri presso il PNS del default place del dominio.
 * In ogni momento il PNS puo' richiedere l'aggiornamento della tabella dei place
 * del dominio sia al default place, sia a un place qualsiasi, ove questo sia necessario.
 *
 * @author Livio Profiri
 * Revised by Alessandro Ghigi
 */

public class PlaceNameService implements Serializable {
	
	transient protected Environment env;
	protected Hashtable places = new Hashtable();
	
	/** Costruttore. env  l'Environment associato al Place */
	public PlaceNameService(Environment env) {
		this.env = env;
	}
	
	/** Restituisce la PlaceInfo corrispondente a placeID, o null se non trovata. */
	synchronized public PlaceInfo getPlace(PlaceID placeID) {
		return (PlaceInfo)places.get(placeID);
	}
	
	/** 
	 * Inserisce una PlaceInfo.
	 * Se e' un default place e PlaceInfo e' nuova, aggiorna gli altri place.
	 * Restituisce il vecchio valore memorizzato o null se si tratta di un nuovo place.
	 */
	synchronized public PlaceInfo putPlace(PlaceInfo newPlaceInfo) {
		PlaceInfo oldPlaceInfo = (PlaceInfo)places.put(newPlaceInfo.placeID,newPlaceInfo);
		if(env.placeID.isDomain() && (!newPlaceInfo.equals(oldPlaceInfo))) {
			sendToAllPlaces(new PutPlaceCommand(newPlaceInfo),newPlaceInfo.placeID);
		}
		return oldPlaceInfo;
	}
	
	/** 
	 * Inserisce una PlaceInfo.
	 * Se e' un default place e PlaceInfo e' nuova, aggiorna gli altri place.
	 * Restituisce il vecchio valore memorizzato o null se si tratta di un nuovo place.
	 */
	synchronized public PlaceInfo putPlaceSlave(PlaceInfo newPlaceInfo) {
		PlaceInfo oldPlaceInfo = (PlaceInfo)places.put(newPlaceInfo.placeID,newPlaceInfo);
		if(env.placeID.isDomain() && (!newPlaceInfo.equals(oldPlaceInfo))) {
			sendToAllPlaces(new PutPlaceCommand(newPlaceInfo),newPlaceInfo.placeID);
			// Invio aggiornamento allo slave
			if(env.repConnection != null && env.repConnection.getStatus() == Daemon.ON) {
				try {
					env.repConnection.send(new SlavePNSTableRefreshCommand(newPlaceInfo));
				}
				catch(Exception e) {
					e.printStackTrace(env.err);
				}
			}
		}
		return oldPlaceInfo;
	}
	
	/** 
	 * Elimina la placeInfo corrispondente a aPlaceID.
	 * Se e' un default place e aPlaceID e' presente, aggiorna gli altri place.
	 * Restituisce il vecchio valore memorizzato o null se si tratta di un nuovo place.
	 */
	synchronized public PlaceInfo removePlace(PlaceID aPlaceID) {
		PlaceInfo oldPlaceInfo = (PlaceInfo)places.remove(aPlaceID);  // Restituisce il vecchio valore o null.
		env.networkManager.connectionStore.removeConnection(aPlaceID);
		if(env.placeID.isDomain() && (oldPlaceInfo != null)) sendToAllPlaces(new RemovePlaceCommand(aPlaceID));
		return oldPlaceInfo;
	}
	
	/** 
	 * Elimina la placeInfo corrispondente a aPlaceID.
	 * Se e' un default place e aPlaceID e' presente, aggiorna gli altri place.
	 * Restituisce il vecchio valore memorizzato o null se si tratta di un nuovo place.
	 */
	synchronized public PlaceInfo removePlaceSlave(PlaceID aPlaceID) {
		PlaceInfo oldPlaceInfo = (PlaceInfo)places.remove(aPlaceID);  // Restituisce il vecchio valore o null.
		env.networkManager.connectionStore.removeConnection(aPlaceID);
		if(env.placeID.isDomain() && (oldPlaceInfo != null)) {
			sendToAllPlaces(new RemovePlaceCommand(aPlaceID));
			// Invio aggiornamento allo slave
			if(env.repConnection != null && env.repConnection.getStatus() == Daemon.ON) {
				try {
					if(env.repConnection != null && env.repConnection.getStatus() == Daemon.ON) {
						env.repConnection.send(new SlavePNSRemoveCommand(aPlaceID));
					}
				}
				catch(Exception e) {
					e.printStackTrace(env.err);
				}
			}
		}
		return oldPlaceInfo;
	}
	
	/** Stampa la lista di tutti i place del dominio su out.*/
	public void listPlaces(PrintStream out) {
		out.println("I know " + places.size() + " places:");
		out.println();
		int i;
		Enumeration list;
		for(i=1,list=places.elements();list.hasMoreElements();) out.println("    " + i++ + ") " + list.nextElement().toString());
		out.println();
	}
	
	/** 
	 * Registrazione presso il PNS del default place del dominio.
	 * Viene inviato un PlaceRegisterCommand al default place, individuato dalla coppia host:port.
	 */
	public boolean register(InetAddress host,int port) {
		return env.networkManager.sendCommand(host,port,new PlaceRegisterCommand(env.networkManager.placeInfo));
	}
	
	/** 
	 * Richiesta di refresh della tabella al PNS del default place.
	 * Viene inviato un PlaceRefreshCommand.Request al PNS del default place.
	 */
	public boolean refresh() {
		if(env.placeID.isDomain()) {
			env.err.println("Can't refresh a default place!");
			return false;
		}
		else {
			return env.networkManager.sendCommand(env.placeID.getDomainID(),new PlaceRefreshCommand.Request());
		}
	}
	
	/** 
	 * Richiesta di refresh della tabella al PNS di un place qualsiasi.
	 * Viene inviato un PlaceRefreshCommand.Request al PNS del place
	 * individuato dalla coppiahost:port.
	 */
	public boolean refresh(InetAddress host,int port) {
		return env.networkManager.sendCommand(host,port,new PlaceRefreshCommand.Request());
	}
	
	/** Spedisce un comando a tutti i place della tabella. */
	public void sendToAllPlaces(Command command) {
		sendToAllPlaces(command,null);
	}
	
	/** Spedisce un comando a tutti i place della tabella, escluso il place DontSendMe. */
	// Salto il place DontSendMe
	public void sendToAllPlaces(Command command,PlaceID DontSendMe) {
		env.out.println("  Sending " + command + " to every place in my domain:");
		env.out.println();
		int i;
		Enumeration list;
		list = places.elements();
		for(i=1;list.hasMoreElements();) {
			PlaceInfo dest = (PlaceInfo)list.nextElement();
			// Attenzione: il metodo sendcommand che usa il PlaceInfo  privato: quindi
			// accedo una seconda volta alla hashtable
			if(env.placeID.equals(dest.placeID)) env.err.println("Skipping " + dest);
			else if(dest.placeID.equals(DontSendMe)) env.err.println("Skipping " + dest);
			else env.out.println("    " + i++ + ") " + dest + " result: " + env.networkManager.sendCommand(dest.placeID,command));
		}
		env.out.println();
	}
	
	/** 
	 * Restituisce un array contenente tutti gli identificatori di domino.
	 * Questa funzionalita' viene fornita agli agenti.
	 */
	public PlaceID[] getPlacesArray() {
		return (PlaceID[])places.keySet().toArray(new PlaceID[places.size()]);
	}
	
	/** Aggiorna il riferimento all'Environment. Usato nel caricamento da disco. */
	public void setEnv(Environment env) {
		if( this.env == null ) this.env = env;
	}
	
	public String toString() {
		return "[pns: " + places.size() + " places]";
	}
	
	public Hashtable getPlaces() {
		return this.places;
	}
	
	public void setPlaces(Hashtable places) {
		this.places = places;
	}
	
}