package SOMA.naming.domain;

import java.io.*;
import java.util.*;
import java.net.InetAddress;
import SOMA.ext.DNSManagement.*;
import SOMA.naming.*;
import SOMA.network.connection.*;
import SOMA.Environment;

/** 
 * Servizio di nomi di dominio, DNS.
 * Il servizio realizza la metafora di una tabella in cui vengono memorizzate
 * le DomainInfo. La chiave di ricerca e' di tipo PlaceID, identificatore del default place del dominio.
 * Questo servizio e' presente solo nel default place di ogni dominio.
 * Infatti, gli altri place hanno informazioni sul solo dominio di appartenenza.
 * E' previsto un sistema gerarchico di scambio di informazioni fra DNS di domini diversi.
 * Ogni DNS ha 1 DNS padre, presso cui e' registrato, e una serie di DNS figli
 * che si sono registrati presso di lui. In questo modo si realizza una struttura ad albero
 * in cui le informazioni passano dai figli al padre e viceversa.
 *
 * @author Livio Profiri
 * Revised by Alessandro Ghigi
 */

public class DomainNameService implements Serializable {
	
	transient Environment env;
	Hashtable domains = new Hashtable();
	PlaceID fatherDNS = null;
	Set childrenDNS = Collections.synchronizedSet(new HashSet());
	
	/** Costruttore. env  l'Environment associato al Place */
	public DomainNameService(Environment env) {
		this.env = env;
	}
	
	/** Restituisce la DomainInfo corrispondente a placeID, o null se non trovata. */
	synchronized public DomainInfo getDomain(PlaceID placeID) {
		return (DomainInfo)domains.get(placeID);
	}
	
	/** 
	 * Inserisce una DomainInfo.
	 * Se DomainInfo e' nuova aggiorna DNS padre e figli.
	 * Restituisce il vecchio valore memorizzato o null se si tratta di un nuovo dominio.
	 */
	synchronized public DomainInfo putDomain(DomainInfo newDomainInfo) {
		DomainInfo oldDomainInfo = (DomainInfo)domains.put(newDomainInfo.placeID,newDomainInfo);
		if(!newDomainInfo.equals(oldDomainInfo)) {
			sendToAllDomains(new PutDomainCommand(newDomainInfo));
		}
		return oldDomainInfo;
	}
	
	/** 
	 * Inserisce una DomainInfo.
	 * Se DomainInfo e' nuova aggiorna DNS padre e figli.
	 * Restituisce il vecchio valore memorizzato o null se si tratta di un nuovo dominio.
	 */
	synchronized public DomainInfo putDomainSlave(DomainInfo newDomainInfo) {
		DomainInfo oldDomainInfo = (DomainInfo)domains.put(newDomainInfo.placeID,newDomainInfo);
		if(!newDomainInfo.equals(oldDomainInfo)) {
			sendToAllDomains(new PutDomainCommand(newDomainInfo));
			// Invio aggiornamento allo slave
			if(env.repConnection != null && env.repConnection.getStatus() == Daemon.ON) {
				try {
					env.repConnection.send(new SlaveDNSTableRefreshCommand(newDomainInfo));
				}
				catch(Exception e) {
					e.printStackTrace(env.err);
				}
			}
		}
		return oldDomainInfo;
	}
	
	/** 
	 * Elimina la DomainInfo corrispondente aaDomainID.
	 * Se aDomainID e' presente trasmette l'operazione a DNS padre e figli.
	 * Restituisce il vecchio valore memorizzato o null se si tratta di un nuovo dominio.
	 */
	// Cancello il vecchio dominio anche dalla lista dei figli e dal padre, se necessario
	synchronized public DomainInfo removeDomain(PlaceID aDomainID) {
		DomainInfo oldDomainInfo = (DomainInfo)domains.remove(aDomainID);  // Restituisce il vecchio valore o null.
		childrenDNS.remove(aDomainID);
		if(aDomainID.equals(fatherDNS)) fatherDNS = null;
		if(oldDomainInfo != null) sendToAllDomains(new RemoveDomainCommand(aDomainID));
		return oldDomainInfo;
	}
	
	/** 
	 * Elimina la DomainInfo corrispondente aaDomainID.
	 * Se aDomainID e' presente trasmette l'operazione a DNS padre e figli.
	 * Restituisce il vecchio valore memorizzato o null se si tratta di un nuovo dominio.
	 */
	// Cancello il vecchio dominio anche dalla lista dei figli e dal padre, se necessario
	synchronized public DomainInfo removeDomainSlave(PlaceID aDomainID) {
		DomainInfo oldDomainInfo = (DomainInfo)domains.remove(aDomainID);  // Restituisce il vecchio valore o null.
		childrenDNS.remove(aDomainID);
		if(aDomainID.equals(fatherDNS)) fatherDNS = null;
		if(oldDomainInfo != null) {
			sendToAllDomains(new RemoveDomainCommand(aDomainID));
			// Invio aggiornamento allo slave
			if(env.repConnection != null && env.repConnection.getStatus() == Daemon.ON) {
				try {
					env.repConnection.send(new SlaveDNSRemoveCommand(aDomainID));
				}
				catch(Exception e) {
					e.printStackTrace(env.err);
				}
			}
		}
		return oldDomainInfo;
	}
	
	/** Stampa la lista di tutti i domini su out. */
	public void listDomains(PrintStream out) {
		out.println("I know " + domains.size() + " domains:");
		out.println();
		int i;
		Enumeration list;
		for(i=1,list=domains.elements();list.hasMoreElements();)
			out.println("    " + i++ + ") " + list.nextElement().toString());
		out.println();
		if(fatherDNS != null) out.println("  My father Domain Name Server is " + fatherDNS);
		if(childrenDNS.size() > 0) {
			out.println("  I have " + childrenDNS.size() + " children Domain Name Servers:"  );
			out.println();
			Iterator it;
			for(i=1,it = childrenDNS.iterator();it.hasNext();)
				out.println("    " + i++ + ") " + it.next().toString());
			out.println();
		}
	}
	
	/** 
	 * Registrazione presso il DNS padre.
	 * Viene inviato un DomainRegisterCommand al DNS padre individuato dalla coppia host:port.
	 */
	public boolean register(InetAddress host,int port) {
		// Il cast viene fatto per sicurezza: non voglio registrare un place per errore.
		// A runtime viene lanciata un'eccezione
		return env.networkManager.sendCommand(host,port,new DomainRegisterCommand((DomainInfo)env.networkManager.placeInfo));
	}
	
	/** Richiesta di refresh della tabella al DNS padre.
	 * Viene inviato un DomainRefreshCommand.Request al DNS padre.
	 */
	public boolean refresh() {
		boolean answer = false;
		DomainInfo di = getDomain(fatherDNS);
		if(di != null) answer = refresh(di.host,di.port);
		return answer;
	}
	
	/** 
	 * Richiesta di refresh della tabella al DNS padre.
	 * Viene inviato un DomainRefreshCommand.Request al DNS padre, individuato dalla coppia host:port.
	 * Attenzione: da usare solo quando il dominio padre cambia host:port, altrimenti si viene registrati in due DNS diversi,
	 * col risultato di avere due DNS padre che spediscono le informazioni di aggiornamento.
	 */
	public boolean refresh(InetAddress host,int port) {
		return env.networkManager.sendCommand(host,port,new DomainRefreshCommand.Request(true));
	}
	
	/** 
	 * Spedisce un comando al DNS padre.
	 * Restituisce true se la spedizione ha avuto successo.
	 */
	public boolean sendToFatherDomain(Command command) {
		boolean returnValue = false;
		if(fatherDNS != null) {
			returnValue = env.networkManager.sendCommand(fatherDNS,command);
			env.out.println("  Sending command " + command + " to father " + fatherDNS + " result: " + returnValue);
			env.out.println();
		}
		return returnValue;
	}
	
	/** Spedisce un comando a tutti i DNS figli. */
	public void sendToChildrenDomains(Command command) {
		if(childrenDNS.size() > 0) {
			env.out.println("  Sending " + command + " to " + childrenDNS.size() + " children DNSs: ");
			env.out.println();
			int i;
			Iterator it;
			for(i=1,it = childrenDNS.iterator();it.hasNext();) {
				PlaceID dest = (PlaceID)it.next();
				env.out.println("    " + i++ + ") " + dest + " result: " + env.networkManager.sendCommand(dest,command));
			}
			env.out.println();
		}
	}
	
	/** Spedisce un comando al DNS padre e a tutti i DNS figli. */
	public void sendToAllDomains(Command command) {
		sendToFatherDomain(command);
		sendToChildrenDomains(command);
	}
	
	/** Spedisce un comando al DNS padre e ai figli eccetto che al mittente. */
	public void sendToAllDomains(Command command,PlaceID dontSendMe) {
		if(!dontSendMe.equals(fatherDNS)) sendToFatherDomain(command);
		if(childrenDNS.size() > 0) {
			env.out.println("  Sending " + command + " to " + childrenDNS.size() + " children DNSs: ");
			env.out.println();
			int i;
			Iterator it;
			for(i=1,it = childrenDNS.iterator();it.hasNext();) {
				PlaceID dest = (PlaceID)it.next();
				if(!dontSendMe.equals(dest)) env.out.println("    " + i++ + ") " + dest + " result: " + env.networkManager.sendCommand(dest,command));
			}
			env.out.println();
		}
	}
	
	/** 
	 * Restituisce un array contenente tutti gli identificatori di domino.
	 * Questa funzionalita' viene fornita agli agenti.
	 */
	public PlaceID[] getDomainsArray() {
		return (PlaceID[])domains.keySet().toArray(new PlaceID[domains.size()]);
	}
	
	/** Aggiorna il riferimento all'Environment. Usato nel caricamento da disco. */
	public void setEnv( Environment env ) {
		if(this.env == null) this.env = env;
	}
	
	public String toString() {
		return "[dns: " + domains.size() + " domains]";
	}
	
	/** Rende il PlaceID del DNS padre. */
	public PlaceID getParentDNS() {
		return fatherDNS;
	}
	
	/** 
	 *  Rende i PlaceID dei DNS figli (zero, uno o pi). 
	 *  Rende un Vector contenente oggetti PlaceID!
	 */
	public Vector getChildrenDNS() {
		Vector reso = new Vector();
		Iterator it = childrenDNS.iterator();
		while(it.hasNext()) {
			PlaceID bimbo = (PlaceID)it.next();
			reso.addElement(bimbo);
		}
		return reso;
	}
	
	/** Imposta il riferimento al padre. */
	public void setFatherDNS(PlaceID fatherDNS) {
		this.fatherDNS = fatherDNS;
	}
	
	/** Imposta i nodi figli. */ 
	public void setChildrenDNS(Vector childrenDNS) {
		for(int i=0;i<childrenDNS.size();i++) {
			this.childrenDNS.add((PlaceID)childrenDNS.get(i));
		}
	}
	
	/** Restituisce i domini. */
	public Hashtable getDomains() {
		return this.domains;
	}
	
	/** Imposta i domini. */
	public void setDomains(Hashtable domains) {
		this.domains = domains;
	}
	
}