/**  SportelloRichieste
 *     Thread che aspetta le richieste di connessione da parte delle "applet" remote.
 *     E` specifico per un certo Place (infatti alla creazione serve un "ActionPlace", 
 *     il quale  per sua natura specifico per un solo Place.
 *     Questo thread si occupa soltanto di accettare le richieste. A ogni richiesta viene
 *     generato un THREAD SEPARATO che esegue la comunicazione vera e propria tramite il
 *     classico modello a scambio di messaggi.
 *     Viene usato il protocollo TCP (e non UDP) in modo che dopo la richiesta la comunicazione
 *     avvenga su una socket separata (scelta dal sistema).
 *     In questa maniera si possono gestire pi comunicazioni separatamente.
 *     Inoltre si beneficia dell'affidabilit della comunicazione offerta da TCP.
 *     Usare UDP non porterebbe a nessun benificio perch il tempo impiegato a eseguire 
 *     un'operazione (come il lancio di un agente) influisce significativamente sul tempo 
 *     complessivo.
 *     NB:  un' oggetto ci classe {@link SOMA.network.connection.Daemon}, cosicch si possa 
 *     attivare o disattivare.
 *     Inoltre si pu facilmente inserirlo nel men a linea di comando di SOMA e.
 *     @autore     Luigi Antenucci
 *     @versione   2.2
 *     @linguaggio jdk 1.2
 */

package SOMA.gui.remotegui;

import SOMA.gui.ActionPlace;
import SOMA.gui.Debug;
import SOMA.network.connection.Daemon;
import SOMA.Environment;

import java.lang.Thread;
import java.net.Socket;
import java.net.ServerSocket;
import java.net.SocketException;
import java.io.IOException;


public class SportelloRichieste implements Daemon, Runnable {

      /**
       *  Porta TCP di default su cui lo sportello si mette in attesa di messaggi.
       *  Poich si presuppone un place per ogni computer, basta usare questa; ma NON si 
       *   obbligati perch il costruttore pu avere il parametro "porta".
       */
  public static final  int PORTA_DEFAULT = 5555;

  // Variabili d'istanza:
      /**
       *  Socket usata su cui attendere i messaggi.
       */
  protected ServerSocket serSock;

      /**
       *  Stato dello sportello: ON=aperto (accetta messaggi), OFF=chiuso (riprova un altro giorno!).
       *  Non l'ho definito io, lo si eredita dall'interfaccia {@link SOMA.network.connection.Daemon}.
       */
  protected Object statoSportello;

      /**
       *  Parametri passati al costruttore.
       */
  protected ActionPlace actionPlace;
  protected int         portaTCP;


      /**
       *  Costruttore di questo sportello di richieste.
       *  Serve le richieste per un certo ActionPlace.
       *  I messaggi devono arrivare sulla porta TCP di default (vedi costante "PORTA_DEFAULT").
       *  NB: lo sportello viene creato MA NON LANCIATO. usare il metodo "start".
       */
  public SportelloRichieste (ActionPlace actionPlace) {
    this (actionPlace, PORTA_DEFAULT);
  } //costruttore

      /**
       *  Costruttore di questo sportello di richieste.
       *  Serve le richieste per un certo ActionPlace; i messaggi devono arrivare sulla porta TCP data.
       *  NB: lo sportello viene creato MA NON LANCIATO. usare il metodo "start".
       */
  public SportelloRichieste (ActionPlace actionPlace, int portaTCP) {
    Debug.outln ("ActionPlace - in creazione per l'ACTION-PLACE "+actionPlace.toString());
    this.actionPlace = actionPlace;
    this.portaTCP    = portaTCP;

    statoSportello = OFF;

  } //costruttore

      /**
       *  Viene lanciato questo demone di "ascolto" delle richieste.
       *  Nota: se il demone  gi stato lanciato, non viene fatto nulla.
       *  Metodo richiesto dall'interfaccia {@link SOMA.network.connection.Daemon}.
       *  Il codice che viene eseguito dal Thread  nel metodo "run()".
       */
  public void start () throws Exception {
    if (statoSportello == OFF) {
      Debug.outln ("SportelloRichieste - in lancio..");
      Environment env = actionPlace.cheEnv ();
      Thread meMedesimo = new Thread (env.threadGroup, this, "SportelloRichieste");

      statoSportello = ON;
      meMedesimo.setDaemon (true);
      meMedesimo.start();

    }
    else
      throw new Exception ("SportelloRichieste is already ON");
  } //start

      /**
       *  Chiude lo sportello. Non accetter pi nessun nuovo messaggio fintantoch
       *  non viene eseguito nuovamente lo "start()" dello sportello.
       *  Nota bene: NON viene distrutto questo oggetto, ma solo interrotto il servizio.
       *  Metodo richiesto dall'interfaccia {@link SOMA.network.connection.Daemon}.
       */
  public void stop () throws Exception {
    if (statoSportello == ON) {

      statoSportello = OFF;  // NB: side-effect sullo stato che si ripercuote nel metodo "run()".
      serSock.close();

      // NB: prima lo pongo a OFF, poi chiudo la socket. 
      // Non viceversa senn ripete molte volte il ciclo "while" di attesa su socket.
    }
  } //stop
  
      /**
       *  Rende lo stato della finestra (ON o OFF).
       *  Lo stato  un oggetto, come richiesto in {@link SOMA.network.connection.Daemon}
       *  e lo stato pu essere {@link SOMA.network.connection.Daemon#ON} o {@link SOMA.network.connection.Daemon#OFF}
       *  Metodo richiesto dall'interfaccia {@link SOMA.network.connection.Daemon}.
       */
  public Object getStatus () {
    return statoSportello;
  } //getStatus

      /**
       *  Mostra l'oggetto in una stringa.
       */
  public String toString() {
   return "["+SportelloRichieste.class.toString()+", status=" + statoSportello + "]";
  } //toString


      /**
       *  Definisce il numero di porta TCP sulla quale lo Sportello accetta le
       *  richieste di connessione.
       *  Va chiamata sempre PRIMA di eseguire lo "start()" o per lo meno dopo
       *  che si  fatto uno "stop().
       */
  public void defPorta (int nuovaPorta) {
    if (statoSportello == OFF)
      portaTCP = nuovaPorta;
  } //defPorta

      /**
       *  Rende il numero di porta TCP sulla quale lo Sportello accetta le
       *  richieste di connessione.
       */
  public int chePorta () {
    return portaTCP;
  } //chePorta

      /**
       *  Codice che si esegue dopo uno "start()".
       *  Crea la socket lato server e si mette in attesa di messaggi (comandi).
       *  Per ogni comando arrivato genera un Thread separato che lo esegue e rende
       *  la risposta all'applet che ha inviato il primo messaggio.
       */
  public void run() {

    try {
      Debug.outln ("SportelloRichieste - creando socket-stream lato server");
      serSock = new ServerSocket (portaTCP, 20);  // Crea una nuova socket

      while (statoSportello == ON) {    // Cicla finche' non si vuole finire (terminazione "dolce")

        try {
          Debug.outln ("SportelloRichieste - sono in attesa di messaggi dalle applet remote..");
          Socket cliSock = serSock.accept();            // Connessione: crea nuova socket

          // crea un processo di gestione per la nuova connessione (gli passa la nuova socket),
          Debug.outln ("SportelloRichieste - innescando un thread AccettatorePlace");
          AcceptPlace nuovoAccettatore = new AcceptPlace (cliSock, actionPlace);
            
          // lo innesca e torna e a accettare connessioni
          nuovoAccettatore.start();

        }
        catch (SocketException errClosed) { 
          Debug.outln ("SportelloRichieste - la Socket  stata chiusa!"); 
        }
        catch (IOException errBoh) {
          Debug.outln ("SportelloRichieste - ERRORE: "+errBoh);
        }

      } // while - uscira' solo quando viene imposto lo stato a "OFF"

      try {
        Debug.outln ("SportelloRichieste - sto chiudendo la socket");
        serSock.close();                          // Chiude la socket (non genera eccezione)
      } 
      catch (Exception X) { }

      Debug.outln ("SportelloRichieste - Chiuso!");
    }
    catch (IOException err) { 
      Debug.outln ("SportelloRichieste - ERRORE: "+err);
      statoSportello = OFF;
    }

  } //run

} //SportelloRichieste