/**  DelegatePlace
 *     Oggetto che mette a disposizione le funzioni usabili dall'applet remota.
 *     A ogni funzionalit corrisponde un oggetto "MessXXX" che viene inviato al sito su
 *     sui risiede il Place (e su cui  in attesa un thread "SportelloRichieste", che generer
 *     un thread "AcceptPlace").
 *     NB: ho usato ancora una volta il modello MVC, ma un po' mascherato; infatti non  questo
 *     oggetto che crea la sua finestra di interfaccia grafica, ma il contrario. Ci  dovuto al
 *     fatto che sul lato CLIENT non si sta creando un'applicazione ma un APPLET, e sono le
 *     applet che vengono create per prima - e ereditano dalla classe "Panel"!
 *     PS: la discendenza  Container-Panel-Applet-JApplet.
 *     @author     Luigi Antenucci
 *     @version    1.6
 *     @language   jdk 1.2.2
 */

package SOMA.gui.remoteapplet;

import SOMA.gui.ActionPlaceInterface;
import SOMA.gui.remotegui.*;
import SOMA.naming.PlaceID;

import java.net.Socket;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Vector;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.DefaultMutableTreeNode;


public class DelegatePlace implements ActionPlaceInterface {

      /**
       *  Porta TCP di default su cui lo "SportelloRichieste" del Place  in attesa di messaggi.
       *  Poich si presuppone un place per ogni computer, basta usare questa; ma NON si
       *   obbligati perch il costruttore pu avere il parametro "porta".
       */
  public static final  int PORTA_DEFAULT = 5555;

      /**
       *  RIFERIMENTO ALLA PROPRIA "AppletPlace" (passata al costruttore).
       */
  protected AppletPlace appletPlace;

      /**
       *  Porta TCP su cui  in attesa lo "SportelloRichieste" del Place remoto.
       */
  protected int portaPlace;

      /**
       *  Indirizzo IP dell'host remoto (su cui  stato creato il Place che si vuole contattare).
       */
  protected InetAddress indirPlace;

  
      /**
       *  Costruzione del "delegatore".
       *  Si usa la porta di default ma nessun indirizzo IP specifico, 
       *  perci si dovr usare la "defIndirPlace".
       */
  public DelegatePlace (AppletPlace appletPlace) {
    this (appletPlace, null, PORTA_DEFAULT);
  } //costruttore

      /**
       *  Costruzione del "delegatore".
       *  Si user l'indirizzo IP passato e la porta di default.
       */
  public DelegatePlace (AppletPlace appletPlace, InetAddress indirPlace) {
    this (appletPlace, indirPlace, PORTA_DEFAULT);
  } //costruttore

      /**
       *  Costruzione del "delegatore".
       *  Si user l'indirizzo IP e la porta passati.
       */
  public DelegatePlace (AppletPlace appletPlace, InetAddress indirPlace, int portaPlace) {

    this.appletPlace = appletPlace;
    this.indirPlace  = indirPlace;
    this.portaPlace  = portaPlace;

  } //costruttore

      /**
       *  Rende la porta TCP usata dallo "SportelloRichieste" del Place sul computer remoto.
       */
  public int chePortaPlace () {
    return portaPlace;
  } //chePortaPlace

      /**
       *  Imposta la porta TCP (a cui inviare messaggi) al valore passato.
       *  Il valore di default  nella costante "PORTA_DEFAULT".
       */
  public void defPortaPlace (int nuovaPorta) {
    portaPlace = nuovaPorta;
  } //defPortaPlace

      /**
       *  Rende l'indirizzo IP del computer remoto su cui risiede lo "SportelloRichieste" del Place.
       */
  public InetAddress cheIndirPlace () {
    return indirPlace;
  } //cheIndirPlace

      /**
       *  Imposta l'indirizzo IP (del nodo a cui inviare i messaggi) al valore passato.
       */
  public void defIndirPlace (InetAddress nuovoIndirizzo) {
    indirPlace = nuovoIndirizzo;
  } //defIndirPlace

      /**
       *  Come la precedente, ma riceve una stringa in ingresso.
       *  Pu generare eccezione se non si riesce a convertire la stringa in un indirizzo IP valido.
       */
  public void defIndirPlace (String nuovoIndStr) throws Exception {
    indirPlace = InetAddress.getByName (nuovoIndStr);  // Pu generare eccezione
  } //defIndirPlace

      /**
       *  Interna, stabilisce la connessione (socket-stream) con lo "SportelloRichieste" del Place.
       *  Ha un SIDE EFFECT su portaPlace e indirPlace.
       *  Pu generare un'eccezione in caso di mancata connessione.
       */
  protected Socket creaConnessione () throws Exception {

    if (indirPlace == null)
      throw new UnknownHostException ();

    System.out.println ("connettendosi allo SportelloPlace ("+indirPlace.getHostAddress()+":"+portaPlace+")");

    Socket miaSock = new Socket (indirPlace, portaPlace);

    miaSock.setSoTimeout (10000);   // ATTENDERO' UNA RISPOSTA CON UN TIME-OUT!
      
    return miaSock;  // Se tutto ok, la rende; altrimenti ha gi generato un'eccezione!

  } //creaConnessione

      /**
       *  Interna, chiude la comunicazione con la socket passata.
       */
  protected void chiudiConnessione (Socket sock) {

    System.out.println ("chiudendo la connessione");

    try {
    	sock.close();
    }
    catch (Exception e) { }
  } //chiudiConnessione

      /**
       *  Interna, invia il messaggi passato (un oggetto "polimorfico" di classe base "Mess") sullo
       *  stream di oggetti passato (in genere  quello generato tramite l'uso della socket resa 
       *  dopo una "creaConnessione").
       *  Pu generare un'eccezione in caso di errore di spedizione (si usa il supporto di TCP).
       */
  protected void inviaMessaggio (ObjectOutputStream ObjOut, Mess mess) throws Exception {

    System.out.println ("inviando il messaggio "+mess.toString());

    ObjOut.writeObject (mess);
    ObjOut.flush();
    
  } //inviaMessaggio

      /**
       *  Interna, riceve un messaggio dallo stream di oggetti passato (in genere  quello generato 
       *  tramite l'uso della socket resa dopo una "creaConnessione").
       *  Pu generare un'eccezione in caso di errore di ricezione (si usa il supporto di TCP).
       *  Nota: sar necessario un "casting" sull'oggetto "Mess" reso.
       */
  protected Mess riceviMessaggio (ObjectInputStream objIn) throws Exception {

    System.out.println ("in attesa di un messaggio di risposta..");

    Mess mess = (Mess) objIn.readObject ();

    System.out.println ("ricevuto il messaggio "+mess.toString());

    return mess;

  } //riceviMessaggio

      /**
       *  Interna, usata per gestire un generico messaggio di "Dammi".
       *  Viene inviato un messaggio "MessDelDammi" inizializzato al "dammiChe" passato; quindi
       *  si attende l'arrivo dell'oggetto di ritorno.
       *  Se non riesce o a stabilire la connessione o non viene reso nulla, questo metodo
       *  render "null".
       *  Nota: sar necessario un "casting" sull'oggetto "Object" reso, in base al "dammiChe".
       */
  protected Object eseguiDammiEPoiRiceviOggetto (int dammiChe) {
    Object oggettoReso;

    try {
      Socket sock = creaConnessione ();  // Pu generare un'eccezione.
      try {
        ObjectOutputStream objOut = new ObjectOutputStream (sock.getOutputStream());
        ObjectInputStream  objIn  = new ObjectInputStream  (sock.getInputStream());

        MessDelDammi mess = new MessDelDammi (dammiChe);
        
        inviaMessaggio (objOut, mess);

        System.out.println ("in attesa dell'oggetto..");

        oggettoReso = objIn.readObject ();   // LEGGO L'OGGETTO (serializzato)

        System.out.println ("ricevuto l'oggetto di classe "+oggettoReso.getClass().getName());

      }
      catch (Exception e) { 
        throw e;
      }
      finally {
        chiudiConnessione (sock);
      }
    }
    catch (Exception e) { 
      oggettoReso = null;
    }

    return oggettoReso;
  } //eseguiDammiEPoiRiceviOggetto

      /**
       *  Lancia un agente (sul place remoto) in base ai dati passati.
       *  Inizia il protocollo per la richiesta di lancio di un agente sul Place remoto.
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public void lanciaAgente (String nomeAgente, String[] arrParam,
                            boolean usaSysClassLoader, boolean rintracciabile,
                            boolean nonFarloPartire) throws Exception {

    Socket sock = creaConnessione ();  // Pu generare un'eccezione.
    ObjectOutputStream objOut = new ObjectOutputStream (sock.getOutputStream());
    ObjectInputStream  objIn  = new ObjectInputStream  (sock.getInputStream());

    MessDelLancia mess = new MessDelLancia (nomeAgente, arrParam,
                                            usaSysClassLoader, rintracciabile,
                                            nonFarloPartire);
    try {
    	
      inviaMessaggio (objOut, mess);

      Mess messRisp = riceviMessaggio (objIn);
      if (messRisp instanceof MessBidErrore)
        throw ((MessBidErrore) messRisp).eccezione;

    }
    catch (Exception e) { 
      throw e;
    }
    finally {
      chiudiConnessione(sock);
    }
  } //lanciaAgente

      /**
       *  Rende il {@link SOMA.naming.PlaceID} del place remoto.
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public PlaceID chePlaceID () {
    Object oggettoReso = eseguiDammiEPoiRiceviOggetto (MessDelDammi.DAMMI_PLACE_ID);
    if (oggettoReso != null)
      return (PlaceID) oggettoReso;
    else
      return new PlaceID ("", "");
  } //chePlaceID

      /**
       *  Rende la lista di tutti i nomi (sottoforma di stringa) degli agenti contenuti 
       *  nel Percorso degli Agenti del place relativo all'ActionPlace.
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector cheElencoAgenti () {
    Object elenco = eseguiDammiEPoiRiceviOggetto (MessDelDammi.DAMMI_AGENTI);
    if (elenco != null)
      return (Vector) elenco;
    else
      return new Vector();
  } //cheElencoAgenti

      /**
       *  Rende la lista di tutti i PlaceID dei place contenuti nel DNS del place remoto
       *  (il quale place deve essere un place di default per poter avere il DNS;
       *  in caso contrario la lista resa sar vuota).
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector elencoDominiiInDNS () {
    Object elenco = eseguiDammiEPoiRiceviOggetto (MessDelDammi.DAMMI_DOMINII);
    if (elenco != null)
      return (Vector) elenco;
    else
      return new Vector();
  } //elencoDominiiInDNS

      /**
       *  Rende la lista di tutti i PlaceID dei place contenuti nel PNS del place remoto.
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector elencoPlaceInPNS () {
    Object elenco = eseguiDammiEPoiRiceviOggetto (MessDelDammi.DAMMI_PLACE);
    if (elenco != null)
      return (Vector) elenco;
    else
      return new Vector();
  } //elencoPlaceInPNS

      /**
       *  Rende un modello di un albero ({@link javax.swing.tree.DefaultTreeModel})
       *  il cui albero  costituito da nodi di classe {@link javax.swing.tree.DefaultMutableTreeNode}.
       *  Tale albero contiene tutti i thread attivi all'interno del Place di riferimento.
       *  Nota: ogni nodo contiene una STRINGA (e non un oggetto Thread!) che rappresenta la
       *  DESCRIZIONE del reltivo thread nel threadgroup.
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public DefaultTreeModel alberoPlaceThread () {
    Object albero = eseguiDammiEPoiRiceviOggetto (MessDelDammi.DAMMI_ALB_THREAD);
    if (albero != null)
      return (DefaultTreeModel) albero;
    else
      return new DefaultTreeModel(new DefaultMutableTreeNode(".."));
  } //alberoPlaceThread

      /**
       *  Rende una lista contenente una descrizione di tutti gli agenti
       *  che sono al momento in esecuzione.
       *  Ogni elemento della lista  a sua volta una lista, cio una terna.
       *    (AgentID, NomeClasse, Stato)
       *  Utile per il costruttore di JTable().
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector elencoAgentiInEsecuzione () {
    Object elenco = eseguiDammiEPoiRiceviOggetto (MessDelDammi.DAMMI_AGENT_EXE);
    if (elenco != null)
      return (Vector) elenco;
    else
      return new Vector();
  } //elencoAgentiInEsecuzione

      /**
       *  Rende una lista contenente la posizione di tutti gli agenti che sono
       *  stati creati nel Place.
       *  Ogni elemento della lista  a sua volta una lista, cio una coppia:
       *    (AgentID, PlaceID)
       *  Utile per il costruttore di JTable().
       *  Metodo richiesto dall'interfaccia "ActionPlaceInterface".
       */
  public Vector elencoPosizioneAgenti () {
      Object elenco = eseguiDammiEPoiRiceviOggetto (MessDelDammi.DAMMI_AGENT_POS);
    if (elenco != null)
      return (Vector) elenco;
    else
      return new Vector();
  } //elencoPosizioneAgenti

      /**
       *  Controlla che il Place a cui si fa riferimento sia ancora attivo (cio il suo oggetto
       *  "SportelloRichieste" sia in stato di ON e possa quindi rispondere ai messaggi futuri).
       *  Se non riesce a stabilire la connessione o il Place non risponde, si genera un'eccezione.
       *  Se invece va tutto bene, il metodo ritorner normalmente.
       */
  public void avvio_AreYouAlive () throws Exception {

    Socket sock = creaConnessione ();  // Pu generare un'eccezione.
    ObjectOutputStream objOut = new ObjectOutputStream (sock.getOutputStream());
    ObjectInputStream  objIn  = new ObjectInputStream  (sock.getInputStream());

    MessDelAYA messAYA = new MessDelAYA();

    try {
      
      inviaMessaggio (objOut, messAYA);

      Mess messIAA = riceviMessaggio (objIn);  // Attende ricezione di un I-AM-ALIVE.
      if (! (messIAA instanceof MessAccIAA))
        throw new Exception ("TIME-OUT");
    }
    catch (Exception e) { 
      throw e;
    }
    finally {
      chiudiConnessione(sock);
    }
  } //avvio_AreYouAlive

} //DelegatePlace