/**  Lingua
 *     Classe "statica" per rendere un programma indipendente dalla lingua in
 *     cui  originalmente scritto.
 *     Questa classe permette a ogni altra classe (che costituisce il programma) di
 *     ottenere le frasi nella lingua corretta.
 *     "Lingua" caricher tali frasi vengono da un opportuno file e le render (una
 *     alla volta) alle varie classi quando queste lo richiedono.
 *     Il funzionamento base prevede che nella classe principale (all'inizio del
 *     programma) si stabilisca una lingua invocando:
 *         <code>  Lingua.defLingua ("Italiano");  </code>
 *       o <code>  Lingua.defLingua ("English");  </code>
 *     Quindi IN OGNI CLASSE (che andr a scrivere frasi all'utente) bisogna
 *     caricare il file di lingua opportuno con:
 *         <code>  Lingua.caricaFileDiLingua (NOMECLASSE.class);  </code>
 *     PER OGNUNA DI TALI CLASSI OCCORRE PREVEDERE UN "FILE DI LINGUA" PER OGNI
 *     POSSIBILE LINGUA! Tale file ha LO STESSO NOME DELLA CLASSE ma con estensione
 *     ".it" (per la lingua italiana) o ".eng" (per quella inglese) o qualsiasi
 *     altra estensione che pu essere aggiunta nel FILE "Lingua.lang"!
 *     Tali file sono dei normalissimi FILE DI TESTO ma che hanno una ben precisa strutturazione.
 *     OGNI LINEA deve contenere UNA SOLA FRASE preceduta da un IDENTIFICATORE.
 *     L'IDENTIFICATORE serve a "Lingua" (e alle classi che la usano) per ritrovare le
 *     singole frasi, e dev'essere una sequenza di caratteri e numeri SENZA SPAZI.
 *     La prima parola alla sinistra di ogni linea  l'IDENTIFICATORE.
 *     Tutto il resto della linea  la frase.
 *     ATTENZIONE: poich gli spazi all'inizio e fine della frase possono essere importanti,
 *      obbligatorio che la frase sia delimitata dal carattere di APICE oppure di DOPPIO APICE
 *     (a scelta,  indifferente usare uno o l'altro).
 *     Quando una classe ha bisogno di una frase andr a farne richiesta a "Lingua":
 *       <code>  Lingua.cheFrase("IDENTIFICATORE"); </code>
 *     Come puoi notare, dando l'IDENTIFICATORE della frase si ottiene la FRASE NELLA "LINGUA ATTUALE".
 *     N.B: per ottimizzare gli accessi NON SI FA NESSUNA SEPARAZIONE TRA GLI IDENTIFICATORI delle varie
 *     classi.  Ogni classe pu "vedere" gli identificatori di ogni altra, ma non  "bello" che si sfrutti
 *     tale caratteristica.  Perci occorre stare attenti a non definire lo stesso identificatore in due
 *     file di classe differenti perch uno "sovrascriver" l'altro.
 *     Io ho pensato di far precedere ogni identificatore dalle iniziali di ogni classe (per es, per la
 *     classe "Inizio" gli identificatori inizieranno con "I_" mentre per "ProvaLingua" inizieranno
 *     con "PL_", e cos via..).
 *     Successivamente occorre PREVEDERE UN CAMBIO DI LINGUA "DINAMICO".
 *     Per prima cosa ogni classe deve implementare l'interfaccia {@link SOMA.gui.lingua.LinguaListener}
 *     e quindi si deve "registrare" per ascoltare gli eventi di "CAMBIO DI LINGUA"
 *     tramite una chimata a:  <code> Lingua.addLinguaListener(this); </code>.
 *     Quindi nel metodo "linguaCambiata()" occorre scrivere tutte le modifiche da fare
 *     alle frasi gi stampate e non modificabili (per esempio, per tutte le "JLabel" presenti
 *     al momento nella finestra occorrer fare una "setText").
 *     @see SOMA.gui.lingua.LinguaListener
 *     @see SOMA.gui.lingua.LinguaGui
 *     @author     Luigi Antenucci
 *     @version    5.55
 *     @language   jdk 1.2.2
 */

package SOMA.gui.lingua;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import java.util.StringTokenizer;
import java.io.*;
import javax.swing.event.EventListenerList;


public class Lingua {

  // Costanti (pubbliche):
      /**
       *  Per default si usa la prima lingua definita in "Lingua.lang"
       */
  public static final int LINGUA_DEFAULT = 0;

  public static final String LINGUA_PACKAGE     = "SOMA"+File.separator+"gui"+File.separator+"lingua"+File.separator;
  public static final String LINGUA_PACKAGE_IMG = "SOMA"+File.separator+"gui"+File.separator+"lingua"+File.separator+"picture"+File.separator;

  // Costanti (protette):
      /**
       *  Caratteri con sui devono iniziare e finire tutte le frasi nei file di lingua.
       */
  protected static final char CAR_APICE1 = '\'';
  protected static final char CAR_APICE2 = '"';

      /**
       *  File contenente l'elenco delle lingue (usato per SOVRASCRIVERE le impostazioni di default)
       *  Se NON ESISTE tale file, si useranno le impostazioni definite tra poco.
       *  Il file "Lingua.lang" DEVE CONTENERE IN OGNI LINEA UNA DEFINIZIONE DI UNA LINGUA scritta
       *  usando delle frasi - separate da spazi - che hanno il seguente significato:
       *  EstensioneDelFile   NomeLingua    IconaMenu   ImmagineBandiera
       *  Per esempio:
       *  .eng   English   IcoEng.gif  FlagEng.gif
       */
  protected static final String FILE_LINGUE = LINGUA_PACKAGE+"Lingua.lang";

  // Variabili di classe (protette):
      /**
       *  Estensioni di DEFAULT dei file si lingua (con anche il punto iniziale)
       *  Se NON ESISTE IL FILE "Lingua.lang" si useranno queste impostazioni!
       */
  protected static String[] FILE_EXT   = {".eng", ".it"};

      /**
       *  Nome delle lingue (ognuna va scritta nella propria lingua di origine!)
       */
  protected static String[] STR_LINGUA = {"English", "Italiano"};

      /**
       *  Immagini delle bandiere di ogni lingua
       */
  protected static String[] FILE_ICO   = {"IcoEng.gif",  "IcoIt.gif"};
  protected static String[] FILE_IMG   = {"FlagEng.gif", "FlagIt.gif"};

      /**
       *  Memorizza la lingua correntemente in uso (la prima, in genere quella inglese)
       */
  protected static int linguaAttuale = LINGUA_DEFAULT;

      /**
       *  Tabella contenente tutte le coppie (identificatore, stringa)
       *  tramite l'ID si ottiene la stringa (nella lingua attuale)
       *  Nel file ogni riga contiene: << Identificatore Frase.. >>
       */
  protected static Hashtable tabIdentFrasi = new Hashtable();

      /**
       *  Lista di tutti i file di lingua caricati
       * (necessario per il cambio dinamico della lingua)
       */
  protected static Vector fileCaricati = new Vector();

      /**
       *  Lista contenente tutti gli oggetti che si sono "registrati"
       *  per ascoltare un cambiamento di lingua!
       *  NON mi serve un "LinguaEvent". Se mi servisse scriverei:
       *  protected static LinguaEvent linguaEvent = null;
       */
  protected static EventListenerList listenerList = new EventListenerList();

      /**
       *  Questo mi dice se non si  ancora cercato un'alternativa al default
       */
  protected static boolean primaVolta = true;


  // NON C'E` IL COSTRUTTORE: ho un oggetto STATICO!


      /**
       *  Metodo INTERNO; carica l'elenco delle lingue (da fare solo la PRIMA volta)
       *  Se il file non c', si usa il DEFAULT interno (inglese e italiano)
       */
  protected static void CaricaElencoLingue() {
    if (primaVolta) {
      try {
        String[] TmpExt = new String [30];
        String[] TmpNom = new String [30];
        String[] TmpIco = new String [30];
        String[] TmpImg = new String [30];
        FileReader FIS = new FileReader (FILE_LINGUE);   // Cerca di aprire il file
        BufferedReader BR = new BufferedReader (FIS);
        String Linea;
        int i = 0;
        while (BR.ready()) {               // LEGGE DAL FILE LE LINEE: << Estensione NomeLingua >>
          Linea = BR.readLine();           // Leggo una linea dal file
          // DEBUG: System.out.println("Linea="+Linea);
          StringTokenizer STok = new StringTokenizer(Linea);
          try {
            TmpExt[i] = STok.nextToken();      // LEGGO PRIMO TOKEN: ESTENSIONE FILE
            TmpNom[i] = STok.nextToken();      // LEGGO SECONDO TOKEN: NOME LINGUA
            TmpIco[i] = STok.nextToken();      // LEGGO TERZO TOKEN: ICONA BANDIERA
            TmpImg[i] = STok.nextToken();      // LEGGO TERZO TOKEN: BANGIERA GRANDE
            // DEBUG: System.out.println("Token:  ["+TmpExt[i]+"]  ["+TmpNom[i]+"]  ["+TmpIco[i]+"]");
            i++;  // SOLO ADESSO POSSO PASSARE ALLA NUOVA CELLA!
          }
          catch (Exception E) {  // Errore se un "nextElement" NON C'E'
            System.err.println ("Lingua: ignoring Line '"+Linea+"' in File "+FILE_LINGUE);
          }
        } //while
        i--;  // ora "i" punta all'ULTIMA CELLA (o -1 se non ne ho creata nessuna!)
        if (i >= 0) {
          // Definisco cio' che ho letto!
          FILE_EXT   = new String [i+1];    // Li creo giusti giusti
          STR_LINGUA = new String [i+1];
          FILE_ICO   = new String [i+1];
          FILE_IMG   = new String [i+1];
          while (i >= 0) {                  // E ci assegno i valori degli array temporanei
            FILE_EXT[i]   = TmpExt[i];
            STR_LINGUA[i] = TmpNom[i];
            FILE_ICO[i]   = TmpIco[i];
            FILE_IMG[i]   = TmpImg[i];
            i--;
          }
        } //if
        // SE NON HA LETTO NULLA, NON VA A MODIFICARE IL DEFAULT!
      }
      catch (FileNotFoundException E) { System.err.println (E.toString()); }
      catch (IOException E)           { System.err.println (E.toString()); }

      primaVolta = false;   // NON LA RIFACCIO PIU'
    } //if
  } //CaricaElencoLingue

      /**
       *  Rende l'elenco di lingue conosciute (non necessariamente supportate dal programma)
       *  L'elenco  definito nel file "Lingua.lang" (vedi costante FILE_LINGUE)
       */
  public static String[] cheElencoLingue () {
    if (primaVolta)
      CaricaElencoLingue();
    String[] Tmp = new String [STR_LINGUA.length];
    for (int i=0; i < STR_LINGUA.length; i++)
      Tmp[i] = new String (STR_LINGUA[i]);   // NON VOGLIO L'ALIASING: I NOMI SONO IMMODIFICABILI!
    return Tmp;
  } //cheElencoLingue

      /**
       *  Rende l'elenco delle estensioni dei file di lingua
       *  L'elenco  definito nel file "Lingua.lang" (vedi costante FILE_LINGUE)
       */
  public static String[] cheExtLingue () {
    if (primaVolta)
      CaricaElencoLingue();
    String[] Tmp = new String [FILE_EXT.length];
    for (int i=0; i < FILE_EXT.length; i++)
      Tmp[i] = new String (FILE_EXT[i]);   // NON VOGLIO L'ALIASING: I NOMI SONO IMMODIFICABILI!
    return Tmp;
  } //cheExtLingue

      /**
       *  Rende l'elenco dei file di icona per ogni lingua
       *  L'elenco  definito nel file "Lingua.lang" (vedi costante FILE_LINGUE)
       */
  public static String[] cheIcoLingue () {
    if (primaVolta)
      CaricaElencoLingue();
    String[] Tmp = new String [FILE_ICO.length];
    for (int i=0; i < FILE_ICO.length; i++)
      Tmp[i] = new String (FILE_ICO[i]);   // NON VOGLIO L'ALIASING: I NOMI SONO IMMODIFICABILI!
    return Tmp;
  } //cheIcoLingue

      /**
       *  Rende l'elenco dei file con le bandiere (grandi) per ogni lingua
       *  L'elenco  definito nel file "Lingua.lang" (vedi costante FILE_LINGUE)
       */
  public static String[] cheImgLingue () {
    if (primaVolta)
      CaricaElencoLingue();
    String[] Tmp = new String [FILE_IMG.length];
    for (int i=0; i < FILE_IMG.length; i++)
      Tmp[i] = new String (FILE_IMG[i]);   // NON VOGLIO L'ALIASING: I NOMI SONO IMMODIFICABILI!
    return Tmp;
  } //cheImgLingue

      /**
       *  Rende l'intero (da usare in "defLingua") che rappresena la lingua passata.
       *  Nota: rende -1 se la lingua non  supportata.
       */
  public static int cheLingua (String NomeLingua) {
    if (primaVolta)
      CaricaElencoLingue();
    int i=0;
    boolean cont = true;
    while ((i < STR_LINGUA.length) && cont) {
      if (STR_LINGUA[i].equals(NomeLingua))
        cont = false;
      else
        i++;
    } //while
    if (! cont)
      return i;    // l'ho trovata!
    else
      return -1;   // NON l'ho trovata
  } //cheLingua

      /**
       *  Rende l'intero corrispondente alla lingua attualmente usata
       */
  public static int cheLinguaAttuale () {
    return linguaAttuale;
  } //cheLinguaAttuale

      /**
       *  DEFINISCE DINAMICAMENTE UNA NUOVA LINGUA.
       *  La stringa passata  il nome logico di una lingua (vedi il file Lingua.lang)
       *  Dopo averlo convertito con "cheLingua()" invoca la "defLingua (int cheLingua)".
       */
  public static void defLingua (String NomeLingua) {
    defLingua (cheLingua(NomeLingua));
  } //defLingua

      /**
       *  DEFINISCE DINAMICAMENTE UNA NUOVA LINGUA.
       *  Se tale lingua  quella attualmente in uso, non viene fatto nulla.
       *  Altrimenti vengono automaticamente caricati tutti i file di lingua
       *  precedentemente caricati da una "caricaFileDiLingua", ma stavolta
       *  li si carica nella NUOVA lingua (con una differente estensione del file).
       *  Dopo il ricaricamento viene generato un evento di "cambio-lingua" che
       *  verr ascoltato da chiunque si sia "registrato" come "listener" di questa classe.
       */
  public static void defLingua (int cheLingua) {
    if (primaVolta)
      CaricaElencoLingue();
    if ((cheLingua >= 0) && (cheLingua < STR_LINGUA.length)) {
      if (cheLingua != linguaAttuale) {
        // Prima di definire la nuova lingua, devo RICARICARE
        // TUTTI i file di lingua precedentemente caricati.
        // Ovviamente, andr a caricare ognuno nella NUOVA lingua.
        Enumeration enumFile = fileCaricati.elements();
        while (enumFile.hasMoreElements())
          caricaFileDiLingua ((String)enumFile.nextElement(), cheLingua);
                             // nota: uso la versione "protetta" del metodo.

        // STABILISCO LA LINGUA ATTUALE
        linguaAttuale = cheLingua;

        // NOTIFICO L'AVVENUTO CAMBIAMENTO A TUTTI I "LinguaListener" REGISTRATI
        // ("sparo" un "linguaCambiata" a tutti i listener
        fireLinguaCambiata ();

      } //if
      // DEBUG: else  System.err.println ("WARNING: Attempting to Define the SAME Language");
    } //if
    else
      System.err.println ("Lingua: WARNING! Language Not Supported");
  } //defLingua

      /**
       *  Carica il file di lingua (in base alla lingua correntemente definita)
       *  relativo alla CLASSE DELL'OGGETTO passato.
       *  Il nome del file sar uguale a quello della classe dell'oggetto
       *  passato nel parametro THIS.
       *  Uso:  <code>  Lingua.caricaFileDiLingua (this);  </code>
       */
  public static void caricaFileDiLingua (Object THIS) {
    Class cheClasse = THIS.getClass();
    String NomeFile = cheClasse.getName();
    // File.separatorChar  il carattere '\' o '/' (dipende dalla piattaforma)
    NomeFile = NomeFile.replace ('.', File.separatorChar);
    caricaFileDiLingua (NomeFile, linguaAttuale);
  } //caricaFileDiLingua

      /**
       *  Carica il file di lingua (in base alla lingua correntemente definita)
       *  relativo alla CLASSE passata.
       *  Il nome del file sar uguale a quello della classe passata.
       *  Uso:  <code>  Lingua.caricaFileDiLingua (NOMECLASSE.class);  </code>
       */
  public static void caricaFileDiLingua (Class cheClasse) {
    String NomeFile = cheClasse.getName();
    // System.getProperty ("File.separatorChar");
    // File.separatorChar  il carattere '\' o '/' (dipende dalla piattaforma)
    NomeFile = NomeFile.replace ('.', File.separatorChar);
    caricaFileDiLingua (NomeFile, linguaAttuale);
  } //caricaFileDiLingua

      /**
       *  Carica il file di lingua (in base alla lingua correntemente definita)
       *  relativo alla CLASSE il cui nome  passato sottoforma di stringa.
       *  Questo permette a una classe di PRE-CARICARE file di lingua di altre classi.
       *  Uso:  <code>  Lingua.caricaFileDiLingua ("NOMECLASSE");  </code>
       */
  public static void caricaFileDiLingua (String NomeFileSenzaExt) {
    caricaFileDiLingua (NomeFileSenzaExt, linguaAttuale);
  } //caricaFileDiLingua

      /**
       *  Metodo INTERNO per caricare il file di lingua relativo alla CLASSE il
       *  cui nome  passato in "NomeFile" (sottoforma di stringa) nella lingua
       *  passata in "cheLingua".
       *  Questo metodo  invocato da tutte le "caricaFileDiLingua" a un solo argomento.
       *  Se il file di lingua  GIA` STATO CARICATO (nella stessa lingua), NON verr
       *  ricaricato.
       *  Se il file di lingua NON VIENE TROVATO, si prover a caricare la versione
       *  in INGLESE di tale file (che in genere  sempre presente); se neppure tale
       *  file viene trovato, viene abortito il programma segnalando l'errore!
       *  Nota: questo metodo invoca la "caricaFileDiLingua" successiva dandogli un
       *  "alias" alla tabella interna di "Lingua".
       */
  protected static void caricaFileDiLingua (String NomeFile, int cheLingua) {
    // INTERNA!  Richiamata dalle precedenti e da "defLingua"
    // "NomeFile" NON deve avere estensione!
    // Per prima cosa controlla che il file non sia GIA' STATO CARICATO
    // (a meno che linguaAttuale<>cheLingua, in tal caso lo ricarica!)
    if (primaVolta)
      CaricaElencoLingue();
    if ((cheLingua < 0) || (cheLingua >= STR_LINGUA.length)) {         // CONTROLLO LINGUA VALIDA
      System.err.println ("Lingua: LANGUAGE NOT SUPPORTED (lingua="+cheLingua+")");
      System.err.println ("        !!! PROGRAM ABORTED !!!");
      System.exit(2);                      // ABORT DEL PROGRAMMA!
    }
    // DEBUG: System.out.println("NomeFile="+NomeFile+" Lingua="+STR_LINGUA[cheLingua]);
    if (linguaAttuale == cheLingua) {
      if (fileCaricati.contains (NomeFile)) {
        // DEBUG: System.out.println(NomeFile+"  gi stato caricato, non lo ricarico");
        return;  // non fa nulla se gi caricato il file nella stessa lingua
      }
    } //if

    caricaFileDiLingua (NomeFile, cheLingua, tabIdentFrasi);   // CHIAMATA AL METODO SUCCESSIVO (con la tabella interna!)

    // SALVA il NomeFile (SENZA estensione) nel vettore fileCaricati
    if (linguaAttuale == cheLingua)    // NON LO FACCIO SE LA LINGUA E' DIVERSA, ALTRIMENTI VADO A
      fileCaricati.add (NomeFile);     // MODIFICARE "fileCaricati" MENTRE LO STA USANDO "defLingua"
  } //caricaFileDiLingua

      /**
       *  Metodo INTERNO per caricare il file di lingua relativo alla CLASSE il
       *  cui nome  passato in "NomeFile" (sottoforma di stringa) nella lingua
       *  passata in "cheLingua".
       *  Questo metodo COMPLETA la "caricaFileDiLingua" a due parametri e carica
       *  il file di lingua passato nell'hash-table passata (in genere viene passato
       *  un aliasing alla tabella interna delle frasi di "Lingua": tabIdentFrasi)
       *  Il file di lingua viene qui caricato nella tabella IN OGNI CASO, anche se
       *  tutte le coppie (ID,frase) sono in essa gi presenti.
       *  Se il file di lingua NON VIENE TROVATO, si prover a caricare la versione
       *  in INGLESE di tale file (che in genere  sempre presente); se neppure tale
       *  file viene trovato, viene abortito il programma segnalando l'errore!
       *  NB: l'aborto viene forzato tramite un "System.exit(1);", perci non c' modo
       *  di impedirlo. Lo si pu per prevenire andando a controllare se esiste il
       *  file di lingua che si vuole caricare; basta invocare la esisteFileDiLingua().
       */
  protected static void caricaFileDiLingua (String NomeFile, int cheLingua, Hashtable tabIdentFrasiPassata) {
    // APRE IL FILE DI LINGUA
    FileReader FIS = null;
    // Aggiunge al NomeFile l'ESTENSIONE in base a cheLingua.
    //vbvbvbvbvbvbvbvbvb
    String FileConExt = NomeFile + FILE_EXT[cheLingua];
    //String FileConExt = NomeFile + FILE_EXT[cheLingua];

    // DEBUG: System.out.println("Aprendo il File "+FileConExt);
    try {
      FIS = new FileReader (FileConExt);   // Cerca di aprire il file
    }
    catch (IOException Err) {     // Il file NON ESISTE
      System.err.println ("Lingua: ERROR! File Not Found: "+FileConExt);
      System.err.println ("        USING THE ENGLISH VERSION");
      // RIPROVA, ma apre il file in lingua INGLESE
      FileConExt = NomeFile + FILE_EXT[LINGUA_DEFAULT];
      try {
        FIS = new FileReader (FileConExt);   // Cerca di aprire il file
      }
      catch (IOException ErrBis) {           // Nemmeno questo esiste, sigh!
        System.err.println ("        ERROR AGAIN! File Not Found: "+FileConExt);
        System.err.println ("        CAN'T FIND ANY LANGUAGE FILE FOR THE CLASS "+NomeFile);
        System.out.println ("        !!! PROGRAM ABORTED !!!");
        System.exit(1);                      // ABORT DEL PROGRAMMA!
      } //catch
    } //catch

    BufferedReader BR = new BufferedReader (FIS);
    String Linea;
    String Identif, Frase;
    int Pos;
    // LEGGE DAL FILE DI LINGUA LE LINEE: << Identificatore Frase.. >>
    // E INSERISCE IN "tabIdentFrasi" LA COPPIA (Identificatore,Frase)
    // DEBUG: System.out.println("Leggendo..");
    try {
      while (BR.ready()) {
        Linea = BR.readLine();               // Leggo una linea dal file

        if (Linea != null)
          if (Linea.length() > 3) {

            // DEBUG: System.out.println ("     Linea  = ["+Linea+"]");
            Pos = Linea.indexOf (' ');
            if (Pos == -1) {               // Non c' nemmeno uno spazio
              System.err.println ("Lingua: WARNING! the identifier "+Linea+" in file "+FileConExt);
              System.err.println ("                 doesn't have any Sentences!");
            }
            else {
              Identif = Linea.substring (0, Pos);
              Frase = Linea.substring (Pos+1);   // Da Pos+1 fino alla fine (salto il car.SPAZIO)
              // Pero' cosi' ho spazi inutili e ancora gli APICI: li devo togliere:
              Frase = Frase.trim();
              char PrimoCar = Frase.charAt(0);
              char UltimoCar = Frase.charAt(Frase.length()-1);

              if ( ((PrimoCar == CAR_APICE1) ||
                    (PrimoCar == CAR_APICE2)   ) &&
                   (PrimoCar  == UltimoCar)        ) {
                Frase = Frase.substring(1, Frase.length()-1);
              }
              else {
                System.err.println ("Lingua: WARNING! the identifier "+Identif+" in file "+FileConExt);
                System.err.println ("                 has a Wrong Sentence!");
              }

              // INSERISCE LA COPPIA (Identif,Frase) IN TABELLA
              // DEBUG: System.out.println ("     Identif= ["+Identif+"]");
              // DEBUG: System.out.println ("     Frase  = ["+Frase+"]");
              Object Occupato = tabIdentFrasiPassata.put (Identif, Frase);  // NB: sulla tabella passata!
              if ((Occupato != null) && (linguaAttuale == cheLingua)) {
                System.err.println ("Lingua: WARNING! the identifier "+Identif+" in file "+FileConExt);
                System.err.println ("                 overwrites an already existent identifier!");
                System.err.println ("                 Please, change the name of this identifier!");
              } //if
            } //else
          } //if
      } //while
      FIS.close();  // chiude il file letto
    }
    catch (IOException E) {  }
  } //caricaFileDiLingua

      /**
       *  Rende 'true' se esiste il file di lingua relativo all'oggetto passato.
       */
  public static boolean esisteFileDiLingua (Object THIS) {
    Class cheClasse = THIS.getClass();
    String NomeFile = cheClasse.getName();
    NomeFile = NomeFile.replace ('.', File.separatorChar);
    return esisteFileDiLingua (NomeFile);
  } //esisteFileDiLingua

      /**
       *  Rende 'true' se esiste il file di lingua relativo alla classe passata.
       */
  public static boolean esisteFileDiLingua (Class cheClasse) {
    String NomeFile = cheClasse.getName();
    NomeFile = NomeFile.replace ('.', File.separatorChar);
    return esisteFileDiLingua (NomeFile);
  } //esisteFileDiLingua

      /**
       *  Rende 'true' se esiste il file di lingua relativo alla classe contenuta nella stringa passata.
       *  Nota: viene cercato il file di lingua relativo alla lingua correntemente impostata; in caso NON
       *  venisse trovato viene cercato il file per la lingua inglese. Viene reso 'false' se neppure
       *  questo file viene trovato.
       */
  public static boolean esisteFileDiLingua (String NomeFileSenzaExt) {
    FileReader FIS;
    try {
      // Provo a aprire il file (nella lingua attuale)
      FIS = new FileReader (NomeFileSenzaExt + FILE_EXT[linguaAttuale]);
    }
    catch (IOException E) {
      try {
        // Prova a aprire il nuovo file, in lingua inglese
        FIS = new FileReader (NomeFileSenzaExt + FILE_EXT[LINGUA_DEFAULT]);
      }
      catch (IOException E2) {
        return false;  // NESSUN FILE DI LINGUA ESISTE!
      } //catch
    } //catch

    // Se sono qui, il file  stato aperto correttamente!
    try {
      FIS.close();
    }
    catch (Exception e) { }
    return true;
  } //esisteFileDiLingua

      /**
       *  Rende la frase (nella lingua corrente) accoppiata all'Identificatore.
       *  E` la funzione base della "conversione".
       *  La stringa resa sar quella associata all'identificatore (passato)
       *  che  stata precedentemente caricata dal file di lingua opportuno
       *  relativo alla classe.
       */
  public static String frase (String Identificatore) {
    String cheFrase = (String) tabIdentFrasi.get(Identificatore);
    if (cheFrase != null) return cheFrase;
                    else  return "<<"+Identificatore+">>";   // se =null,  un errore
  } //frase

      /**
       *  Metodo utile per il "trasporto" del proprio (e solo) file di lingua.
       *  Funziona come le "caricaFileDiLingua" ma le coppie "(ID,frase)" NON vengono caricate
       *  in memoria (nella tabella interna a "Lingua") ma su una tabella (hash-table) che viene
       *  creata dal metodo e resa.
       *  Tale tabella pu essere caricata nella tabella interna di "Lingua" tramite l'invocazione
       *  del metodo "caricaFileDiLinguaDaTabella".
       *  L'utilit  che la tabella  "serializzabile" e pu quindi facilmente essere trasferita da
       *  un computer a un altro (tramite socket).
       *  Come per la "caricaFileDiLingua" (a un parametro) anche questo metodo ha tre "varianti",
       *  questa  la prima e ha come parametro l'oggetto di cui caricare il file di lingua.
       */
  public static Hashtable caricaFileDiLinguaSuTabella (Object THIS) {
    Class cheClasse = THIS.getClass();
    String NomeFile = cheClasse.getName();
    NomeFile = NomeFile.replace ('.', File.separatorChar);

    Hashtable tabIdFrasiDaRendere = new Hashtable();

    // NB: NON CHIAMO QUELLA A DUE PARAMETRI PERCHE' VOGLIO CHE IL CARICAMENTO AVVENGA IN OGNI CASO!
    caricaFileDiLingua (NomeFile, linguaAttuale, tabIdFrasiDaRendere);

    return tabIdFrasiDaRendere;
  } //caricaFileDiLinguaSuTabella

      /**
       *  Seconda variante: si passa la CLASSE di cui si vuol caricare il file di lingua.
       */
  public static Hashtable caricaFileDiLinguaSuTabella (Class cheClasse) {
    String NomeFile = cheClasse.getName();
    NomeFile = NomeFile.replace ('.', File.separatorChar);

    Hashtable tabIdFrasiDaRendere = new Hashtable();

    // NB: NON CHIAMO QUELLA A DUE PARAMETRI PERCHE' VOGLIO CHE IL CARICAMENTO AVVENGA IN OGNI CASO!
    caricaFileDiLingua (NomeFile, linguaAttuale, tabIdFrasiDaRendere);

    return tabIdFrasiDaRendere;
  } //caricaFileDiLinguaSuTabella

      /**
       *  Ultima variante: si passa la STRINGA contenente il nome del file (senza estensione) di
       *  lingua da caricare.
       */
  public static Hashtable caricaFileDiLinguaSuTabella (String NomeFileSenzaExt) {
    Hashtable tabIdFrasiDaRendere = new Hashtable();

    // NB: NON CHIAMO QUELLA A DUE PARAMETRI PERCHE' VOGLIO CHE IL CARICAMENTO AVVENGA IN OGNI CASO!
    caricaFileDiLingua (NomeFileSenzaExt, linguaAttuale, tabIdFrasiDaRendere);

    return tabIdFrasiDaRendere;
  } //caricaFileDiLinguaSuTabella

      /**
       *  Questo metodo "completa" la funzionalit dei "caricaFileDiLinguaSuTabella": permette di
       *  caricare nella tabella interna di "Lingua" tutte le coppie "(ID,frase)" che sono
       *  contenute nella tabella (hash-table) passata.
       *  Tale tabella deve essere stata creata tramite la "caricaFileDiLinguaSuTabella".
       */
  public static void caricaFileDiLinguaDaTabella (Hashtable tabella) {
    String ident;
    Enumeration enum = tabella.keys ();           // Scorre le chiavi (gli identificatori)!
    while (enum.hasMoreElements()) {              // e le inserisce nella lista
      ident = (String) enum.nextElement();
      tabIdentFrasi.put (ident, tabella.get(ident));   // Passo da tabella a tabella-interna!
    }
  } //caricaFileDiLinguaDaTabella

      /**
       *  Ci si registra come LISTENER dell'oggetto Lingua.
       *  Ogni listener deve implementare l'interfaccia "LinguaListener"
       *  e sara' avvisato quando avviene un cambio di lingua.
       *  Anche se l'accostamento delle parole inglesi "add" e "Listener" con
       *  l'italiano "Lingua" non  molto grazioso, l'ho fatto per mantenere
       *  la "compatibilit" di sintassi definita dal modello a eventi di Java.
       */
  public static void addLinguaListener (LinguaListener cheListener) {
    listenerList.add (LinguaListener.class, cheListener);
  } //addLinguaListener

      /**
       *  Funzione inversa di "addLinguaListener"; serve per annullare
       *  una precedente registrazione di un Listener.
       */
  public static void removeLinguaListener (LinguaListener cheListener) {
    listenerList.remove (LinguaListener.class, cheListener);
  } //removeLinguaListener

      /**
       *  Metodo INTERNO, usato per notificare a tutti i listener (che si sono
       * registrati con "addLinguaListener") che  avvenuto un evento di "cambio di lingua".
       */
  protected static void fireLinguaCambiata () {
    // Prelevo l'elenco dei listener e li metto in un array
    Object[] listeners = listenerList.getListenerList();

    // NON MI SERVE CREARE UN EVENTO DA DARE con "linguaCambiata(linguaEvent)"
    // Se mi servisse farei:
    //    if (linguaEvent == null)
    //      linguaEvent = new LinguaEvent(this);

    // Processo i listener in lista dall'ULTIMO al PRIMO e
    // gli notifico un cambio di lingua!
    int i = listeners.length-2;
    while (i >= 0) {
      if (listeners[i] == LinguaListener.class)
        ((LinguaListener)listeners[i+1]).linguaCambiata();
      i=i-2;
    }
  } //fireLinguaCambiata

} //Lingua