/**  PanPlace
 *     Pannello per l'input (tramite combo-box) del nome di un place tramite il
 *     suo identificativo di dominio e di place di un dominio.
 *     Ne ho fatto una classe apposta perch questo "particolare" tipo di input
 *     viene richiesto in pi di una finestra. La modularit innanzitutto, no?.
 *     @author     Luigi Antenucci
 *     @version    1.0
 *     @language   jdk 1.2.2
 */

package SOMA.gui;

import SOMA.gui.lingua.Lingua;
import SOMA.mobilePlace.MobilePlaceID;
import SOMA.naming.PlaceID;
import java.util.Vector;
import java.util.Enumeration;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;


public class PanPlace extends Pannello {

  // COSTANTI
      /**
       *  Costanti per il costruttore.
       */
  public static final int INPUT_DOMINIO = 0;
  public static final int INPUT_PLACE   = 1;

  // Variabili PROTETTE:
      /**
       *  Oggetti che mostrano frasi, coi che dipendono dalla lingua.
       */
  protected JLabel    labDom,   labPla;
  protected JComboBox comboDom, comboPla;

      /**
       *  Oggetti passati al costruttore del pannello.
       */
  protected int         inputChe;
  protected ActionPlaceInterface actionPlace;
  protected Anagrafe    anagrafe;
  protected PlaceID     placeID;   // Place iniziale

     /**
       *  Lista contenente tutti gli oggetti che si sono "registrati"
       *  per ascoltare un cambiamento di contenuto del pannello!
       */
  protected EventListenerList listenerList = new EventListenerList();

      /**
       *  Costruisco un pannello place non inizializzato (tutti i campi saranno vuoti).
       *  Il parametro "inputChe" pu essere una delle costanti INPUT_DOMINIO o INPUT_PLACE
       *  e indica il tipo di input che si vuole chiedere (un place di default o uno normale).
       *  I dati nei "combo-box" sono ottenuti attraverso richieste all'oggetto "Anagrafe" passato.
       */
  public PanPlace (int inputChe, Anagrafe anagrafe) {
    this (inputChe, anagrafe, null, null);
  } //costruttore

      /**
       *  Costruttore analogo al precedente, ma permette di stabilire il Place Iniziale.
       */
  public PanPlace (int inputChe, Anagrafe anagrafe, PlaceID placeID) {
    this (inputChe, anagrafe, null, placeID);
  } //costruttore

      /**
       *  Costruisco un pannello place i cui campi sono inizializzati coi dati del place relativo
       *  all'oggetto ActionPlace passato e a esso verranno richiesti i dati con cui riempire i
       *  combo-box.
       *  Il parametro "inputChe" pu essere una delle costanti INPUT_DOMINIO o INPUT_PLACE
       *  e indica il tipo di input che si vuole chiedere (un place di default o uno normale).
       *  NB: se il Place  un Place di Default, il combo-box "dominio"  riempito con tutti i dominii 
       *  contenuti nel DNS; mentre se  un Place normale, l'unico dominio conosciuto  quello a cui
       *  il Place stesso fa riferimento.
       *  In ogni caso, se l'utente imposter il combo-box del dominio sul dominio del Place corrente,
       *  il secondo combo-box (quello del Place) viene riempito con tutti i Place contenuti nel PNS,
       *  altrimenti verr riempito col solo place di default.
       */
  public PanPlace (int inputChe, ActionPlaceInterface actionPlace) {
    this (inputChe, null, actionPlace, actionPlace.chePlaceID());
  } //costruttore
  
      /*
       *  Costruttore interno! Richiamato da tutti gli altri.
       */
  protected PanPlace (int inputChe, Anagrafe anagrafe, 
                      ActionPlaceInterface actionPlace, PlaceID placeID) {
    super ();

    this.inputChe    = inputChe;
    this.anagrafe    = anagrafe;
    this.actionPlace = actionPlace;
    this.placeID     = placeID;

    costruisciPannello ();
  } //costruttore

      /**
       *  Costruzione del pannello
       */
  public void costruisciPannello () {
    labDom   = new JLabel ();

    // CARICA nei COMBO BOX TUTTI I DOMINII (nel 1) E I PLACE RELATIVI AL DOMINIO (nel 2)
    if (anagrafe != null) {
      comboDom = new JComboBox (anagrafe.listaPlaceDefault());
    }
    else {  // usa actionPlace
      if (placeID.isDomain() ||
          (placeID instanceof MobilePlaceID))               // chiedo all'ActionPlace che chiede al DNS!
        comboDom = new JComboBox (actionPlace.elencoDominiiInDNS());
      else {
        comboDom = new JComboBox ();         // Se NON ho il DNS, metto il solo dominio di appartenenza
        comboDom.addItem (placeID.getDomainID());
      }
      // NO: comboDom.addItem (placeID.getDomainID());  // anche il dominio di se stesso.
    }

    if (inputChe == INPUT_PLACE) {
      labPla   = new JLabel ();
      comboPla = new JComboBox ();
    }

    if (placeID != null) {
      PlaceID domID = placeID.getDomainID();
      comboDom.setSelectedItem (domID);    // Imposta combo box su PLACE PASSATO!
      if (inputChe == INPUT_PLACE) {
        RiempiComboPlace (domID);
        comboPla.setSelectedItem (placeID);
      }
    }

    if (inputChe == INPUT_PLACE)
      comboDom.addActionListener(new ActionListener() {
                                    public void actionPerformed (ActionEvent e) {
                                      PlaceID domSelez = (PlaceID) comboDom.getSelectedItem();
                                      RiempiComboPlace (domSelez);
                                    } //actionPerformed
                                 });

    // SE SI CAMBIA IL CONTENUTO DI UNO DEI COMBO SI LANCERA' UN PROPERTY-CHANGE EVENT
    ActionListener cambioCombo = new ActionListener () {
                                    public void actionPerformed (ActionEvent e) {
                                      // NOTIFICO L'AVVENUTO CAMBIAMENTO A TUTTI GLI "AnagrafeListener" REGISTRATI
                                      // ("sparo" un "anagrafeCambiata" a tutti i listener
                                      firePanPlaceCambiato ();
                                    } //actionPerformed
                                 }; //ActionListener
    if (inputChe == INPUT_DOMINIO)
      comboDom.addActionListener(cambioCombo);
    else
      comboPla.addActionListener(cambioCombo);   
      // NB: MODIFICARE IL COMBO DEL DOMINIO MODIFICA QUELLO DEL PLACE IL QUALE LANCERA` L'EVENTO'

    GridBagLayout GBL     = new GridBagLayout();
    GridBagConstraints CC = new GridBagConstraints();
    setLayout (GBL);

    CC.gridx = 1;   CC.gridy = 1;
    CC.anchor = GridBagConstraints.EAST;
    GBL.setConstraints (labDom, CC);
    add (labDom);

    CC.gridx = 2;   CC.gridy = 1;
    CC.anchor = GridBagConstraints.WEST;
    GBL.setConstraints (comboDom, CC);
    add (comboDom);

    if (inputChe == INPUT_PLACE) {
      CC.gridx = 1;   CC.gridy = 2;
      CC.anchor = GridBagConstraints.EAST;
      GBL.setConstraints (labPla, CC);
      add (labPla);
    
      CC.gridx = 2;   CC.gridy = 2;
      CC.anchor = GridBagConstraints.WEST;
      GBL.setConstraints (comboPla, CC);
      add (comboPla);
    }

    impostaFrasi ();

    FineCostruisciPannello();
  } //costruisciPannello

      /**
       *  Definisce/modifica tutte le frasi mostrate a video (nella GUI).
       */
  protected void impostaFrasi () {
    labDom.setText (Lingua.frase("PP_DOMAIN"));
    if (inputChe == INPUT_PLACE)
      labPla.setText (Lingua.frase("PP_PLACE"));

    comboDom.setToolTipText (Lingua.frase("PP_TIP_DOMAIN"));
    if (inputChe == INPUT_PLACE)
      comboPla.setToolTipText (Lingua.frase("PP_TIP_PLACE"));

    FineImpostaFrasi();
  } //impostaFrasi

      /**
       *  Disabilita gli elementi contenuti nel pannello.
       */
  public void disabilitami () {
    labDom  .setEnabled(false);
    comboDom.setEnabled(false);
    if (inputChe == INPUT_PLACE) {
      labPla  .setEnabled(false);
      comboPla.setEnabled(false);
    }
  } //disabilitami

      /**
       *  Riabilitazione gli elementi contenuti nel pannello (dopo una "disabilitami").
       */
  public void abilitami () {
    labDom  .setEnabled(true);
    comboDom.setEnabled(true);
    if (inputChe == INPUT_PLACE) {
      labPla  .setEnabled(true);
      comboPla.setEnabled(true);
    }
  } //abilitami

      /**
       *  Ci si registra come LISTENER dell'oggetto PanPlace.
       *  Ogni listener deve implementare l'interfaccia "PanPlaceListener" 
       *  e sara' avvisato quando avviene una modifica del contenuto del pannello.
       */
  public void addPanPlaceListener (PanPlaceListener cheListener) {
    listenerList.add (PanPlaceListener.class, cheListener);
  } //addPanPlaceListener

      /**
       *  Funzione inversa di "addPanPlaceListener"; serve per annullare
       *  una precedente registrazione di un Listener.
       */
  public void removePanPlaceListener (PanPlaceListener cheListener) {
    listenerList.remove (PanPlaceListener.class, cheListener);
  } //removePanPlaceListener

      /**
       *  Metodo INTERNO, usato per notificare a tutti i listener (che si sono 
       * registrati con "addPanPlaceListener") che  avvenuto un evento di cambiamento del pannello.
       */
  protected void firePanPlaceCambiato () {

    // Prelevo l'elenco dei listener e li metto in un array
    Object[] listeners = listenerList.getListenerList();

    // Processo i listener in lista dall'ULTIMO al PRIMO e gli notifico il cambiamento!
    int i = listeners.length-2;
    while (i >= 0) {
      if (listeners[i] == PanPlaceListener.class)
        ((PanPlaceListener)listeners[i+1]).panPlaceCambiato();
      i=i-2;
    }
  } //firePanPlaceCambiato

      /**
       *  Riempie il "combobox" relativo al "place nel dominio" con l'elenco di
       *  tutti i place contenuti nel dominio passato come parametro.
       */
  protected void RiempiComboPlace (PlaceID inCheDominio) {

    Vector elenco;
    if (anagrafe != null) {

      elenco = anagrafe.listaPlaceInDominio (inCheDominio);

    }
    else {  // uso l'actionPlace
      if (inCheDominio.sameDomain(placeID)) {   // Se sono nello stesso dominio, leggo dal PNS

        elenco = actionPlace.elencoPlaceInPNS ();

      }
      else {                                   // Altrimenti, metto il solo dominio di default
        
        elenco = new Vector();
        elenco.addElement (inCheDominio);

      }
    }

    try {
      comboPla.removeAllItems();
      // BACO DI SWING SOTTO WIN95: genera eccezione se il combo-box  vuoto!
      // NB: sotto Win NT funziona tutto correttamente (no eccezione)
    }
    catch (Exception e) { }

    Enumeration enum = elenco.elements();
    while (enum.hasMoreElements())
      comboPla.addItem ((PlaceID) enum.nextElement());

    comboPla.revalidate();
  } //RiempiComboPlace

      /**
       *  Rende il {@link SOMA.naming.PlaceID} del dominio contenuto nel "combobox" del "dominio"
       *  Non  il PlaceID della coppia "dominio+place" ma solo quello relativo alla parte "dominio".
       */
  public PlaceID cheDominio () {
    return (PlaceID) comboDom.getSelectedItem();
  } //cheDominio

      /**
       *  Imposta il "combobox" del dominio al dominio indicato dal {@link SOMA.naming.PlaceID} passato.
       *  Se si sta inserendo un place, verr impostato il "combobox" del place sul place di default.
       */
  public void defDominio (PlaceID placeID) {
    PlaceID domID = placeID.getDomainID();
    comboDom.setSelectedItem (domID);
    if (inputChe == INPUT_PLACE)
      RiempiComboPlace (domID);
  } //defDominio

      /**
       *  Rende il {@link SOMA.naming.PlaceID} del place contenuto nella coppia di "combobox".
       *  Viene reso il PlaceID della coppia "dominio+place".
       *  Se si  stabilito "INPUT_DOMINIO" al costruttore, l'unico PlaceID che pu rendere questo
       *  metodo  quello relativo al place di default del solo dominio selezionato sul combo-box.
       */
  public PlaceID chePlace () {
    if (inputChe == INPUT_PLACE) 
      return (PlaceID) comboPla.getSelectedItem();
    else
      return (PlaceID) comboDom.getSelectedItem();
  } //chePlace

      /**
       *  Imposta il "combobox" del place-nel-dominio al place indicato dal {@link SOMA.naming.PlaceID} passato.
       *  Se si  stabilito "INPUT_DOMINIO" al costruttore, l'unico PlaceID che pu essere impostato 
       *  quello del place di default.
       */
  public void defPlace (PlaceID placeID) {
    if (inputChe == INPUT_PLACE) {
      comboPla.setSelectedItem (placeID);
      comboPla.revalidate();
    }
    else {
      PlaceID domID = placeID.getDomainID();
      comboDom.setSelectedItem (domID);
      comboDom.revalidate();
    }
  } //defPlace

} //PanPlace