/**  PanLocRem
 *     Pannello per input (con radio-button) di "nessuno/locale/remoto)".
 *     Ne ho fatto una classe apposta perch questo "particolare" tipo di input
 *     viene richiesto in pi di una finestra. La modularit innanzitutto, no?.
 *     @author     Luigi Antenucci
 *     @version    3.3
 *     @language   jdk 1.2.2
 */

package SOMA.gui;

import SOMA.gui.lingua.Lingua;
import SOMA.naming.PlaceID;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;


public class PanLocRem extends Pannello {

  // Costanti pubbliche:
      /**
       *  Costanti da passare al costruttore per indicare il tipo di pannello da creare.
       */
  public static final  int  PAN_NO_LOC_REM  = 0;  // No-locale-remoto per inserire un dominio
  public static final  int  PAN_LOC_REM_DOM = 1;  // Locale-remoto per inserire un dominio
  public static final  int  PAN_LOC_REM_ALL = 2;  // Locale-remoto per inserire un place (tra tutti)

  // Variabili PROTETTE:
      /**
       *  Oggetti che mostrano frasi, coi che dipendono dalla lingua.
       */
  protected JRadioButton radioNessun, radioLocale, radioRemoto;
  protected PanPlace     panPlaceLocale;
  protected JLabel       labRadioRemDom, labRadioRemIP, labRadioRemPo;
  protected JTextField   txtRemDom, txtRemIP, txtRemPort;

      /**
       *  Oggetti passati al costruttore del pannello.
       */
  protected int      panNoLocRem;
  protected Anagrafe anagrafe;
  protected PlaceID  dominioIniziale;


      /**
       *  Costruisce un pannello "locale-remoto".
       *  Il parametro "panNoLocRem" pu contenere "PAN_NO_LOC_REM" per avere un 
       *  pannello con le DUE sole scelte "nessuno/remoto" oppure "PAN_LOC_REM_DOM"
       *  o "PAN_LOC_REM_DOM" per il "remoto" soltanto (inserire un qualsiasi place, e non solo dominio).
       *  Con questo costruttore, il combo-box "locale" sar SEMPRE DISABILITATO!
       */
  public PanLocRem (int panNoLocRem) {
    this (panNoLocRem, null, null);
  } //costruttore

      /**
       *  Costruisce un pannello "locale-remoto".
       *  Il parametro "panNoLocRem" pu contenere "PAN_NO_LOC_REM" per avere un 
       *  pannello con le tre scelte "nessuno/locale/remoto" oppure "PAN_LOC_REM_DOM"
       *  per avere le due sole scelte "locale/remoto" o "PAN_LOC_REM_DOM" per inserire
       *  un qualsiasi place (e non solo dominio).
       *  Se l'oggetto "Anagrafe" NON  "null", si abiliter il combo-box "locale" che
       *  conterr i dati dei place richiesti all'Anagrafe stessa.
       */
  public PanLocRem (int panNoLocRem, Anagrafe anagrafe) {
    this (panNoLocRem, anagrafe, null);
  } //costruttore

      /**
       *  Costruisce un pannello "locale-remoto".
       *  Simile al costruttore precedente; in pi, il parametro "dominioIniziale"
       *  permette di impostare gi un dominio iniziale nella sottovoce "locale".
       */
  public PanLocRem (int panNoLocRem, Anagrafe anagrafe, PlaceID dominioIniziale) {
    super ();

    this.panNoLocRem     = panNoLocRem;
    this.anagrafe        = anagrafe;
    this.dominioIniziale = dominioIniziale;

    costruisciPannello ();
  } //costruttore

      /**
       *  Costruzione del pannello
       */
  public void costruisciPannello () {

    radioNessun  = new JRadioButton ();
    radioLocale  = new JRadioButton ();  // Lo creo, ma posso NON USARLO (non lo stampo a video)
    radioRemoto  = new JRadioButton ();

    labRadioRemDom = new JLabel ();
    labRadioRemIP  = new JLabel ();
    labRadioRemPo  = new JLabel ();

    txtRemDom   = new JTextField (30);
    txtRemIP    = new JTextField (16);
    txtRemPort  = new JTextField (6);

    if (anagrafe != null) {
      // Carica nel pannello di scelta "locale" tutti i dominii gi definiti localmente
      // Inoltre imposta il "combo" sul dominio passato in "dominioIniziale".
      int cheInputLocale;
      if (panNoLocRem == PAN_LOC_REM_ALL) 
        cheInputLocale = PanPlace.INPUT_PLACE;     // Locale per input "place"
      else
        cheInputLocale = PanPlace.INPUT_DOMINIO;   // oppure per input "default place"

      if (dominioIniziale == null)
        panPlaceLocale = new PanPlace (cheInputLocale, anagrafe);
      else
        panPlaceLocale = new PanPlace (cheInputLocale, anagrafe, dominioIniziale.getDomainID());
    }

    ButtonGroup gruppo = new ButtonGroup();    // Creo un gruppo di bottoni
    if (panNoLocRem == PAN_NO_LOC_REM)
      gruppo.add (radioNessun);        // Non lo inserisco nel gruppo
    if (anagrafe != null)
      gruppo.add (radioLocale);
    gruppo.add (radioRemoto);

    impostaFrasi ();

    GridBagLayout GBL     = new GridBagLayout();
    GridBagConstraints CC = new GridBagConstraints();
    setLayout (GBL);

    if (panNoLocRem == PAN_NO_LOC_REM) {
      CC.gridx = 1;   CC.gridy = 1;
      CC.anchor = GridBagConstraints.WEST;
      GBL.setConstraints (radioNessun, CC);
      add (radioNessun);                     // Non lo stampo a video
    }

    if (anagrafe != null) {
      CC.gridx = 1;   CC.gridy = 5;
      CC.anchor = GridBagConstraints.WEST;
      GBL.setConstraints (radioLocale, CC);
      add (radioLocale);

        JPanel panLocale = new JPanel();  // 2 righe e 1 colonna
          panLocale.setBorder (BorderFactory.createEmptyBorder(0, 60, 10, 0));   // alto,sx,basso,dx
          panLocale.add (panPlaceLocale);
        CC.gridx = 1;   CC.gridy = 6;
        CC.anchor = GridBagConstraints.WEST;
        GBL.setConstraints (panLocale, CC);
        add (panLocale);
   }

    CC.gridx = 1;   CC.gridy = 10;
    CC.anchor = GridBagConstraints.WEST;
    GBL.setConstraints (radioRemoto, CC);
    add (radioRemoto);

    Component spazio;

      JPanel panRemoto = new JPanel();  // 2 righe e 1 colonna
        panRemoto.setBorder (BorderFactory.createEmptyBorder(0, 45, 8, 0));   // alto,sx,basso,dx
        GridBagLayout GBL3     = new GridBagLayout();
        GridBagConstraints CC3 = new GridBagConstraints();
        panRemoto.setLayout (GBL3);

        CC3.anchor = GridBagConstraints.EAST;
        if (panNoLocRem != PAN_NO_LOC_REM) {
          CC3.gridx = 1;   CC3.gridy = 1;
          GBL3.setConstraints (labRadioRemDom, CC3);
          panRemoto.add (labRadioRemDom);
        }

        CC3.gridx = 1;   CC3.gridy = 3;
        GBL3.setConstraints (labRadioRemIP, CC3);
        panRemoto.add (labRadioRemIP);

        CC3.gridx = 1;   CC3.gridy = 5;
        GBL3.setConstraints (labRadioRemPo, CC3);
        panRemoto.add (labRadioRemPo);

        CC3.anchor = GridBagConstraints.WEST;
        if (panNoLocRem != PAN_NO_LOC_REM) {
          CC3.gridx = 2;   CC3.gridy = 1;
          GBL3.setConstraints (txtRemDom, CC3);
          panRemoto.add (txtRemDom);

          spazio = Box.createRigidArea (new Dimension(4,4));
          CC3.gridx = 2;   CC3.gridy = 2;
          GBL3.setConstraints (spazio, CC3);
          panRemoto.add (spazio);
        }

        CC3.gridx = 2;   CC3.gridy = 3;
        GBL3.setConstraints (txtRemIP, CC3);
        panRemoto.add (txtRemIP);

        spazio = Box.createRigidArea (new Dimension(4,4));
        CC3.gridx = 2;   CC3.gridy = 4;
        GBL3.setConstraints (spazio, CC3);
        panRemoto.add (spazio);

        CC3.gridx = 2;   CC3.gridy = 5;
        GBL3.setConstraints (txtRemPort, CC3);
        panRemoto.add (txtRemPort);
      
      CC.gridx = 1;   CC.gridy = 11;
      CC.anchor = GridBagConstraints.WEST;
      GBL.setConstraints (panRemoto, CC);
      add (panRemoto);

    // AZIONI: actionListener
    ListRadio radiera = new ListRadio (this);
    if (panNoLocRem == PAN_NO_LOC_REM) {
      radioNessun.addActionListener (radiera);
      radioNessun.setActionCommand ("nessuno");
    }
    if (anagrafe != null) {
      radioLocale.addActionListener (radiera);
      radioLocale.setActionCommand ("locale");
    }
    radioRemoto.addActionListener (radiera);
    radioRemoto.setActionCommand ("remoto");

    if ((anagrafe != null) && (dominioIniziale != null)) {
      radioLocale.setSelected(true);
      eseguiAzione (radioLocale.getActionCommand());
    }
    else
      if (panNoLocRem == PAN_NO_LOC_REM) {
        radioNessun.setSelected(true);                  // Chi  selezionato all'inizio?
        eseguiAzione (radioNessun.getActionCommand());
      }
      else
        if (dominioIniziale == null) {
          radioRemoto.setSelected(true);
          eseguiAzione (radioRemoto.getActionCommand());
        }

    FineCostruisciPannello();
  } //costruisciPannello

      /**
       *  Permette la disabilitazione degli elementi contenuti nel pannello.
       */
  public void disabilitami () {
    panPlaceLocale.disabilitami ();
    radioNessun.setEnabled(false);
    radioLocale.setEnabled(false);
    radioRemoto.setEnabled(false);
    txtRemDom .setEditable(false);
    txtRemIP  .setEditable(false);
    txtRemPort.setEditable(false);
  } //disabilitami

      /**
       *  Permette la riabilitazione degli elementi contenuti nel pannello (dopo una "disabilitami").
       */
  public void abilitami () {
    panPlaceLocale.abilitami ();
    radioNessun.setEnabled(true);
    radioLocale.setEnabled(true);
    radioRemoto.setEnabled(true);
    txtRemDom .setEditable(true);
    txtRemIP  .setEditable(true);
    txtRemPort.setEditable(true);
    radioRemoto.setSelected(true);
    eseguiAzione (radioRemoto.getActionCommand());
  } //abilitami

      /**
       *  Definisce/modifica tutte le frasi mostrate a video (nella GUI).
       */
  protected void impostaFrasi () {
    radioNessun.setText (Lingua.frase("PLR_NONE"));
    if (anagrafe != null)
      radioLocale.setText (Lingua.frase("PLR_LOCALE"));
    radioRemoto.setText (Lingua.frase("PLR_REMOTE"));
    if (panNoLocRem == PAN_LOC_REM_ALL) 
      labRadioRemDom.setText (Lingua.frase("PLR_REM_ALL"));
    else
      labRadioRemDom.setText (Lingua.frase("PLR_REM_DOM"));

    labRadioRemIP.setText (Lingua.frase("PLR_REM_IP"));
    labRadioRemPo.setText (Lingua.frase("PLR_REM_PORT"));

    radioNessun.setToolTipText (Lingua.frase("PLR_TIP_R_NO"));
    if (anagrafe != null)
      radioLocale.setToolTipText (Lingua.frase("PLR_TIP_R_LOC"));
    radioRemoto.setToolTipText (Lingua.frase("PLR_TIP_R_REM"));
    if (panNoLocRem == PAN_LOC_REM_ALL) 
      txtRemDom.setToolTipText (Lingua.frase("PLR_TIP_REMALL"));
    else
      txtRemDom.setToolTipText (Lingua.frase("PLR_TIP_REMDOM"));
    txtRemIP   .setToolTipText (Lingua.frase("PLR_TIP_REMIP"));
    txtRemPort .setToolTipText (Lingua.frase("PLR_TIP_REMPRT"));

    FineImpostaFrasi();
  } //impostaFrasi

      /**
       *  Metodo interno.
       *  Una volta che l'utente ha impostato un "radio" su una delle possibili
       *  voci (nessuno/locale/remoto) questo metodo permette di abilitare tutti
       *  i "campi" (text-field e altro) che sono "associati" alla voce scelta e
       *  contemporaneamente di disabilitare tutti quelli delle voci non selezionate,
       *  in modo che l'utente non possa andare a scriversi dentro niente e che,
       *  essendo disabilitati, appaiano in un colore differente da quelli abilitati.
       */
  protected void eseguiAzione (String actionCommand) {
    if (actionCommand.equals("nessuno")) {
      if (anagrafe != null)
        panPlaceLocale.disabilitami();

      txtRemDom .setEditable(false);
      txtRemIP  .setEditable(false);
      txtRemPort.setEditable(false);
      labRadioRemDom.setEnabled(false);
      labRadioRemIP .setEnabled(false);
      labRadioRemPo .setEnabled(false);
    }
    else
      if (actionCommand.equals("locale")) {
        if (anagrafe != null)
          panPlaceLocale.abilitami();  //NB: riabilito!

        txtRemDom .setEditable(false);
        txtRemIP  .setEditable(false);
        txtRemPort.setEditable(false);
        labRadioRemDom.setEnabled(false);
        labRadioRemIP .setEnabled(false);
        labRadioRemPo .setEnabled(false);
      }
      else
        if (actionCommand.equals("remoto")) {
          if (anagrafe != null)
            panPlaceLocale.disabilitami();

          txtRemDom .setEditable(true);   //NB: riabilito!
          txtRemIP  .setEditable(true);
          txtRemPort.setEditable(true);
          labRadioRemDom.setEnabled(true);
          labRadioRemIP .setEnabled(true);
          labRadioRemPo .setEnabled(true);
        }
    } //eseguiAzione

      /**
       *  CLASSE INTERNA PER ASCOLTARE LA PRESSIONE DEI BOTTONI "RADIO" DELLA FINESTRA.
       *  Permette di disabilitare tutti i "campi" di input che sono relativi 
       *  ai "radio" non abilitati.
       */
  class ListRadio implements ActionListener {
    protected PanLocRem mioPanLocRem;

    public ListRadio (PanLocRem mioPanLocRem) {
      this.mioPanLocRem = mioPanLocRem;
    } //costruttore

    public void actionPerformed (ActionEvent e) {
      String azione = e.getActionCommand();
      mioPanLocRem.eseguiAzione (azione);
    } //actionPerformed
  } // ListRadio

      /**
       *  Rende il nome del dominio prelevandolo dai campi relativi alla voce selezionata
       */
  public String cheNomeDom () {
    if (radioLocale.isSelected()) {      // Selezionato "locale"
      PlaceID placeID = panPlaceLocale.cheDominio();
      return placeID.toString();
    }
    else
      if (radioRemoto.isSelected())     // Selezionato "remoto"
        return txtRemDom.getText();
      else
        return "";                      // Selezionato "nessuno"
  } //cheNomeDom

      /**
       *  Rende il nome del place prelevandolo dai campi relativi alla voce selezionata
       *  Un place pu essere sempre il place di default del dominio.
       *  Questo metodo render un place NON di default soltanto se si era indicato "PAN_LOC_REM_ALL"
       *  al costruttore! In tal caso alla voce "locale" pu essere scelto un place tra tutti quelli
       *  creati localmente (di default o meno) o specificare una stringa (dominio+place) nel "remoto".
       */
  public String cheNomePlace () {
    if (radioLocale.isSelected()) {      // Selezionato "locale"
      PlaceID placeID = panPlaceLocale.chePlace();
      return placeID.toString();
    }
    else
      if (radioRemoto.isSelected())     // Selezionato "remoto"
        return txtRemDom.getText();
      else
        return "";                      // Selezionato "nessuno"
  } //cheNomePlace

      /**
       *  Rende l'indirizzo IP del place/dominio prelevandolo dai campi relativi 
       *  alla voce selezionata
       */
  public String cheIndirIP () {
    if (radioLocale.isSelected()) {     // Selezionato "locale"
      PlaceID placeID = panPlaceLocale.chePlace();
      return anagrafe.cheHost(placeID).getHostAddress();
    }
    else
      if (radioRemoto.isSelected())     // Selezionato "remoto"
        return txtRemIP.getText();
      else
        return "";                      // Selezionato "nessuno"
  } //cheIndirIP

      /**
       *  Rende il numero di porta del dominio/place prelevandolo dai campi 
       *  relativi alla voce selezionata
       */
  public String chePorta () {
    if (radioLocale.isSelected()) {     // Selezionato "locale"
      PlaceID placeID = panPlaceLocale.chePlace();
      return String.valueOf (anagrafe.chePorta(placeID));
    }
    else
      if (radioRemoto.isSelected())     // Selezionato "remoto"
        return txtRemPort.getText();
      else
        return "";                      // Selezionato "nessuno"
  } //chePorta

      /**
       *  Usate per la pressione di INVIO sulle aree di testo
       */
  public void addActionListener (ActionListener AL) {
    txtRemDom .addActionListener (AL);
    txtRemIP  .addActionListener (AL);
    txtRemPort.addActionListener (AL);
  } //addActionListener

  public void removeActionListener (ActionListener AL) {
    txtRemDom .removeActionListener (AL);
    txtRemIP  .removeActionListener (AL);
    txtRemPort.removeActionListener (AL);
  } //removeActionListener

/*
  public static void main (String[] args) {
    // PER DEBUGGING
    JFrame F = new JFrame ();
    F.addWindowListener(new java.awt.event.WindowAdapter() {
                        public void windowClosing(java.awt.event.WindowEvent e) { 
                          System.exit(0); 
                        } });
    Container contentPane = F.getContentPane();
    Anagrafe anagrafe = new Anagrafe();
    //contentPane.add (new PanLocRem (PAN_NO_LOC_REM,  anagrafe));
    //contentPane.add (new PanLocRem (PAN_LOC_REM_DOM, anagrafe));
    contentPane.add (new PanLocRem (PAN_LOC_REM_ALL, anagrafe));
    F.pack();
    F.setVisible (true);          // INFINE RENDO VISIBILE LA FINESTRA
  }
*/

} //PanLocRem