/**  ModelloDNS
 *     @author     Luigi Antenucci
 *     @version    1.0
 *     @language   jdk 1.2.2
 */

package SOMA.gui;

import SOMA.gui.lingua.Lingua;
import SOMA.Environment;
import SOMA.naming.PlaceID;
import SOMA.naming.domain.DomainInfo;
import SOMA.naming.domain.DomainNameService;

import javax.swing.table.TableModel;
import javax.swing.table.AbstractTableModel;


public class ModelloDNS implements ModelloXNS {

      /**
       *  L'Environment del place; in questo oggetto  contenuto il riferimento al DNS (vedi dopo).
       */
  protected Environment mioEnv;

      /**
       *  Il Domain Name Service a cui fare riferimento.
       *  Lo "tiro fuori" da "mioEnv" per un accesso diretto.
       */
  protected DomainNameService mioDNS;

      /**
       *  La finestra a cui (eventualmente) fa riferimento questo modello.
       */
  protected FinestraXNS miaFinestraXNS = null;

      /**
       *  Costruisce il Modello per il DNS dell'Environment passato come parametro
       *  e ne apre la finestra.
       *  NB: l'Environment DEVE appartenere a un PLACE DI DEFAULT (i place normali NON hanno il DNS)
       */
  public ModelloDNS (Environment env) {
    this (env, true);
  } //costruttore

      /**
       *  Costruisce il Modello per il DNS dell'Environment passato come parametro
       *  e ne apre la finestra (con tutti i bottoni di modifica) a seconda del 
       *  valore di "apriFin".
       */
  public ModelloDNS (Environment env, boolean apriFin) {
    Lingua.caricaFileDiLingua (ModelloDNS.class);  // Carico file di lingua per questa classe
    mioEnv = env;
    mioDNS = env.domainNameService;
    if (apriFin) {
      miaFinestraXNS = new FinestraXNS (this);
    }
  } //costruttore

      /**
       *  Costruisce il Modello per il DNS dell'Environment passato come parametro
       *  e ne apre la finestra relativa (in ogni caso).
       *  La finestra avr o meno i bottoni di modifica a seconda del valore di 
       *  "finXNSModif" (vedi costanti in {@link SOMA.gui.FinestraXNS}).
       */
  public ModelloDNS (Environment env, int finXNSModif) {
    Lingua.caricaFileDiLingua (ModelloDNS.class);  // Carico file di lingua per questa classe
    mioEnv = env;
    mioDNS = env.domainNameService;
    miaFinestraXNS = new FinestraXNS (this, finXNSModif);
  } //costruttore

      /**
       *  Rende una stringa contenente il tipo di Name-Service a cui il modello fa riferimento
       */
  public String cheTipo () {
    return (Lingua.frase("MD_WIN_TITLE"));
  } //cheTipo

      /**
       *  Rende il PlaceID del place a cui appartiene il Name-Service del modello
       */
  public PlaceID perChePlace () {
    return mioEnv.placeID;
  } //perChePlace

      /**
       *  Rende una tabella che rappresenta la stessa tabella contenuta nel Name-Service.
       *  Il TableModel conterr tutti oggetti di classe DomainInfo.
       *  Vedi {@link SOMA.naming.domain.DomainInfo}
       */
  public TableModel cheTabella () {
    return new TabellaDNS();
  } //cheTabella

      /**
       *  Rende un "Pannello" contenente informazioni aggiuntive sul Name-Service.
       */
  public Pannello pannelloInfoInPiu () {
    return new PanInfoDNS (mioEnv);
  } //pannelloInfoInPiu

      /**
       *  Invocazione del metodo "put" del DNS.
       *  Tutte le informazioni sono contenute nell'oggetto passato, che sar della classe opportuna.
       */
  public void invocaPut (Object informazioni) {
    Debug.outln ("ModelloDNS - INVOCO LA putDomain");
    mioDNS.putDomain ((DomainInfo)informazioni);
  } //invocaPut

      /**
       *  Invocazione del metodo "del" del DNS.
       *  Viene eseguito sul PlaceID relativo alla colonna "col" passata.
       */
  public void invocaDel (PlaceID placeID) {
    Debug.outln ("ModelloDNS - INVOCO LA delDomain");
    mioDNS.removeDomain (placeID);
  } //invocaDel

      /**
       *  Invocazione del metodo "refresh" del DNS.
       */
  public void invocaRefresh () {
    Debug.outln ("ModelloDNS - INVOCO LA refresh");
    mioDNS.refresh ();
  } //invocaRefresh

      /**
       *  Rende true se si pu invocare il "refresh", ossia se esiste un DNS genitore 
       *  a cui chiedere il refresh.
       */
  public boolean refreshInvocabile () {
    Debug.outln ("ModelloDNS - INVOCO LA refresh");
    return ( mioDNS.getParentDNS() != null );
  } //refreshInvocabile

      
      /**
       *  CLASSE INTERNA CHE DEFINISCE IL "MODELLO" A CUI SI RIFERISCE LA TABELLA DEL DNS.
       *  L'istanza di questa classe  resa dal metodo "cheTabella".
       *  Vedi classe {@link javax.swing.table.AbstractTableModel}
       *  Nota Bene: c' un side-effect sulla variabile "mioDNS" della classe contenitrice!
       */
  protected class TabellaDNS extends AbstractTableModel {

      /**
       *  Rende il numero di colonne della tabella.
       */
    public int getColumnCount () { 
      return 3;
    } //getColumnCount

      /**
       *  Rende il numero di righe della tabella.
       */
    public int getRowCount () { 
      // Chiedo al DNS l'elenco di tutti i dominii
      PlaceID[] arrPlaceID = mioDNS.getDomainsArray();
      return arrPlaceID.length;
    } //getRowCount

      /**
       *  Rende il valore (sottoforma di oggetto) contenuto nella tabella in posizione (row,col).
       */
    public Object getValueAt (int row, int col) {
      // Chiedo al DNS l'elenco di tutti i dominii
      PlaceID[] arrPlaceID = mioDNS.getDomainsArray();

      if ((row >= 0) && (row < arrPlaceID.length)) {
        // Prelevo il DomainInfo ralativo alla riga in esame
        DomainInfo domAtt = mioDNS.getDomain (arrPlaceID[row]);

        // E rendo l'oggetto relativo alla colonna
        switch (col) {
          case  0: return domAtt.placeID;
          case  1: return domAtt.host.getHostAddress();
          case  2: return new Integer(domAtt.port);
          default: return null;
        }
      }
      else
        return null;
    } //getValueAt

      /**
       *  Rende il nome della colonna "col", cio l'informazione "intensiva".
       */
    public String getColumnName (int col) {
      switch (col) {
        case  0: return Lingua.frase("MD_DOMAIN_ID");
        case  1: return Lingua.frase("MD_ADDRESS");
        case  2: return Lingua.frase("MD_PORT");
        default: return null;
      }
    } //getColumnName

      /**
       *  Rende la classe degli oggetti relativi alla colonna "col".
       *  Necessaria per un buon "allineamento" delle stringhe a sinistra e dei numeri a destra!
       */
    public Class getColumnClass(int col) {
      return getValueAt(0,col).getClass();
    } //getColumnClass

      /**
       *  La cella in posione (row,col) pu essere modificata dall'utente?
       */
    public boolean isCellEditable (int row, int col) {
      return false;
    } //isCellEditable

  } //TabellaDNS (classe)

} //ModelloDNS