package SOMA.agent.mobility;

import SOMA.agent.*;
import SOMA.agent.classLoading.*;
import SOMA.ext.AgentManagement.*;
import SOMA.naming.*;
import SOMA.explorer.*;
import SOMA.Environment;
import SOMA.utility.IndexHashtable;
import java.util.*;
import java.io.*;
import java.io.PrintStream;

/** 
 * Gestore degli agenti di un place.
 *
 * @author Livio Profiri
 * Revised by Alessandro Ghigi
 */
public class AgentManager {
	
	Environment env;
	AgentSystem agentSystem;
	public ClassManager agentClassManager;
	public ClassManager cacheClassManager;
	public IndexHashtable indexStore = new IndexHashtable();
	StringExplorerItem path, cache;
	public int AgentIDCounter = 1;

	/** Memorizza i worker degli agenti. */
	public AgentWorkerStore agentWorkerStore = new AgentWorkerStore();
	public AgentPositionStore agentPositionStore = new AgentPositionStore();
	DirExplorerItem agentManagerDir;
	
	/** 
	 * Numero massimo di tentativi di ricerca di un agente
	 * per recapitargli un messaggio.
	 */
	public final int MAX_MESSAGE_ATTEMPTS = 6;
		
	public AgentManager () {}
	
	/** 
	 * Costruttore.
	 * @param env L'environment del place.
	 */
	public AgentManager(Environment env) {
		this.env = env;
		agentSystem = new AgentSystem(env);
		agentManagerDir = new DirExplorerItem("agentManager");
		env.dir.addItem(agentManagerDir);
		// Entrambe NON creano la directory se non esiste gia'
		try {
			agentClassManager = new ClassManager(System.getProperty("user.dir") + File.separator + "agents", false);
			cacheClassManager = new ClassManager(System.getProperty("user.dir") + File.separator + "cache", false);
		}
		catch(Exception e) {
			e.printStackTrace(env.err);
		}
		path = new ClassManagerExplorerItem(agentClassManager) {
			public void update(ClassManager classManager) {
				agentClassManager = classManager;
			}
		};
		cache = new ClassManagerExplorerItem(cacheClassManager) {
			public void update(ClassManager classManager) {
				cacheClassManager = classManager;
			}
		};
		agentManagerDir.addItem("path",path);
		agentManagerDir.addItem("cache",cache);
		agentManagerDir.addItem("store",new ObjectExplorerItem(indexStore));
		agentManagerDir.addItem("list",new ExplorerItem("List of agent workers") {
			public Object Execute(Collection Parameters,PrintStream out) {
				agentWorkerStore.printWorkers(out);
				return null;
			}
		});
		agentManagerDir.addItem("a",new AgentWorkerExplorerItem(env));
		agentManagerDir.addItem("pos",new ExplorerItem("List of agents positions.") {
			public Object Execute(Collection Parameters,PrintStream out) {
				agentPositionStore.printPositions(out);
				return null;
			}
		});
		agentManagerDir.addItem("death",new ExplorerItem("\"Agent ID\"") {
			public Object Execute(Collection Parameters,PrintStream out) {
				try {
					agentDeath(new AgentID((String)Parameters.iterator().next()));
					out.println("DONE!");
				}
				catch(Exception e) {
					e.printStackTrace();
				}
				return null;
			}
			public String Help(PrintStream out) {
				out.println("Notifies an agent's death to his home place.");
				out.println("  Use it only if the agent is really dead!");
				return "Notifies an agent's death to his home place.";
			}
		});
		agentManagerDir.addItem("save",new ExplorerItem("Save agents to disk.") {
			public Object Execute(java.util.Collection Parameters,PrintStream out) {
				try {
					save();
					out.println("Saved!");
				}
				catch(Exception e) {
					e.printStackTrace(out);
				}
				return null;
			}
		});
		agentManagerDir.addItem("load",new ExplorerItem("Load agents from disk.") {
			public Object Execute(java.util.Collection Parameters,PrintStream out) {
				try {
					load();
					out.println("Loaded!");
				}
				catch(Exception e) {
					e.printStackTrace(out);
				}
				return null;
			}
		});
		ExplorerItem launch = new AgentLauncherExplorerItem(env);
		agentManagerDir.addItem("launch",launch);
		env.dir.addItem("launch",launch);
	}
	
	/** 
	 * Creazione di un agente.
	 */
	public AgentWorker createAgent(String agentName,Object argument,boolean isSystemAgent,boolean traceable) {
		AgentWorker worker = null;
		try {
			AgentID newID = newAgentID();
			ClassLoader classLoader;
			if(isSystemAgent) classLoader = getClass().getClassLoader(); // Lo stesso ClassLoader della classe attuale!
			else classLoader = new AgentClassLoader(env,agentName,newID);
			Agent agent = (Agent)classLoader.loadClass(agentName).newInstance();
			agent.setID(newID);
			agent.putArgument(argument);
			agent.setTraceable(traceable);
			worker = createWorker(agent);
		}
		catch(Exception e) {
			e.printStackTrace(env.err);
		}
		return worker;
	}
	
	/** 
	 * Creazione del worker di un agente.
	 * Prima di creare il worker si verificano i sui diritti
	 * di accesso al place.
	 */
	public AgentWorker createWorker(Agent agent) {
		AgentWorker worker = null;
		try {
			// Se non c' il security manager, non effettuo neanche i controlli.
			if(System.getSecurityManager() == null ||
					agent.getClass().getProtectionDomain().implies(new PlaceAccessPermission(env.placeID))) {
				agent.agentSystem = agentSystem;
				worker = new AgentWorker(agent,env);
				agentWorkerStore.putWorker(worker);
				// Aggiorno la posizione dell'agente: attenzione: posso spedire un comando anche nel place in cui mi trovo;
				// l'operazione ha costo bassissimo (SelfConnection).
				if(agent.traceable) env.networkManager.sendCommand(agent.getID().getHome(),new AgentPositionUpdateCommand(agent.getID(),env.placeID));
			}
			else env.err.println("AgentManager.createWorker: Agent " + agent + " has no PlaceAccessPermission to " + env.placeID);
		}
		catch(Exception e) {
			e.printStackTrace(env.err);
		}
		// Viene creato un worker anche presso lo Slave
		if(env.repConnection != null && !env.slave) {
			try {
				env.repConnection.send(new SlaveTransportCommand(env,agent));
			}
			catch(Exception e) {
				e.printStackTrace();
			}
		}
		return worker;
	}
	
	/** 
	 * Spedizione di un messaggio ad un altro agente.
	 * Non e' possibile statilire se il messaggio sara' correttamente recapitato.
	 * Questo metodo si interfaccia direttamente al NetworkManager.
	 */
	public void sendMessage(Message message) {
		sendMessage(message,0);
	}
	
	public synchronized void sendMessage(Message message,int attemptsCount) {
		AgentWorker destinationWorker = env.agentManager.agentWorkerStore.getWorker(message.to);
		if(destinationWorker != null && destinationWorker.getStatus() != AgentWorker.GONE) { // Ho gi trovato l'agente! 
			destinationWorker.agent.mailbox.storeMessage( message );
			// ATTENZIONE: Agenti in idle ==> riavviati.
			if(destinationWorker.getStatus() == AgentWorker.OFF ||
				destinationWorker.getStatus() == AgentWorker.IDLE ||
				destinationWorker.getStatus() == AgentWorker.STOPPED) {
				env.out.println("Waking up agent " + message.to);
				try {
					destinationWorker.start();
				}
				catch(AgentWorker.AgentWorkerException e) {
					e.printStackTrace();
				}
			}
		}
		else if(attemptsCount++ < MAX_MESSAGE_ATTEMPTS) { // Numero massimo di tentativi
			// se destinationWorker = null agente non e' locale
			SendMessageCommand sendMessageCommand = new SendMessageCommand(message,attemptsCount);
			// Trovo in quale place  scritta la sua posizione attuale.
			if(env.placeID.equals(message.to.getHome())) {
				// mi trovo nella Home dell'Agente: trovo la posizione dell'agente
				// 2) Qui c' scritto dov' l'agente?
				PlaceID currentAgentPosition = env.agentManager.agentPositionStore.get(message.to);
				if(currentAgentPosition != null) {
					env.networkManager.sendCommand(currentAgentPosition,sendMessageCommand);
				}
				else {
					env.err.println("AgentWorker.sendMessage: ERROR can't find agent: " + message.to);
					env.err.println("MESSAGE " + message + "LOST!");
				}
			}
			else {
				// 4) Dov' l'agente?
				env.networkManager.sendCommand(message.to.getHome(),sendMessageCommand);				
			}
		}
		else {
			env.err.println("I have already tried " + attemptsCount + " times to deliver this message:");
			env.err.println("   " + message + " DISCARDED.");
		}
	}
	
	/** Fa le pulizie, oppure notifica la morte alla home place dell'agente. */
	public void agentDeath(AgentID agentID) {
		if(agentID.getHome().equals(env.placeID)) agentPositionStore.remove(agentID);
		else env.networkManager.sendCommand(agentID.getHome(),new AgentDeathCommand(agentID));
	}
	
	/** Creazione di un nuovo AgentID. */
	public synchronized AgentID newAgentID() {
		return new AgentID(env.placeID,AgentIDCounter++);
	}
	
	/** Restituisce il numero di worker e quindi di agenti del place. */
	public int agentsNumber() {
		return agentWorkerStore.workersNumber();
	}
	
	/** Impacchetta l'agente per il trasporto.
	 * <P>Modificare questa funzione
	 * per introdurre nuovo AgentPacket.
	 */
	public AgentPacket packAgent(Agent agent) {
		AgentPacket agentPacket = null;
		try {
			if(agent.getClass().getClassLoader() instanceof AgentClassLoader) agentPacket = new BasicAgentPacket(agent);
			else agentPacket = new SystemAgentPacket(agent);
		}
		catch(Exception e) {
			System.out.println("----------PROBLEMS DURING PACKETIZING OF THE AGENT: " + agent + " of the class: " + agent.getClass());
		}
		return agentPacket;
	}
	
	public void loadWorkers(List list) {
		Iterator i = list.iterator();
		env.out.println("AgentManager: Loading workers:");
		for(int j=1;i.hasNext();j++){
			SavedWorker saved = (SavedWorker)i.next();
			try {
				AgentID agentID = saved.restoreWorker(env);
				env.out.println("  " + j + ") " + agentID + " loaded!");
			}
			catch(Exception e) {
				env.out.println("  " + j + ") " + saved + ": ERROR: " + e);
			}
		}
		env.out.println();
	}
	
	public void load() throws IOException, ClassNotFoundException {
		load(env.placeID.toString() + "_agents.dat");
	}
	
	public void save() throws IOException {
		save(env.placeID.toString() + "_agents.dat" );
	}
	
	public void save(String fileName) throws IOException {
		ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(new File(fileName)));
		Hashtable table = new Hashtable();
		table.put("count",new Integer(AgentIDCounter));
		env.out.println("count: " + AgentIDCounter);
		table.put("positions",agentPositionStore);
		env.out.println("positions: " + agentPositionStore);
		table.put("agents",agentWorkerStore.saveWorkers(env));
		env.out.println("agents: DONE" );
		oos.writeObject(table);
		oos.close();
		env.out.println( "Status saved!" );
	}
	
	public void load(String fileName) throws IOException, ClassNotFoundException {
		ObjectInputStream ois = new ObjectInputStream(new FileInputStream(new File(fileName)));
		Hashtable table = new Hashtable();
		table = (Hashtable)ois.readObject();
		ois.close();
		int newCount = ((Integer)table.get("count")).intValue();
		int oldCount = AgentIDCounter;
		if(newCount > oldCount) {
			AgentIDCounter = newCount;
			env.out.println("agent counter:" + oldCount + " -->" + newCount);
		}
		agentPositionStore = (AgentPositionStore)table.get("positions");
		env.out.println("positions: " + agentPositionStore);
		loadWorkers((List)table.get("agents"));
		env.out.println("agents: DONE");
		env.out.println("Status loaded!");
	}
	
	public String toString() {
		return "[agentManager: " + agentsNumber() + " agents]";
	}
	
	public void restartAllAgents() {
		AgentID[] workers = agentWorkerStore.cheElencoAgenti();
		for(int i=0;i<workers.length;i++) {
			AgentID currentWorker = workers[i];
			AgentWorker worker = agentWorkerStore.getWorker(currentWorker);
			env.out.println("Restarting agent " + currentWorker);
			try {
				worker.start();	
			}
			catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
}