package SOMA.agent;

import SOMA.Environment;
import SOMA.naming.*;
import SOMA.agent.mobility.PlaceAccessPermission;
import java.security.*;
import java.io.*;
import SOMA.utility.IndexHashtable;
import SOMA.resourceManagement.*;

/** 
 * Classe di interfaccia fra agenti e sistema.
 * Si tratta dell'unico riferimento fornito ad un agente al momento
 * dell'attivazione in un place. Quindi, per interagire con il mondo esterno
 * l'agente dovra' necessariamente utilizzare riferimenti e metodi
 * dell'AgentSystem che gli e' stato associato.
 * L'AgentSystem, se necessario, effettuer i dovuti controlli di sicurezza
 * prima di consentire ad un agente l'accesso ad una determinata risorsa.
 * E' possibile costruire sottoclassi di AgentSystem in modo
 * da fornire interfacce di tipo diverso ad agenti diversi per motivi di
 * flessibilit, efficienza o compatibilit.
 * 
 * @author Livio Profiri
 * Revised by Alessandro Ghigi
 */

public class AgentSystem {
	
	private Environment env;
	
	/** 
	 * Contenitore per oggetti condivisi fra agenti.
	 * Ad ogni oggetto e' associato un indice di tipo int.
	 */
	public IndexHashtable sharedObjects = new IndexHashtable();
	
	/** Costruttore. */
	public AgentSystem(Environment env) {
		this.env = env;
	}
	
	/** 
	 * Metodo di migrazione.
	 * Effettua i controlli di sicurezza, quindi
	 * invoca il metodo AgentWorker.go(PlaceID destination).
	 */
	protected void go(final Agent agent,final PlaceID destination) throws CantGoException {
		try {
			if(System.getSecurityManager() != null) AccessController.checkPermission(new PlaceAccessPermission(destination));
			AccessController.doPrivileged(
					new PrivilegedExceptionAction() {
						public Object run() throws CantGoException {
							agent.worker.go(destination);
							return null;
						}
					});
		}
		catch(AccessControlException e) {
			throw new CantGoException("Place Access Permission Denied.");
		}
		catch(PrivilegedActionException e) {
			throw (CantGoException)e.getException();
		}
	}
	
	protected void idle(Agent agent) {
		// Potrei aggiungere un controllo di sicurezza o lanciare una
		// throw( new UnsupportedOperationException( ... ) );
		agent.worker.idle();
	}
	
	/* Metodi aggiunti per il resource management -------------------------------- */
	
	/** 
	 * Metodo usato da un agente per conoscere i valori delle soglie
	 * di consumo del place
	 */
	public Threshold getResourceConsumptionThresholds() {
		return env.placeResourceManager.getConsumptionThresholds();
	}
	
	/** 
	 * Metodo usato da un agente per conoscere i consumi complessivi
	 *  dei suoi thread dal momento del lancio
	 */
	public AgentInfo[] getResourceTotalConsumption(Agent ag) {
		return env.placeResourceManager.getAgentTotConsumptions(ag.getID());
	}
	
	/** 
	 * Metodo usato da un agente per conoscere i consumi differenziali,
	 *  cio relativi ad un intervallo di t secondi, di ciascuno dei suoi thread
	 */
	public AgentInfo[] getResourceDiffConsumption(Agent ag) {
		return env.placeResourceManager.getAgentDiffConsumptions(ag.getID());
	}
	
	public PlaceID getPlaceID() {
		return env.placeID;
	}
	
	public InputStream getIn() {
		return env.in;
	}
	
	public PrintStream getOut() {
		return env.out;
	}
	
	public PrintStream getErr() {
		return env.err;
	}
	
	public Environment getEnvironment() {
		if(System.getSecurityManager() != null) AccessController.checkPermission(new AgentPermission("Environment"));
		return env;
	}
	
	/** Restituisce l'elenco degli identificatori dei place di questo dominio. */
	public PlaceID[] getPlaces() {
		return env.placeNameService.getPlacesArray();
	}
	
	/** Restituisce l'elenco degli identificatori dei domini, o
	 * un array vuoto se non e' presente un Domain Name Service, perche'
	 * non siamo in un default place.
	 */
	public PlaceID[] getDomains() {
		if(env.domainNameService != null) return env.domainNameService.getDomainsArray();
		else return new PlaceID[0];
	}
	
	/** 
	 * Spedizione di un messaggio
	 * Non e' possibile statilire se il messaggio sara' correttamente recapitato.
	 * Questo metodo richiama {@AgentWorker.sendMessage( Message )
	 */
	public void sendMessage(final Message message) {
		AccessController.doPrivileged(
				new PrivilegedAction() {
					public Object run() {
						env.agentManager.sendMessage( message );
						return null;
					}
				});
	}
	
	/** Restituisce il numero di worker e quindi di agenti del place. */
	public int agentsNumber() {
		return env.agentManager.agentsNumber();
	}
	
}