package SOMA.agent;

import SOMA.naming.*;
import SOMA.security.infrastructure.*;

/**
 * Classe astratta che rappresenta un agente. Per creare un agente e' necessario sottoclassare la classe Agent
 * ridefinendo i metodi run() e, se necessario, putArgument(Object obj).
 * Il place mette a disposizione dell'agente un insieme di risorse, i cui riferimenti si trovano nel campo agentSystem.
 * Il place associa ad ogni agente un AgentWorker che permette di mandare in esecuzione l'agente stesso.
 *
 * @author Livio Profiri
 * Revised by Alessandro Ghigi
 */

public abstract class Agent implements java.io.Serializable {
	
	private AgentID myID = null;  // Attenzione: non voglio che sia modificato direttamente
	
	/**
	 * Metodo che verra' eseguito alla prossima attivazione dell'agente.
	 */
	public String start = "run";
	
	/** 
	 * Interfaccia fra agente e sistema.
	 * agentSystem contiene i riferimenti agli oggetti cui l'agente ha accesso. In questo modo 
	 * e' garantita la protezione del sistema da possibili attacchi di agenti ostili, a cui non 
	 * verranno forniti riferimenti a risorse critiche.
	 * Questo campo e' transient perche' viene asegnato all'agente dal place in cui si trova.
	 */
	public transient AgentSystem agentSystem = null;

	public Mailbox mailbox = null;
	public boolean traceable = false;
	
	/** Riferiomento al worker dell'agente.
	 * Attenzione: il campo worker ha visibilit di package perch
	 * non deve essere visto dalle sottoclassi di Agent che si trovano in un altro package,
	 * ossia dagli agenti creati dall'utente.
	 * Questo campo e' transient perche' viene asegnato all'agente
	 * dal place in cui si trova.
	 */
	transient AgentWorker worker = null;
	
	/** Costruttore vuoto. */
	public Agent() {}
	
	/** 
	 * Permette di specificare l'ID di un agente.
	 * L'implementazione permette di specificare l'ID di un agente una sola volta.
	 * Quest'operazione  fatta dal sistema al momento della creazione di un agente:
	 * createAgent(String agentName,Object argument).
	 * Se si fosse usato un costruttore, anche tutte le sottoclassi lo avrebbero dovuto fornire.
	 */
	// una volta fissato l'agentID non si pu pi cambiare!
	// non lo metto nel costruttore per non complicare troppo la procedura di creazione agenti.
	public void setID(AgentID myID) {
		if(this.myID == null) this.myID = myID;
		else throw new RuntimeException( "Agent.setID: trying to change agentID!!!" );
	}
	
	/** Restituisce l'identificatore dell'agente.*/
	public AgentID getID() {
		return myID;
	}
	
	/** 
	 * Rende rintracciabile o meno l'agente.
	 * Se l'agente e' rintracciabile e non ha ancora una mailbox, viene creata.
	 * Attenzione: traceable e' public, quindi e' comunque possibile cambiarne il valore.
	 */
	public void setTraceable(boolean traceable) {
		this.traceable = traceable;
		if(traceable && (mailbox == null)) mailbox = new Mailbox();
	}
	
	/** 
	 * Permette di definire lo stato iniziale dell'agente.
	 * Questo metodo  vuoto in nella classe Agent e
	 * deve essere ridefinito dalle sottoclassi che implementano agenti,
	 * in maniera analoga a run().
	 * Ovviamente l'oggetto pu anche contenere una struttura dati complessa.
	 */
	public void putArgument(Object obj) {}
	
	/** 
	 * Metodo mandato in esecuzione all'avvio di un agente.
	 * Questo metodo astratto va ridefinito nell'implementazione di
	 * ogni agente.
	 */
	public abstract void run();
	
	/** 
	 * Metodo di migrazione verso un altro place.
	 * Viene invocato il metodo AgentSystem.go(Agent agent,PlaceID destination).
	 * Attenzione: se un agente ha il riferimento ad un altro agente pu chiamarne il metodo di
	 * migrazione, come ogni altro metodo. In sostanza ne pu assumere il controllo completo.
	 * E' quindi estremamente opportuno mantenere nascosti ad ogni agente i
	 * riferiemnti ad altri agenti.
	 */
	public void go(PlaceID destination,String method) throws CantGoException {
		start = method;
		agentSystem.go(this,destination);
	}
	
	/** Metodo di messa in attesa di un agente. */
	public void idle(String method) {
		start = method;
		// Passo attraverso agentSystem per coerenza:
		// voglio che tutte le chiamate facciano riferimento a quella classe,
		// cosicch  sufficiente ridefinirla per cambiare il comportamento del sistema.
		agentSystem.idle(this);
	}
	
	/**
	 * Ridefinire questo metodo per liberare le risorse dell'agente e
	 * lasciarlo in uno stato consistente, nel caso sia necessario interromperlo.
	 * Ricordarsi aggiornare la variabile start con il nome del metodo da lanciare
	 * al prossimo avvio dell'agente.
	 */
	public void stop() {
		throw(new UnsupportedOperationException("Abstract Agent doesn't support this operation."));
	}
	
	/** Fornisce una rappresentazione in forma di stringa dell'agente. */
	public String toString() {
		return "[Agent: " + myID + (mailbox == null ? " not traceable" : " " + mailbox) + "]";
	}
	
	/** Il seguente metodo firma l'agente. */
	public final boolean signatureCode(ProfileManager profile) {
		String agentName;
		if((agentName = this.getClass().getName()) == null) return false;
		if(this.myID instanceof AgentIDSigned) return ((AgentIDSigned)this.myID).signatureCode ( profile, agentName );
		return false;
	}
	
	public final boolean signatureState (ProfileManager profile) {
		if(this.myID instanceof AgentIDSigned) return ((AgentIDSigned)this.myID).signatureState(profile);
		return false;
	}
	
}
